/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 15/10/2024
 */
 
#ifndef VAMPIRIA_CRYPTO_EVENTP2P_H

#define VAMPIRIA_CRYPTO_EVENTP2P_H 1

namespace vampiria { namespace crypto {

const vmp_int p2p_ok=0; /*!< Used by p2p event for message states 'Success'*/
const vmp_int p2p_err=-1; /*!< Used by p2p event for message states ''*/
const vmp_int p2p_accessdenied=-2; /*!< Used by p2p event for message error 'access denied'*/
const vmp_int p2p_duplexsession=-3; /*!< Used by p2p event for message error 'duplex session found'*/
const vmp_int p2p_protocolbad=-4; /*!< Used by p2p event for message error 'protocol bad'*/
const vmp_int p2p_extbad=-5; /*!< Used by p2p event for message error 'extensions framing error'*/
const vmp_int p2p_sessionbad=-6; /*!< Used by p2p event for message error 'peer session bad'*/

//! Takes in input a p2p int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str p2p_msg_status(vmp_int status);

//! Create p2p session message
/*!
    @param buf output buffer
    @param subprotocols subprotocols type
    @param session session info
    @param extensions frame extensions
*/
void p2p_session(vmp::Buf *buf,vmp::str subprotocols,vmp::str session,vmp::str extensions);

//! Create p2p session error message
/*!
    @param buf output buffer
    @param retcode p2p error code
    @param retmsg  p2p error message
*/
void p2p_session_error(vmp::Buf *buf,vmp_int retcode,vmp::str retmsg);

//!Class shared by p2p events used for connection table and callbacks to execute.
class P2pCommon:public crypto::SslCommon
{
    public:
        //! A constructor
        /*!
            @param manager event timer manager
        */
        P2pCommon(event::Manager *manager);
        
        //! A destructor
        ~P2pCommon();
        
        crypto::Ctx_Peer *ctx_;/*!<context local peer associated*/
        vmp::Table<vmp::str,event::Cell *> p2ptable_;/*!<Connection table*/
        
        event::EVTCB csession_;/*!<called when session init in client connection*/
        event::EVTCB ssession_;/*!<called when session init in server connection*/
        
        //! init object
        /*!
            @param ctx ssl context peer associated associated with connections  
            @param ctimeout connection timeout(setting if value >=0)
        */
        void set_p2p_init(crypto::Ctx_Peer *ctx,vmp::time::Time ctimeout);
    
        //! Set p2p event client(if input value = 0 doesn't know anything)
        /*!
            @param tcpconnect called when a connection tcp connect is complete
            @param csession called when session init in client connection 
            @param crecv called when client recv data
            @param cclose called when the client is close
        */
        void set_p2p_client_event(event::EVTCB tcpconnect,event::EVTCB csession,event::EVTCB cclose);
        
        //! Set p2p event server(if input value = 0 doesn't know anything)
        /*!
            @param tcpaccept called when tcp server accept connection
            @param ssession called when session init in server connection
            @param lclose called when the listen close connection
            @param sclose called when the server close connection
        */
        void set_p2p_server_event(net::EVTCBACCEPT tcpaccept,event::EVTCB ssession,event::EVTCB lclose,event::EVTCB sclose);
        
};

//!A p2p event ssl
class EventP2p:public crypto::EventSsl
{
    protected:
        //! Reset p2p event
        void evt_p2p_reset();
    public:
        //! A Constructor
        EventP2p();

        //! A Destructor
        ~EventP2p();
        
        vmp_bool isopensession_;/*!<Internal usage.Session is open?*/
        crypto::Ctx_Peer_Acl peer_;/*!<Peer associated with the connection*/
        
        //!Create a new event p2p client
        /*!
            @param ui user interface (see Class UI)
            @param peer remote server address
            @param common shared p2p class
            @param proxy contains the information of proxy connections(0 no proxy connection)
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_p2p_client(event::UI *ui,net::Address *peer,crypto::P2pCommon *common,net::ProxyChain *proxy=0);

        //!Create a new event p2p server
        /*!
            @param ui user interface (see Class UI)
            @param local address to listen server connection
            @param common shared p2p class
            @param backlog the maximum length to which the  queue  of pending  connections
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_p2p_listen(event::UI *ui,net::Address *local,crypto::P2pCommon *common,vmp_uint backlog);
        
        //! Close event
        void evt_p2p_close();
       
        //! Free event
        void evt_p2p_free();
};

}}

#endif


