/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 02/10/2024
 */
 
#ifndef VAMPIRIA_CRYPTO_CTX_PEER_H

#define VAMPIRIA_CRYPTO_CTX_PEER_H 1

namespace vampiria { namespace crypto {

class Ssl;
class Ctx_Peer;

//! Class used Ctx_Peer_Tls for acl table
class Ctx_Peer_Acl
{
    private:
        friend crypto::Ctx_Peer;
        vmp::str fingerprint_;/*!< fingerprint field*/
        vmp::str subject_;/*!< subject field*/
        vmp_uint permits_;/*!<user permissions level field*/
    public:
        //! A costructor(empty)
        Ctx_Peer_Acl();
        
        //! A costructor
        /*!
            @param fingerprint fingerprint field
            @param subject subject field
            @param user permissions level
        */
        Ctx_Peer_Acl(vmp::str fingerprint,vmp::str subject,vmp_uint permits);
        
        //! A destructor
        ~Ctx_Peer_Acl();
        
        //! Reset Acl peer
        void reset();
        
        //! Gets fingerprint field
        /*!
            @sa fingerprint_
            @return fingerprint field
        */ 
        vmp::str fingerprint();
        
        //! Gets subject field
        /*!
            @sa subject_
            @return subject field
        */ 
        vmp::str subject();
        
        //! Gets permits field
        /*!
            @sa subject_
            @return user permissions level field
        */
        vmp_uint permits();
};

//!Builds peer tls context generic
class Ctx_Peer:public crypto::Ctx_I
{
    private:
       vmp::str subject_;/*!< subject field*/
       vmp::str fingerprint_;/*!< fingerprint field*/
       vmp_uint defp_;/*!<Permissions given to users who do not appear in the acl(0 no permits)*/
       
       //! Virtual function
       SSL_CTX *init();
    public:
       //! A costructor
       /*!
           @param dircert directory files
           @param defp permissions given to users who do not appear in the acl(0 no permission)
           @param subject subject associated(in x509 mode)
           @param days validity certified
       */
       Ctx_Peer(vmp::str dircert,vmp_uint defp,vmp::str subject,vmp_uint days=365000);
       
       //! A destructor
       ~Ctx_Peer();
       
       vmp::str dircert_; /*!<directory files*/
       vmp::str pkey_;/*!<Private key pem file*/
       vmp::str x509_;/*!<X509 pem file*/
       vmp::str info_;/*!<Credentials info file(text file with subject and fingerprint associates with peer)*/
       vmp_uint days_;/*!<days of validity of the associated certificate*/
       db::Sqlite *acl_;/*!< Acl database table*/

       //! Gets certificate associated with the context
       /*!
           @param x509 certificate output
           @return void or except in case failure
       */
       void get_x509(crypto::X509_Wrap *x509);
       
       //! Returns subject peer
       /*!
           @sa subject_
           @return subject
       */
       vmp::str subject();
       
       //! Returns fingerprint peer
       /*!
           @sa fingerprint_
           @return fingerprint
       */
       vmp::str fingerprint();
       
       //! Returns default Permissions.Permissions given to users who do not appear in the acl(0 no permits)
       /*!
           @sa defp_
           @return default permissions
       */
       vmp_uint defp();
       
       //! Adds a peer to the permissions acl
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @param subject peer subject
           @param permits peer permits
       */
       void add_acl_peer(vmp::str fingerprint,vmp::str subject,vmp_uint permits);
       
       //! Change the permissions of the peer
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @param permits peer permits
       */
       void change_permits(vmp::str fingerprint,vmp_uint permits);
       
       //! Reset acl table database
       void reset_acl();

       //! Delete a peer to the permissions acl
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
       */
       void del_acl_peer(vmp::str fingerprint);
       
       //! Returns the list of fingerprints contained in the acl peer
       /*!
           @sa acl_
           @return fingerprint list
       */
       vmp::vector<vmp::str> fingerprints_acl_peer();
       
       //! Gets the subject of an acl peer
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @return subject associated with the fingerprint or except in case failure
       */
       vmp::str get_acl_subject(vmp::str fingerprint);
       
       //! Returns a peer's permissions to the acl
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @param subject peer subject
           @return permits associated(0 no permits)
       */
       vmp_uint verify_acl_peer(vmp::str fingerprint,vmp::str subject="");
       
       //! Verify an ssl peer
       /*!
           @param ssl associated ssl connection
           @param ret acl structure
           @return permits associated(0 no permits)
       */
       vmp_uint verify_peer(crypto::Ssl *ssl,crypto::Ctx_Peer_Acl *ret);
       
       //! Print Acl(for debug)
       /*!
           @return acl in string mode
       */
       vmp::str print_acl();
};

//!Builds peer web context generic(crypto::Ctx_peer extensions)
class Ctx_Peer_Web:public crypto::Ctx_I
{
    private:
        //! Virtual function
        SSL_CTX *init();
    public:
        //! A constructor
        /*!
            @param peer crypto::Ctx_peer to which it refers
        */
        Ctx_Peer_Web(crypto::Ctx_Peer *peer);

        //! A destructor
        ~Ctx_Peer_Web();
        
        crypto::Ctx_Peer *peer_;/*!<Context peer openssl associated*/
        
        //!Returns peer context
        /*!
            @ref peer_
            @return peer context
        */
        crypto::Ctx_Peer *peer();
        
        //! Add or edit an user in acl
        /*!
            @param user user added or modified
            @param password user password
            @param permits user permits
            @param only_localhost if true, access to this user is only allowed from localhost
            @param md5_password if true the input password parameter is encrypted with the md5 function,default false
            @return void or except in case of failure
        */
        void add_acl_user(vmp::str user,vmp::str password,vmp_uint permits,vmp_bool only_localhost,vmp_bool md5_password=false);
        
        //! Change password to user input
        /*!
            @param user user to change password
            @param password user password
            @param md5_password if true the input password parameter is encrypted with the md5 function,default false
        */
        void change_password(vmp::str user,vmp::str password,vmp_bool md5_password=false);
        
        //! Change the user’s access
        /*!
            @param user user to change permits
            @param permits user permits
            @param only_localhost if true, access to this user is only allowed from localhost
        */
        void change_permits(vmp::str user,vmp_uint permits,vmp_bool only_localhost);
        
        //! Reset acl user table database
        void reset_acl_users();
       
        //! Delete an user to the permissions acl
        /*!
            @param user user cancel
        */
        void del_acl_user(vmp::str user);
       
        //! Returns the list of users contained in the acl peer
        /*!
            @return fingerprint list
        */
        vmp::vector<vmp::str> list_acl_users();
        
        //! Returns the password in md5 associated with the user
        /*!
            @param user user
            @return password in md5 or except in case of failure
        */
        vmp::str get_md5_password(vmp::str user);
        
        //! Returns only_localhost field associated with the user
        /*!
            @param user user
            @return only localhost field or except in case of failure
        */
        vmp_bool get_only_localhost(vmp::str user);
        
        //! Returns permits associated with the user
        /*!
            @param user user
            @return permits or except in case of failure
        */
        vmp_uint get_permits(vmp::str user);
        
        //! Returns a user's permissions to the acl
        /*!
            @param user user added or modified
            @param password user password
            @param islocalhost is localhost connection
            @param md5_password if true the input password parameter is encrypted with the md5 function,default false
            @return permits associated(0 no permits)
        */
        vmp_uint verify_acl_user(vmp::str user,vmp::str password,vmp_bool islocalhost,vmp_bool md5_password=false);
        
        //! Print Acl user(for debug)
        /*!
            @return acl in string mode
        */
        vmp::str print_acl_users();
};

}}

#endif

