/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 03/10/2024
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

void p2psession(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp_uint retcode=crypto::p2p_ok;
    vmp::str retmsg="";
    vmp::Buf buf;
    vmp::pair<vmp::str,vmp::str> sdata;
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    crypto::P2pCommon *common=(crypto::P2pCommon *)p2p->common_;
    p2p->sub_->sessiontype_="p2p";
    if(common->ctx_->verify_peer(&(p2p->ssl_),&(p2p->peer_)) == 0)
        retcode=crypto::p2p_accessdenied;
    else
    {
        p2p->sub_->sessionid_=p2p->peer_.fingerprint();
        p2p->sub_->sessionpermits_=p2p->peer_.permits();
    }
    if(p2p->sub_ == 0)
    {    
        retcode=crypto::p2p_err;
        retmsg="Internal error subprotocols not setting";
    }
    else
    {
        try
        {
            common->p2ptable_.insert(p2p->peer_.fingerprint(),cell);
            manager->cell_alloc(cell);
            p2p->isopensession_=true;
            if(p2p->evt_connection_isclient())
            {
                try
                {
                    sdata=p2p->sub_->evtsub_session_client(p2p->peer_.permits());
                    p2p_session(&buf,sdata.first,sdata.second,p2p->common_->framing_ext_.message_client());
                    p2p->evt_connection_send(&buf);
                }
                catch(vmp::exception &x)
                {
                    retcode=crypto::p2p_err;
                    retmsg=x.what();
                }
            }
        }
        catch(vmp::exception &x)
        {
            retcode=crypto::p2p_duplexsession;
        }    
    }
    if(retcode != crypto::p2p_ok)
    {
        if(retmsg == "")
            retmsg=crypto::p2p_msg_status(retcode);
        p2p_session_error(&buf,retcode,retmsg);
        p2p->evt_connection_send(&buf);
        manager->cell_close_err_spec(cell,"crypto::EventP2p",retcode,retmsg);
    }
    manager->unlock();
}

void p2psession_recv(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    vmp::Buf send;
    json::Json json;
    json::JsonObj *root;
    vmp::str p2ptype,subprotocols,session,extensions,retmsg="",init="";
    vmp_int retcode=crypto::p2p_ok;
    vmp::pair<vmp::str,vmp::str> sdata;
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    buf->index();
    try
    {
        root=json.parse_from_str(buf->read_str(buf->size()));
        p2ptype=root->get_object_str("p2ptype");
        if(p2ptype == "session")
        {
            subprotocols=root->get_object_str("subprotocols");   
            session=root->get_object_str("session");
            extensions=root->get_object_str("extensions");
            if(p2p->evt_connection_isserver())
            {
                try
                {
                    sdata=p2p->sub_->evtsub_session_server(p2p->peer_.permits(),subprotocols,session);
                }
                catch(vmp::exception &x)
                {
                    retcode=crypto::p2p_sessionbad;
                    retmsg=x.what();
                    vmp::except_s("");
                }
                p2p->framing_.add_exts(extensions,true);
                p2p_session(&send,sdata.first,sdata.second,p2p->framing_.exts_info());
                p2p->evt_connection_send(&send);
                init="server";
            }
            else 
            {
                try
                {
                    p2p->sub_->evtsub_session_client_confirm(subprotocols,session);
                }
                catch(vmp::exception &x)
                {
                    retcode=crypto::p2p_sessionbad;
                    retmsg=x.what();
                    vmp::except_s("");
                }
                if(!p2p->framing_.add_exts(extensions,false))
                {    
                    retcode=crypto::p2p_extbad;
                    vmp::except_s("");
                }
                init="client";
            }
            p2p->evt_ssl_framing_active();
        }
        else if (p2ptype == "error")
        {
            retcode=(vmp_int)root->get_object_number("retcode");
            retmsg=root->get_object_str("retmsg");
            manager->cell_close_err_spec(cell,"crypto::EventP2pPeer",retcode,retmsg);
        }
        else
            vmp::except_s("");
    }
    catch(vmp::exception &x)
    {
        if(retcode == crypto::p2p_ok)
            retcode=crypto::p2p_protocolbad;
        if(retmsg == "")
            retmsg=crypto::p2p_msg_status(retcode);
        if(p2p->evt_connection_isserver())
        {
            p2p_session_error(&send,retcode,retmsg);
            p2p->evt_connection_send(&send);
        }
        manager->cell_close_err_spec(cell,"crypto::EventP2p",retcode,retmsg);
    }
    manager->unlock();
    crypto::P2pCommon *common=(crypto::P2pCommon *) p2p->common_;
    if(init == "client")
        common->csession_(cell);
    else if(init == "server")
        common->ssession_(cell);
}

}}


