/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 08/01/2024
 */

#include "event_.h"

class PyTimerUI:public event::TimerUI
{
   public:
        PyTimerUI(event::Manager *manager);
        ~PyTimerUI();

        pylib::PyCall pyevent_;
        pylib::PyCall pycevent_;
};

PyTimerUI::PyTimerUI(event::Manager *manager):event::TimerUI(manager)
{
}

PyTimerUI::~PyTimerUI()
{
}

struct stub_timerui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PyTimerUI *>(cell->ui<PyTimerUI>(),false);

        }
        return 0;
    }
};

struct TimerUI_Constructor:public pylib::Constructor<PyTimerUI>
{
    PyTimerUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 1)
         {
              event::Manager *manager=args.get<event::Manager *>(0);
              return new PyTimerUI(manager);
         }
         return 0;
    }
};

void timer_pyevent(event::Cell *cell)
{
    PyTimerUI *timerui=cell->ui<PyTimerUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    pylib::PyObj res=timerui->pyevent_.call(cargs);
}

void timer_pycevent(event::Cell *cell)
{
    PyTimerUI *timerui=cell->ui<PyTimerUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    pylib::PyObj res=timerui->pycevent_.call(cargs);
}

struct stub_TimerUI_set_event:public pylib::Member_def<PyTimerUI>
{
    pylib::Objref *def(PyTimerUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 2)
        {
            event::EVTCB event;
            event::EVTCB cevent;
            
            args.get_callable(0,instance->pyevent_);
            args.get_callable(1,instance->pycevent_);
            
            if(pylib::is_none(instance->pyevent_.ref()))
               event=0;
            else 
               event=timer_pyevent;
            if(pylib::is_none(instance->pycevent_.ref()))
               cevent=0;
            else 
               cevent=timer_pycevent;
            instance->set_event(event,cevent);
            return pylib::retnone();                      
        }
        return 0;
    }
};

struct stub_TimerUI_new_timer:public pylib::Member_def<PyTimerUI>
{
    pylib::Objref *def(PyTimerUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            vmp::str id=args.get<vmp::str>(0);
            return pylib::Converter<event::Cell *>(instance->new_timer(id),false);
        }
        else if(args.size() == 2)
        {
            vmp::str id=args.get<vmp::str>(0);
            vmp::time::Time timeval=args.get<vmp::time::Time>(1);
            return pylib::Converter<event::Cell *>(instance->new_timer(id,timeval),false);
        }
        return 0;
    }
};

void event_init_timerui(pylib::Module module)
{
    //Wrapper cell->ui()
    module.def_with_stub("timerui",stub_timerui());
    //event/ui/timerui.h
    pylib::Class<PyTimerUI> timerui=module.class_with_constructor<PyTimerUI>("TimerUI",TimerUI_Constructor());
    timerui.def_with_stub("set_event",stub_TimerUI_set_event());
    timerui.def_with_stub("new_timer",stub_TimerUI_new_timer());
    timerui.def_("active_timer",&PyTimerUI::active_timer);
    timerui.def_("deactive_timer",&PyTimerUI::deactive_timer);
    timerui.def_("isactive_timer",&PyTimerUI::isactive_timer);
    timerui.def_("get_timeval",&PyTimerUI::get_timeval);
    timerui.def_("id",&PyTimerUI::id);
}

class PyCroniterRef:public event::CroniterRef
{
    public:
        //!A Constructor
       PyCroniterRef();

       //!A Destructor
       ~PyCroniterRef();
       
        pylib::PyCall pyevent_;
        pylib::PyCall pyclevent_;
};

PyCroniterRef::PyCroniterRef():event::CroniterRef()
{
}

PyCroniterRef::~PyCroniterRef()
{
}

class PyCroniterUI:public event::CroniterUI
{
   public:
        PyCroniterUI(event::Manager *manager);
        ~PyCroniterUI();

        vmp::utils::Storage<PyCroniterRef> pycref_;/*!<Storage event*/
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
};

PyCroniterUI::PyCroniterUI(event::Manager *manager):event::CroniterUI(manager)
{
}

PyCroniterUI::~PyCroniterUI()
{
}

void PyCroniterUI::free_ref(event::Cell *cell)
{
    PyCroniterRef *cevent=cell->event<PyCroniterRef>();
    cevent->evtname_="";
    cevent->crontab_="";
    cevent->event_=0;
    cevent->badnext_=false;
    cevent->evt_timer_free();
    pycref_.free(cevent);
}

struct stub_croniterui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PyCroniterUI *>(cell->ui<PyCroniterUI>(),false);
        }
        return 0;
    }
};

struct CroniterUI_Constructor:public pylib::Constructor<PyCroniterUI>
{
    PyCroniterUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 1)
         {
              event::Manager *manager=args.get<event::Manager *>(0);
              return new PyCroniterUI(manager);
         }
         return 0;
    }
};

void croniter_evt(event::Cell *cell)
{
    PyCroniterRef *cevent=cell->event<PyCroniterRef>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        cevent->evt_timer_active(cevent->evt_croniter_next());
    }
    catch(vmp::exception &x)
    {
        vmp::str error;
        vmp::unicode::str_write(&error,"event::CroniterUI::active_croniter(evtname=%s) 'Bad next croniter'",cevent->evtname_.c_str());
        manager->cell_close(cell,event::ERROR,error);
        cevent->badnext_=true;
    }
    if(!pylib::is_none(cevent->pyevent_.ref()))
    {
        pylib::CArgs cargs(1);
        cargs.add<event::Cell*>(cell,false);
        manager->unlock();
        pylib::PyObj res=cevent->pyevent_.call(cargs);
    }
    else
        manager->unlock();
}

void croniter_close(event::Cell *cell)
{
    PyCroniterRef *cevent=cell->event<PyCroniterRef>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell*>(cell,false);
    pylib::PyObj res=cevent->pyclevent_.call(cargs);
}

struct stub_CroniterUI_new_croniter:public pylib::Member_def<PyCroniterUI>
{
    pylib::Objref *def(PyCroniterUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        event::Manager *manager=instance->manager();
        event::Cell *cell;
        event::EVTCB clevent;
        vmp_bool active=true;
        manager->lock();
        if(args.size() == 4 || args.size() == 5)
        {
            PyCroniterRef *cevent;
            try
            {
                cevent=instance->pycref_.get();
                cevent->evtname_=args.get<vmp::str>(0);
                cevent->crontab_=args.get<vmp::str>(1);
                args.get_callable(2,cevent->pyevent_);
                args.get_callable(3,cevent->pyclevent_);
                if(pylib::is_none(cevent->pyclevent_.ref()))
                    clevent=0;
                else 
                    clevent=croniter_close;
                if(args.size() == 5)
                    active=args.get<vmp_bool>(4);
                cevent->next_.set();
                cevent->evt_croniter_next();
                cell=cevent->evt_timer_new(instance,0,croniter_evt,clevent);
            }
            catch(vmp::exception &x)
            {
                cevent->evtname_="";
                cevent->crontab_="";
                instance->pycref_.free(cevent);
                manager->unlock();
                vmp::except_s(x.what());    
            }
            manager->unlock();
            if(active)
                instance->active_croniter(cell);
            return pylib::Converter<event::Cell *>(cell,false);
        }
        manager->unlock();
        return 0;
        
    }
};

void event_init_croniterui(pylib::Module module)
{
    //Wrapper cell->ui()
    module.def_with_stub("croniterui",stub_croniterui());
    pylib::Class<PyCroniterUI> croniterui=module.class_with_constructor<PyCroniterUI>("CroniterUI",CroniterUI_Constructor());
    croniterui.def_with_stub("new_croniter",stub_CroniterUI_new_croniter());
    croniterui.def_("active_croniter",&PyCroniterUI::active_croniter);
    croniterui.def_("deactive_croniter",&PyCroniterUI::deactive_croniter);
    croniterui.def_("isactive_croniter",&PyCroniterUI::isactive_croniter);
    croniterui.def_("evtname",&PyCroniterUI::evtname);
    croniterui.def_("crontab",&PyCroniterUI::crontab);
    croniterui.def_("nextcron",&PyCroniterUI::nextcron);
    croniterui.def_("badnext",&PyCroniterUI::badnext);
}
