#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 26/06/2024

try:
  import sys
  import os
  import subprocess
  import ast
  import importlib
except Exception as e:
  raise ValueError(e)

##Exit success return value
exit_ok_=0

##Check warning string
checkWARNING_="WARNING"

##Exit failure return value
exit_failure_=1

##Check ok string
checkOK_="OK"

##Check failed string
checkFAILED_="FAILED"

##Vampiria root path
root_="{0}/".format(os.path.dirname(os.path.dirname(os.path.dirname(os.path.realpath(__file__)))))

##Return Python executable path
def executable():
  return sys.executable

##Returns the path to a directory in the framework<BR>
#
#@param subdir sub directory path<BR>
#@return subdirectory path,if subdir is None return root directory 
def path(subdir=None):
  if subdir is None:
    return root_
  if not subdir[len(subdir)-1] == '/':
      subdir="{0}/".format(subdir)
  return "{0}{1}".format(root_,subdir)

##Verify if path is a reading file<BR>
#
#@param path file path to verify<BR>
#@return true ok otherwise false
def isfile(path): 
  return os.path.isfile(path)

##Verify if path is a directory<BR>
#
#@param path directory path to verify<BR>
#@return true ok otherwise false
def isdir(path): 
  return os.path.isdir(path)

##Reads a file and returns the file with its contents<BR>
#
#@param filepath
def read_file(filepath):
   f=open(filepath,'r')
   ret=f.read()
   f.close()
   return ret
 
##Create a file and write strvalue<BR>
#
#@param path file path<BR>
#@param strvalue  string to write in file<BR>
def write_file(path,strvalue):
  f=open(path,"w")
  f.write(strvalue)
  f.close()

##Copy the src file or directory recursively to dst<BR>
#
#@param src source file<BR>
#@param dst destination file
def cp_all(src,dst):
  system("cp -r {0} {1} > /dev/null 2>&1".format(src,dst))

##Remove file<BR>
#
#@param filepath file path to remove
def rm_file(filepath):
  system("rm {0} > /dev/null 2>&1".format(filepath))

##Remove Removes everything inside a directory<BR>
#
#@param dirpath directory path<BR>
#@param drem remove directory empty?
def rm_dir_all(dirpath,drem=False):
  if drem:
    system("rm -rf {0}/ > /dev/null 2>&1".format(dirpath))
  else:
    system("rm -rf {0}/* > /dev/null 2>&1".format(dirpath))

##Removes all object files(generated compiler c)<BR>
#
#@param rootpath root path object<BR>
#@param files files list(see vmpbuild.setup.files)
def rm_cobj(rootpath,files):
  local=getcwd()
  chdir(rootpath)
  for f in files:
    chdir(f[0])
    system("rm *.o > /dev/null 2>&1")
    chdir(rootpath)
  chdir(local)

##Get current directory path
def getcwd():
  return "{0}/".format(os.getcwd())

##Create directory empty
#
#@param directory path
def mkdir(path):
  try:
    os.mkdir(path)
  except Exception as e:
    raise ValueError(e)
  
##Change directory to work<BR>
#
#@param path new path directory
def chdir(path):
  try:
    os.chdir(path)
  except Exception as e:
    raise ValueError(e)
  
##Listen directory<BR>
#
#@param path directory path to listen
def listdir(path):
  try:
    return os.listdir(path)
  except Exception as e:
    raise ValueError(e)

##Print in the standard error
#
#@param fmt string message  
def stderr(fmt):
  sys.stderr.write("{0}\n".format(fmt))
  sys.stderr.flush()

##Print in the standard output
#
#@param fmtstring message  
def stdout(fmt):
  sys.stdout.write("{0}".format(fmt))
  sys.stdout.flush()

##Required input line from standanrd input<BR>
#
#@return value to read or empty string 
def stdin():
  value=input()
  if (not value) or value == '\n':
      return ""
  return value

##Exit with code Success
def exit_ok():
  sys.exit(exit_ok_)

##Exit with code Failure
def exit_failure():
  sys.exit(exit_failure_)

##Message ok check<BR>
#
#@param preamble message preamble
def msg_ok(preamble):
  stderr("{0} [{1}]".format(preamble,checkOK_))

##Message warning<BR>
#
#@param preamble preamble message
def msg_warning(preamble):
  stderr("{0} [{1}]".format(preamble,checkWARNING_))

##Message failure check<BR>
#
#@param preamble message preamble<BR>
#@param error error info
def msg_failure(preamble,error):  
  stderr("{0} ({1}) [{2}]".format(preamble,error,checkFAILED_))
  exit_failure()

##Read the input arguments of a python script,returns args
def argv_get():
  l=len(sys.argv)
  args=[]
  for i in range(1,l):
    args.append(sys.argv[i])
  return args

##Read the input arguments of a python script,returns the first argument, if it is otherwise None.<BR>
#
##@param args output list of arguments from the second on
def argv_get2(args):
  l=len(sys.argv)
  if l > 1:
    ret=sys.argv[1]
    for i in range(2,l):
      args.append(sys.argv[i])
    return ret
  return None

##Returns the version of the python interpreter
def pyversion():
  return sys.version_info

##Exec external command<BR>
#
#@param cmd command to exe(string format)
def system(cmd):
  return os.system(cmd)

##Search external command<BR>
#
#@param cmd command to search(list format ex.['ls','-l']
def search_cmd(cmd):
  with open(os.devnull, 'w')  as FNULL:  
    try:    
      subprocess.check_call(cmd,stderr=FNULL,stdout=FNULL)
      msg_ok("Search cmd \'{0}\'".format(cmd[0]))
      return True
    except:
      msg_failure("Search cmd \'{0}\'".format(cmd[0]),"not found")
      return False 

##Compiler cpp source<BR>
#
#@param rootpath root path compiler<BR>
#@param cpp compiler command string<BR>
#@param files files list<BR>
#@param objfiles object files list
def compiler(rootpath,cpp,files):
  objfiles=[]
  local=getcwd()
  chdir(rootpath)
  for f in files:
    compiler=cpp
    for fe in f[1]:
      compiler="{0} {1}.cc".format(compiler,fe)
      objfiles.append("{0}{1}.o".format(f[0],fe))
    chdir(f[0])
    if system(compiler) != 0:
      raise
    chdir(rootpath) 
  chdir(local)
  return objfiles

##Linker cpp source<BR>
#
#@param rootpath root path compiler<BR>
#@param cpp linker command string<BR>
#@param objfiles object files list<BR>
#@param loptions options linker or object path<BR>
#@param dst destination exe file
def linker(rootpath,cpp,objfiles,loptions,dst):
  local=getcwd()
  chdir(rootpath)
  linker=cpp
  for obj in objfiles:
    linker="{0} {1}".format(linker,obj)
  for opt in loptions:
    linker="{0} {1}".format(linker,opt)
  linker="{0} -o {1}".format(linker,dst)
  if system(linker) != 0:
    raise
  chdir(local)

##Find an attribute of the python input module<BR>
#
#@param pymod python module<BR>
#@param attrname attribute name<BR>
#@return attribute python,raise if not found<BR>
def pymod_attr(pymod,attrname):
  return getattr(pymod,attrname)

##Append dir in sys.path<BR>
#
#@param directory directory append to sys.path 
def append(directory):
  sys.path.append(directory)
  
##Remove dir in sys.path<BR>
#
#@param directory: directory to remove from sys.path
def remove(directory):
  sys.path.remove(directory)
  
##Append subdir in sys.path<BR>
#
#@param subdir vampiria subdir to append 
def append_subdir(subdir):
  append("{0}".format(path(subdir)))
 
##Append subdir in sys.path<BR>
#
#@param subdir vampiria subdir to remove
def remove_subdir(subdir):
  remove("{0}".format(path(subdir)))

##Reload python module<BR>
#
#@param module python
def reload_module(module): 
  importlib.reload(module)

##Import software setup file
#
#@return python setup module
def setup_import():
  import setup
  return setup

##Returns main def in software(file main.py)<BR>
#
#@return vmp_main
def main_def():
  import main
  return pymod_attr(main,"vmp_main")

##import module name<BR>
#
#@param name module name<BR>
#@param preamble module preamble (ex preamble.name). if '' import name<BR>
#@return module handle
def module_import(name,preamble=''):
  if preamble == '':
    return importlib.import_module(name)
  else:
    return importlib.import_module("{0}.{1}".format(preamble,name))
  
##Verify if input is dictionary<BR>
#
#@param value value to verify<BR>
#@return dictionary value or None
def isdict(value):
  try:  
    args=ast.literal_eval(value)
    if type(args) is not dict:
      return None
  except Exception as e:
    return None
  return args  

##Create config file<BR>
#
#@param path config file path<BR>
#@param configlist string line to write in config file
def config_create(path,configlist):
  strvalue=""
  for line in configlist:
    strvalue="{0}{1}\n".format(strvalue,line)
  write_file(path,strvalue)

##Remove config file<BR>
#
#@param path config file path<BR>
def config_remove(path):
  if isfile(path):
    system("rm {0}".format(path))

##Create html index file for javascript project
#
#@param filepath file path to writes
#@param title title page
#@param rtag root tag
#@param css css file list
#@param js javascript file list
def jsindex_create(filepath,title,rtag,css,js):
  strcss=""
  for c in css:
    strcss="{0}\r\n    <link rel='StyleSheet' type='text/css' href='{1}'/>".format(strcss,c)
  strjs=""
  for j in js:
    strjs="{0}\r\n    <script type='text/javascript' src='{1}' defer></script>".format(strjs,j)
  rscript="\r\n    <script>\r\n      window.onload=function(){0}root=document.getElementById('root');root.init(){1};\r\n      window.onerror=function(message, url, line, col, errorObj){0}alert(`${0}message{1}\n${0}url{1}, ${0}line{1}:${0}col{1}`);{1};\r\n    </script>".format('{','}',rtag)
  ret="<!doctype html>\r\n<html>\r\n  <head>\r\n    <title>{0}</title>{1}\r\n  </head>\r\n  <body>\r\n    <root-tag id='root'></root-tag>{2}{3}\r\n  </body>\r\n</html>".format(title,strcss,rscript,strjs)
  write_file(filepath,ret)
  

