/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 19/12/2024
 */
/** 
  *Returns the value associated with the input css variable
  *@param name {String} variable name
  *@return the value associated with the input or undefined in case the variable not found
*/
function vmp_rootvar(name)
{
  let ret = getComputedStyle(document.documentElement).getPropertyValue(name);
  return (ret != '')?ret:undefined;
}
/**
  *@throws throws exception
  *@param msg {String} message error
*/
function vmp_except(msg){throw new Error(msg);}
/**
  *Wrapper of the javascript function document.createElement
  *@param tagname {string} tag name
  *@param type {string} tag type('div','table' etc.. for element derived)
  *@return tag element
*/
function vmp_wrap_createElement(tagname,type='')
{
  if(type === '')
    return document.createElement(tagname);
  else
    return document.createElement(type,{is:tagname});
}
/**
  *Wrapper of the javascript function customElements.define that creates custom tag
  *@param modname {String} module name(the name must have a hyphen in the middle)
  *@param modtype class type element
  *@param type {String}tag type(div,table etc.. derivative element)
  *throws in case of failure
*/
function vmp_wrap_define(modname,modtype,type='')
{
  try
  {
    if(type === '')
      customElements.define(modname,modtype);
    else
      customElements.define(modname,modtype,{extends:type});
  }
  catch(error)
  {
    vmp_except("vmp_define(name='" + modname + "') " + error);
  }  
}
/**
  *Define root-tag
  *@param main class derived from Vmp_Main
  *throws in case of failure
*/
function vmp_main_define(main)
{
  vmp_wrap_define("root-tag",main);
}
/**
  *Wrapper of the javascript function document.createTextNode that creates text tag
  *@param message {String} message text
  *@return tag text
*/
function vmp_wrap_createTextNode(message){return document.createTextNode(message);}
/**
  *Wrapper of the javascript function alert that creates alert message
  *@param message {String} message text
*/
function vmp_wrap_alert(message){alert(message);}
/**
  *Wrapper of the javascript function prompt that creates prompt message
  *@param message {String} message text display
  *@param defvalue {String} defvalue default value
  *@return If the user clicks "OK", the input value is returned otherwise null is returned.

*/
function vmp_wrap_prompt(message,defvalue){return prompt(message,defvalue);}
/**
  *Wrapper of the javascript function confirm that creates prompt message
  *@param message {question} question display
  *@return true if the user clicked OK, otherwise false.
*/
function vmp_wrap_confirm(question){return confirm(question);}
/**
  *Wrapper of the javascript for create Map object
  *@return Map object.
*/
function vmp_wrap_map(){return new Map();}
/**
  *Wrapper of the javascript for create Set object
  *@return Set object.
*/
function vmp_wrap_set(){return new Set();}
/**
  *Wrapper of the javascript for create WeakMap object
  *@return WeakMap object.
*/
function vmp_wrap_weakMap(){return new WeakMap();}
/**
  *Wrapper of the javascript for create WeakMap object
  *@return WeakMap object.
*/
function vmp_wrap_weakSet(){return new WeakSet();}
/**
  *Wrapper of the javascript JSON.parse for constructing the JavaScript value or object described by the string.
  *@param text {String} the string to parse as JSON.
  *@throws in case of failure
  *@return The Object, Array, string, number, boolean, or null value corresponding to the given JSON text
*/
function vmp_wrap_json_parse(text){return JSON.parse(text);}
/**
  *Wrapper of the javascript JSON.stringify for converts a JavaScript value to a JSON string.
  *@param data {String} The value to convert to a JSON string.
  *@return {String} json string
*/
function vmp_wrap_json_stringify(data){return JSON.stringify(data);}
/**
  *Wrapper of the javascript text encoder
  *@param etype {String} encoder type(ex "utf-8").
  *@return {String} text enoder object.
*/
function vmp_wrap_TextDecoder(etype){return new TextDecoder(etype);}
/**
  *@param n {number} Max n accepted
  *@return {Number} returns a random number between 0(inclusive),and n(exclusive)
*/
function vmp_wrap_random(n)
{
  return Math.floor(Math.random()*n);
}
function vmp_typeCheck(value) 
{
  const return_value = Object.prototype.toString.call(value);
  const type = return_value.substring(return_value.indexOf(" ") + 1,return_value.indexOf("]"));
  return type.toLowerCase();
}
/**
  *Check if the input is an integer
  *@param n input verify
  *@param min {Number} Minimum value of the range
  *@param max {Number} Maxinum value of the range
  *@return {Boolean} true if n is numeric between min and max,otherwise false
*/
function vmp_isint(n,min=Number.MIN,max=Number.MAX)
{
  if (!(vmp_typeCheck(n) === "number") || (n % 1 != 0) || n < min || n > max)
    return false;
  return true;
}
/**
  *Check if the input is a number
  *@param n input verify
  *@param min {Number} Minimum value of the range
  *@param max {Number} Maxinum value of the range
  *@return {Boolean} true if n is number between min and max,otherwise false
*/
function vmp_isnumber(n,min=Number.MIN,max=Number.MAX)
{
  if (!(vmp_typeCheck(n) === "number") || n < min || n > max)
    return false;
  return true;
}
/**
  *Check if the input is a bool
  *@param b {Object} input verify
  *@return {Boolean} true if n is numeric between min and max,otherwise false
*/
function vmp_isbool(b)
{
  return (vmp_typeCheck(b) === "boolean");
}
/**
  *Check if the input is a string
  *@param str {Object} input verify
  *@return {Boolean} true if str is string,otherwise false
*/
function vmp_isstring(str){return (vmp_typeCheck(str) === "string");}
/**
  *Check if the input is a word string
  *@param str {Object} input verify
  *@return {Boolean} true if str is string,otherwise false
*/
function vmp_isword(str)
{
  if(vmp_isstring(str) && (str.length != 0))
  {
    for(let s of str)
    {
      s=s.charCodeAt(0);
      if((s <= 0x20)||(s >= 0x7E))
        return false;
    }
    return true;
  }
  return false;
}
/**
  *Check if the input is exadecimal string
  *@param str {Object} input verify
  *@return {Boolean} true if str is string,otherwise false
*/
function vmp_isxstring(str)
{
  if(!vmp_isstring(str))
    return false;
  for(let s of str)
  {
    code=charCodeAt(s);
    if(((code >= 0x30 && code <= 0x39) || (code >= 0x41 && code <= 0x46) || (code >= 0x61 && code <= 0x66))) 
      return false;
  }
  return true;
}
/**
  *Check if the input is an array
  *@param arr {Object} input verify
  *@return {Boolean} true if arr is array,otherwise false
*/
function vmp_isarray(arr){return (Array.isArray(arr));}
/**
  *Check if the input is a simplex object ({})
  *@param obj {Object} input verify
  *@return {Boolean} true if obj is simplex object,otherwise false
*/
function vmp_isobj(obj){return (Object.entries(obj).length != 0);}
/*
  *Class representing a timeout operation.
  *@class
  *@public
*/
class Vmp_Timeout
{
  /**
    *Create an timer.
    *@constructor
  */
  constructor(){this.id_=null;}
  /**
    *Start a timeout 
    *@param func {callback} the function to execute if the timeout is activated
    *@param timeout {number} timeout number
    *@param args function parameters
  */
  start(func,timeout,...args){this.stop();this.id_=setTimeout(func,timeout,args);}
  /**Stop the timeout*/
  stop(){if(this.id_!=null){clearTimeout(this.id_);this.id_=null;}}
};
/**
  *Return parent tag
  *@param tag input tag
  *@return {Tag} parents tag o null in case of root tag
*/
function vmp_tag_parent(tag){return tag.parentElement;}
/**
  *Return index children tag
  *@param tag input tag
  *@param children index
  *@return {Tag} the children referred to the index
*/
function vmp_tag_children(tag,index){return tag.children[index];}
/**
  *Insert an item in the index position in the tag
  *@param tag input tag
  *@param child element to be inserted
  *@param index {Number} index position
  *@return the appended node 
*/
function vmp_tag_childInsert(tag,child,index)
{
  let e=tag.children[index];
  return (e != null)?e.before(child):tag.appendChild(child);
}
/**
  *Replace an item in the index position,if index is out range it append at the end in the tag
  *@param tag input tag
  *@param child element to be inserted
  *@param index {Number} index position
  *@return the replaced node 
*/
function vmp_tag_childReplace(tag,child,index)
{
  let e=tag.children[index];
  return (e != null)?e.replaceWith(child):tag.appendChild(child);
}
/**
  *Remove an item in the index position in the tag
  *@param tag input tag
  *@param index {Number} index position
*/
function vmp_tag_childRemove(tag,index)
{
  let e=tag.children[index];
  if(e != null){e.remove()};
}
/**
  *Removes the first item in the tag
  *@param tag input tag
*/
function vmp_tag_childRemovePop(tag)
{
  if(tag.firstElementChild != null)
    tag.firstElementChild.remove();
}
/**
  *removes the last item in the tag
  *@name vmp_component#childRemoveBack
  *@public
  *@function   
*/
function vmp_tag_childRemoveBack(tag)
{
  if(tag.lastElementChild != null)
    tag.lastElementChild.remove();
}
/**
  *Remove all childs in the tag
  *@param tag input tag
*/
function vmp_tag_childRemoveAll(tag)
{
  while(tag.firstElementChild)
    tag.firstElementChild.remove();
}
/**
  *Enable tag
  *@param tag input tag
*/
function vmp_tag_enable(tag)
{
  tag.removeAttribute("disabled");
}
/**
  *Disable tag
  *@param tag input tag
*/
function vmp_tag_disable(tag)
{
  tag.setAttribute("disabled","");  
}
/** 
  *@constant
  *@typedef {Object} Vmp_WebSocketCode Manages the closing codes of a websocket
  *@default
  *@property {Number} Normal the connection close
  *@property {Number} Shutdown Server is shutting down
  *@property {Number} Protocol Some error in the protocol has happened 
  *@property {Number} Type The type(text, binary) was not supported
  *@property {Number} Abnormal Abnormal local close
  *@property {Number} Utf8 The message wasn't in UTF8
  *@property {Number} Policy The policy of the server has been broken
  *@property {Number} Big The messages received is too big
  *@property {Number} Extension Mandatory extension missing
  *@property {Number} Unexpected Unexpected happened
*/
const Vmp_WebSocketCode={
  Normal:1000,
  Shutdown:1001,
  Protocol:1002,
  Type:1003,
  Abnormal:1006,
  Utf8:1007,
  Policy:1008,
  Big:1009,
  Extension:1010,
  Unexpected:1011,
  /**
    *@param code {Number} input code
    *@return {String} string reason
  */
  reason(code)
  {
    switch(code)
    {
      case this.Normal:return "Connection close";
      case this.Shutdown:return "Server is shutting down ";
      case this.Protocol:return "Some error in the protocol has happened"; 
      case this.Type:return "type was not supported";
      case this.Abnormal:return "Abnormal local close";
      case this.Utf8:return "The message wasn't in UTF8";
      case this.Policy:return "The policy of the server has been broken";
      case this.Big:  return "The messages received is too big";
      case this.Extension:return "Mandatory extension missing";
      case this.Unexpected:return "Unexpected happened";
      default:return "Unknown";
    }
  }
}
/**
  *Takes an input array and returns an ordered array
  *@param v input array
  *@param reverse false ascending order,true descending
  *@param duplicate if false remove duplicate value
  *@return ordered carrier
**/
function vmp_sort(v,reverse=false,duplicate=true)
{
  ret=v.sort();
  if(reverse)
    ret.reverse();
  if(!duplicate)
    ret=ret.filter((item, pos, array)=>!pos || item != array[pos - 1]);
  return ret;
}
/**
  *Checks if a value is inside the vector
  *@param value value to be verified
  *@param values value vector
  *@return true if the value is in the vector,otherwise false
**/
function vmp_invector(value,values)
{
  for(let v of values)
    if(value == v) return true;
  return false;
}
/** 
  *Inserts a value into the vector if it is not present
  *@param value value to be inserted
  *@param values pointer value vector
  *@return if the value was not present and was entered, otherwise false
**/
function vmp_insinvector(value,values)
{
  if(vmp_invector(value,values)) return false;    
  values.push(value);
  return true;
}
/**
  *Delete a value within the vector(first occurrence)
  *@param value value to be deleted
  *@param values pointer value vector
  *@return if the value was deleted, false if the value was not inside the vector
**/
function vmp_delinvector(value,values)
{
  for(let i=0;i<values.length;i++)
    if(value == values[i]){values.splice(i,1);return true;}
  return false;
}

