/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 04/12/2024
 */
/**Used to work with JData lists*/
class Vmp_JList
{
  /**Constructor*/
  constructor()
  {
    /**
     *Object List references
     *@type {Array}
     *@public
    */
    this.list_=undefined;
    /**
     *Json type of the items in the list(if == "" alltype)
     *@type {String}
     *@public
    */
    this.jtype_="";
  }     
  /**Internal usage*/
  check_init(fname)
  {
    if(this.list_ === undefined)
      vmp_except(`Vmp_JList.%s(${fname}) list json obj not setting`);
  }
  /**Internal usage*/
  set(list,jtype)
  {
    if(!vmp_isstring(jtype))
      vmp_except("Vmp_JList.set(jtype) bad type");
    try
    {
      if(!vmp_isarray(list)) vmp_except("");
      let jdata=new Vmp_JData();
      for(l of list)
      {
        jdata.set_data(l);
        if((jtype != "") && (jdata.jtype() != jtype))
          vmp_except(""); 
      }
    }
    catch(error)
    {
      vmp_except("Vmp_JList.set(list) bad object");
    }
    this.reset();
    this.list_=list;
    this.jtype_=jtype;
  }
  /**Reset json data list*/
  reset()
  {
    this.list_=undefined;
    this.jtype_="";
  }   
  /**  
    *Insert a JData at the end of the list
    *@param jdata {Vmp_JData} jdata to insert
    *@throws in case of failure
  */
  push(jdata)
  {
    this.check_init("push");
    try
    {
      if((jtype_ != "") && (jdata.jtype() != jtype_))
        vmp_except("");
      this.list_.push(jdata.obj_);
    }
    catch(error)
    {
        vmp_except("Vmp_JList.push() input jdata bad values");
    }
  }
  /**
    *Replaces the jdata of index i
    *@param i {Index} index to data replace
    *@param jdata {Vmp_JData} new data replace
    *@throws in case of failure
  */
  replace(i,jdata)
  {
    this.check_init("replace");
    try
    {
      if((jtype_ != "") && (jdata.jtype() != jtype_))
        vmp_except("");
      this.list_[i]=jdata.obj_;
    }
    catch(error)
    {
        vmp_except("Vmp_JList.replace() input jdata bad values");
    }
    if(i >= this.list_.length)
      vmp_except("Vmp_JList.replace() index out bound range");
  }
  /**
    *Gets a copy jdata of index i
    *@param i {Index} index to data replace
    *@throws in case of failure
    *@return {Vmp_JData} json data object
  */
  get(i)
  {
    this.check_init("get");
    if(i >= this.list_.length)
      vmp_except("Vmp_JList.get() index out bound range");
    let jdata=new Vmp_JData();
    jdata.set_data(this.list_[i]);
    return jdata;
  }
  /**
    *Returns the size of the list
    *@throws in case of failure
    *@return {Size} the size list
  */
  size()
  {
    this.check_init("size");
    return this.list_.length;
  }
}
/**Used to work with JData table <string key,jData>*/
class Vmp_JTable
{
  /**Constructor*/
  constructor()
  {
    /**
     *Object List references
     *@type {Object}
     *@public
    */
    this.table_=undefined;
    /**
     *Json type of the items in the list(if == "" alltype)
     *@type {String}
     *@public
    */
    this.jtype_="";
  }   
  /**Internal usage*/
  check_init(fname)
  {
    if(this.table_ === undefined)
      vmp_except(`Vmp_JTable.%s(${fname}) list json obj not setting`);
  }
  /**Internal usage*/
  set(table,jtype)
  {
    if(!vmp_isstring(jtype))
      vmp_except("Vmp_JTable.set(jtype) bad type");
    try
    {
      if(!vmp_isobj(table)) vmp_except("");
      let jdata=new Vmp_JData();
      for(t in table)
      {
        jdata.set_data(table[t]);
        if(((jtype != "") && (jdata.jtype() != jtype)))
          vmp_except(""); 
      }
    }
    catch(error)
    {
      vmp_except("Vmp_JTable.set(table) bad object");
    }
    this.reset();
    this.table_=table;
    this.jtype_=jtype;
  }
  /**Reset json data table*/
  reset()
  {
    this.table_=undefined;
    this.jtype_="";
  }
  /** 
    *Returns the list of table keys 
    *@throws in case of failure
    *@return {Array} the list of table keys or except in case of failure
  */           
  keys()
  {
    this.check_init("keys");
    return Object.keys(this.table_);
  }
  /**     
    **Insert a jdata in the table
    *@param jkey {String} Key associated with the jdata
    *@param jdata {Object} json data
    *@throws in case of failure
            
  */ 
  insert(jkey,jdata)
  {
    this.check_init("insert");
    try
    {
      if((jtype_ != "") && (jdata.jtype() != jtype_))
        vmp_except("");
      this.table_[jkey]=jdata.obj_;
    }
    catch(error)
    {
        vmp_except("Vmp_JTable::input() input jdata bad values");
    }
  }
  /**
    *Delete the JData associated with the key from the table
    *@param jkey  {String} Key associated with the jdata
    *@throws in case of failure
    *@return {Boolean} true if the value is cancelled,false if value not found
  */
  cancel(jkey)
  {
    this.check_init("cancel");
    if (jkey in this.table_)
    {
      Reflect.deleteProperty(this.table_,jkey);
      return true;
    }
    return false;
  }
  /**
    *Search the JData associated with the key from the table
    *@param jkey Key associated with the jdata
    *@return {Object} object if data found,or undefined if not found
    *@throws in case of failure
  */
  search(jkey)
  {
    this.check_init("cancel");
    if(jkey in this.table_)
    {
      let jdata=new Vmp_JData();
      jdata.set_data(this.table_[jkey]);
      return jdata;
    }
    return undefined;
  }
};
/**Utility class for managing json data*/
class Vmp_JData
{
  /**Constructor*/
  constructor()
  {
    /**
     *Object references
     *@type {Object}
     *@public
    */
    this.obj_=null
  }
  /**
    *create a new json data
    *@param jtype {String} json data type
  */
  new_data(jtype){this.obj_={jtype:jtype,jbody:{},}}
  /**
    *set a json given by a json string
    *@throws in case of failure
    *@param obj {String_or_Object} text input
  */
  set_data(obj)
  {
    try
    {
      if(vmp_isstring(obj))
        this.obj_=vmp_wrap_json_parse(obj);
      else
        this.obj_=obj;
      if(!(("jtype" in this.obj_) && ("jbody" in this.obj_) && Object.keys(this.obj_) != 2))vmp_except("");
    }
    catch(error){this.obj_=null;vmp_except(error);}
  }
  /**@return json data type*/
  jtype(){return this.obj_.jtype;}
  /**@return json data body*/
  jbody(){return this.obj_.jbody;}
  /**
    *set body object
    *@param obj {Object} body objct
  */
  setjbody(obj){this.obj_.jbody=obj;}
  /**
    *stringify of the jdata object
    *@return {String} json string
  */
  stringify(){return vmp_wrap_json_stringify(this.obj_);}
  /**Internal usage*/
  set_value(label,value,fname)
  {
    if(!vmp_isstring(label))
      vmp_except(`Vmp_JData.${fname}(label) bad type`);
    this.jbody()[label]=value
  }
  /**Internal usage*/
  get_value(label,fname)
  {
    if(!vmp_isstring(label))
      vmp_except(`Vmp_JData.${fname}(label) bad type`);
    if(label in this.jbody())
      return this.jbody()[label];
    vmp_except(`Vmp_JData.${fname}(label) bad value`);
  }
  /**
    *Insert a text primitive in jbody
    *@param label {String} primitive label
    *@param value {String} primitive value
    *@throws in case of failure
  */
  set_text(label,value)
  {
    if(!vmp_isstring(value))
      vmp_except("Vmp_JData.set_text(value) bad type");  
    this.set_value(label,value,"set_text");
  }
  /**
    *Gets a text primitive in jbody
    *@param label {String} primitive label
    *@throws in case of failure
    *@return {String} primitive value
  */
  get_text(label)
  {
    let value=this.get_value(label,"get_text");
    if(!vmp_isstring(value))
      vmp_except("Vmp_JData.get_text() bad return value");
    return value;
  }
  /**
    *Insert an integer primitive in jbody
    *@param label {String} primitive label
    *@param value {Integer} primitive value
    *@throws in case of failure
  */
  set_integer(label,value)
  {
    if(!vmp_isint(value))
      vmp_except("Vmp_JData.set_integer(value) bad type");
    this.set_value(label,value,"set_integer");
  }
  /**
    *Gets an integer primitive in jbody
    *@param label {String} primitive label
    *@throws in case of failure
    *@return {String} primitive value
  */
  get_integer(label)
  {
    let value=this.get_value(label,"get_integer");
    if(!vmp_isint(value))
      vmp_except("Vmp_JData.get_integer() bad return value");
    return value;
  }
  /**
    *Insert an integer primitive in jbody(between min and max)
    *@param label {String} primitive label
    *@param min {Integer} minimum value of the data
    *@param max {Integer} maximum value of the data
    *@param value {Integer} primitive value
    *@throws in case of failure
   */
   set_integer_range(label,min,max,value)
   {
     if(!vmp_isint(value,min,max))
      vmp_except("Vmp_JData.set_integer_range(value) bad type");
     this.set_value(label,value,"set_integer_range");
   }
   /**
     *Gets an integer primitive in jbody(between min and max)
     *@param label {String} primitive label
     *@param min {Integer} minimum value of the data
     *@param max {Integer} maximum value of the data
     *@throws in case of failure
     *@return primitive value
   */
   get_integer_range(label,min,max)
   {
     let value=this.get_value(label,"get_integer_range");
     if(!vmp_isint(value,min,max))
       vmp_except("Vmp_JData.get_integer_range() bad return value");
     return value;
   }
   /**
     *Insert a real primitive in jbody
     *@param label {String} primitive label
     *@param value {Number} primitive value
     *@throws in case of failure
   */
   set_real(label,value)
   {
     if(!vmp_isnumber(value))
       vmp_except("Vmp_JData.set_real(value) bad type");
     this.set_value(label,value,"set_real");
   }
   /**
     *Gets a real primitive in jbody
     *@param label {String} primitive label
     *@throws in case of failure
     *@return {Number} primitive value
   */
   get_real(label)
   {
     let value=this.get_value(label,"get_real");
     if(!vmp_isint(value))
       vmp_except("Vmp_JData.get_real() bad return value");
     return value;
   }
   /**
     *Insert a real primitive in jbody(between min and max)
     *@param label {String} primitive label
     *@param min {Integer} minimum value of the data
     *@param max {Integer} maximum value of the data
     *@param value primitive value
     *@throws in case of failure
   */
   set_real_range(label,min,max,value)
   {
     if(!vmp_isreal(value,min,max))
      vmp_except("Vmp_JData.set_real_range(value) bad type");
     this.set_value(label,value,"set_real_range");
   }
   /**
     *Gets a real primitive in jbody(between min and max)
     *@param label {String} primitive label
     *@param min {Number} minimum value of the data
     *@param max {Number} maximum value of the data
     *@throws in case of failure
     *@return {Number} primitive value or except in case of failure
   */
   get_real_range(label,min,max)
   {
     let value=this.get_value(label,"get_real_range");
     if(!vmp_isnumber(value,min,max))
       vmp_except("Vmp_JData.get_real_range() bad value");
     return value;
   }
   /**
     *Insert an boolean in jbody
     *@param label {String} primitive label
     *@param value {Boolean} primitive value
     *@throws in case of failure
   */  
   set_bool(label,value)
   {
     if(!vmp_isbool(value))
       vmp_except("Vmp_JData.set_bool(value) bad type");
     this.set_value(label,value,"set_bool");
   }
   /**
     *Gets an boolean in jbody
     *@param label {String} primitive label
     *@throws in case of failure
     *@return {Boolean} primitive value
   */
   get_bool(label)
   {
     let value=this.get_value(label,"get_bool");
     if(!vmp_isbool(value))
       vmp_except("Vmp_JData.get_bool() bad return value");
     return value;
   }
   /**     
     *Create a empty list primitive in the jbody.
     *@param label {String} primitive label
     *@param jtype {String} type of json data entered in the list(jtype="" generic list not typed)
     *@throws in case of failure
     *@return {Vmp_JList} primitive value
   */
   new_list(label,jtype="")
   {
     let ret=new Vmp_JList();
     ret.set([],jtype);
     this.set_value(label,ret.list_,"new_list");
     return ret;
   }
   /**
     *Insert a list primitive in the jbody.
     *@param label {String} primitive label
     *@param list {Vmp_JList} input list to insert
     *@throws in case of failure      
   */        
   set_list(label,list)
   {
     if(!(list instanceof Vmp_JList))
       vmp_except("Vmp_JData.set_list(list) bad value");
     this.set_value(label,list.list_,"set_list");
     return ret;
   }
   /**
     *Gets a list primitive in the jbody.
     *@param label {String} primitive label
     *@param jtype type of json data entered in the list(jtype="" generic list not typed)
     *@throws in case of failure
     *@return {Vmp_JList} primitive value
    */  
    get_list(label,jtype="")
    {
      let value=this.get_value(label,"get_list");
      let list=new Vmp_JList();
      try
      {
        list.set(value,jtype); 
      }
      catch(error)
      {
        vmp_except("Vmp_JData.get_list() bad return value");
      }
      return list;
    }
    /**
      *Create a empty table primitive in the jbody.
      *@param label {String} primitive label
      *@param jtype {String} type of json data entered in the data object in table(jtype="" generic table not typed)
      *@throws in case of failure
      *@return {Vmp_JTable} primitive value
    */
    new_table(label,jtype="")
    {
      let ret=new Vmp_JTable();
      ret.set({},jtype);
      this.set_value(label,ret.table_,"new_table");
      return ret;
    }
    /**
      *Create a table primitive in the jbody.
      *@param label {String} primitive label
      *@param table {Vmp_JTable} input table to insert
      *@throws in case of failure     
    */
    set_table(label,table)
    {
      if(!(list instanceof Vmp_JTable))
        vmp_except("Vmp_JData.set_table(table) bad value");
      this.set_value(label,table.table_,"set_table");
      return ret;
    }
    /**   
      *Gets a table primitive in the jbody.
      *@param label {String} primitive label
      @param jtype type of json data entered in the data object in table(jtype="" generic table not typed)
      *@throws in case of failure 
      *@return {Vmp_JTable} primitive value
    */
    get_table(label,jtype="")
    {
      let value=this.get_value(label,"get_table");
      let table=new Vmp_JTable();
      try
      {
        table.set(value,jtype); 
      }
      catch(error)
      {
        vmp_except("Vmp_JData.get_table() bad return value");
      }
      return table;
    }
    /**   
      *Insert an args primitive in the jbody(string vector).
      *@param label {String} primitive label
      *@param value {ArrayString} primitive value
      *@throws in case of failure 
      *@return {Vmp_JTable} primitive value 
    */       
    set_args(label,args)
    {
      if(!vmp_isarray(args))
        vmp_except("Vmp_JData.set_args(args) bad value");
      for(arg of args)
      {
        if(!vmp_isstring(arg))
          vmp_except("Vmp_JData.set_args(args) bad value");
      }
      this.set_value(label,args,"set_args");
    }
    /**
      *Gets an args primitive in jbody(string vector).
      *@param label {String} primitive label
      *@throws in case of failure 
      *@return {ArrayString} primitive value
    */        
    get_args(label)
    {
      let args=this.get_value(label,"get_args");
      if(!vmp_isarray(args))
        vmp_except("Vmp_JData.get_args() bad return value");
      for(arg of args)
      {
        if(!vmp_isstring(arg))
          vmp_except("Vmp_JData.get_args() bad return value");
      }
      return args;
    }
    /**
      *Insert an vars primitive in the jbody(table<string,string>).
      *@param label {String} primitive label
      *@param vars {ObjectString} primitive value
      *@throws in case of failure 
    */
    set_vars(label,vars)
    {
      if(!vmp_isobj(vars))
        vmp_except("Vmp_JData.set_vars(vars) bad value");
      for(v in vars)
      {
        if(!vmp_isstring(vars[v]))
          vmp_except("Vmp_JData.set_args(vars) bad value");
      }
      this.set_value(label,vars,"set_args");
    }
    /**
      *Gets an vars primitive in the jbody(table<string,string>).
      *@param label {String} primitive label
      *@throws in case of failure 
      *@return {ObjectString} primitive value
    */        
    get_vars(label)
    {
      let vars=this.get_value(label,"get_vars");
      if(!vmp_isobj(vars))
        vmp_except("Vmp_JData.get_vars() bad return value");
      for(v in vars)
      {
        if(!vmp_isstring(vars[v]))
          vmp_except("Vmp_JData.get_vars() bad return value");
      }
      return args;
    }
    /**   
      *Insert a json object primitive in the jbody(custom json object).if obj == Undefined create empty custom object
      *@param label {String} primitive label
      *@param obj {Object} json object to insert(if obj == undefined create empty object)
      *@throws in case of failure 
    */
    set_custom(label,obj=undefined)
    {
      if(obj == undefined)
        obj={}
      else if(!vmp_isobj(obj))
        vmp_except("Vmp_JData.set_custom(obj) bad value");
      this.set_value(label,obj,"set_custom");
    }
    /**
      *Gets json object primitive in the jbody(custom json object).
      *@param label {String} primitive label
      *@throws in case of failure 
      *@return {Object} primitive object
    */        
    get_custom(label)
    {
      let value=this.get_value(label,"get_custom");
      if(!vmp_isobj(obj))
        vmp_except("Vmp_JData.get_custom() bad return value");
      return value;
    }
    /**
      *Insert a json data object(subtype)in the jbody.
      *@param label {String} primitive label
      *@param jdata {Vmp_JData} json data object
      *@throws in case of failure
    */
    set_subtype(label,jdata)
    {
      if(!jdata instanceof Vmp_JData)
        vmp_except("Vmp_JData.set_subtype(jdata) bad value");
      this.set_value(label,jdata.obj_,"set_subtype");
    }
    /**
      *Gets an pointer json data object(subtype) in the jbody.
      *@param label {String} primitive label
      *@throws in case of failure      
      *@return {Vmp_JData} primitive data
    */
    get_subtype(label)
    {
      let value=this.get_value(label,"get_subtype");
      let jdata=new Vmp_JData();
      try
      {
        jdata.set_data(value);
      }
      catch(error)
      {
        vmp_except("Vmp_JData.get_subtype() bad return value");
      }
      return jdata;
    }
    /**
      *Insert a buffer primitive in the jbody.
      *@param label {String} primitive label
      *@param buf {Vmp_Buf} Buffer
      *@throws in case of failure 
    */
    set_buf(label,buf)
    {
      if(!(buf instanceof Vmp_Buf))
        vmp_except("Vmp_JData.set_buf(buf) bad value");
      buf.index();
      this.set_value(label,buf.read_xstr(buf.size()),"set_buf");
    }
    /**
      *Gets a buffer primitive in the jbody.
      *@param label {String} primitive label
      *@throws in case of failure 
      *@return {Vmp_Buf} primitive buffer
    */       
    get_buf(label)
    {
      let value=this.get_value(label,"get_subtype");
      let buf=Vmp_Buf();
      try
      {
        buf.write_xstr(value);
      }
      catch(error)
      {
        vmp_except("Vmp_JData.get_buf() bad return value");
      }
      return buf;
    }
    /**
      *Insert a netaddress primitive in the jbody.
      *@param label {String} primitive label
      *@param address {String} input net address
      *@param address {String} input net service
      *@throws in case of failure 
    */
    set_netaddress(label,address,service)
    {
      if(!vmp_isstring(value))
        vmp_except("Vmp_JData.set_netaddress(address) bad type");
      if(!vmp_isstring(service))
        vmp_except("Vmp_JData.set_netaddress(service) bad type");
      try
      {
        vmp_unicode_str_todigit_range(service,1,65535);
      }
      catch(error)
      {
        vmp_except("Vmp_JData.set_netaddress(service) bad type");
      }
      obj={address:address,service:service};
      this.set_value(label,obj,"set_netaddress");
    }
    /**
      *Gets a netaddress primitive in the jbody.
      *@param label {String} primitive label
      *@throws in case of failure 
      @return {Object} (address:'address',service:'service')
    */ 
    get_netaddress(label)
    {
      let value=this.get_value(label,"get_netaddress");
      if(!vmp_isobj(value) || Object.keys(value).length != 2 || !('address' in value) || !('service' in value))
        vmp_except("Vmp_JData.get_netaddress() bad return value");
      return value;
    }
    /**
      *Insert a macaddress primitive in the jbody.
      *@param label {String} primitive label
      *@param mac {String} macaddress
      *@throws in case of failure 
    */   
    set_macaddress(label,mac)
    {
      if(!vmp_is_macaddress_raw(mac))
        vmp_except("Vmp_JData.set_macaddress(mac) bad type");
      this.set_value(label,mac,"set_macaddress");
    }
    /**
      *Gets a macaddress primitive in the jbody.
      *@param label {String} primitive label
      *@return {String} macaddress
      *@throws in case of failure
    */ 
    get_macaddress(label)
    {
      let mac=this.get_value(label,"get_macaddress");
      if(!vmp_is_macaddress_raw(mac))
        vmp_except("Vmp_JData.set_macaddress() bad return type");
      return mac;
    }
};
/**
  *Returns the full name of the data type json(example json::'jtype')
  *@param jtype short name data type json
  *@return {string} full name data type json 
*/
function vmp_japi_type(jtype){return `json::${jtype}`;}
/**
  *Returns json type list [subtype] or []
  *@param jtype json type in list("" list not typed)
  *@return {Array} list type
*/
function vmp_japi_type_list(jtype=""){return `[${jtype}]`;}
/**
  *Returns json type teble {subtype} or {}
  *@param jtype json type in table("" table not typed)
  *@return {string} table type
*/
function vmp_japi_type_table(jtype=""){return `{${jtype}}`;}
/**
  *Create a json data vmp_japi_type("void")
  *<pre>
  *{ 
  *  jtype: json::japi("void"),
  *  jbody: {}
  *}
  *</pre>
  *@return {Vmp_JData} json data object
*/
function vmp_japi_void()
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("void"));
  return ret;
}
/**
  *Create a json data vmp_japi_type("[datatype]") or vmp_japi_type("[]") for generic list
  * <pre>
  *{
  *  jtype: vmp_japi_type("[datatype]"),
  *  jbody:
  *  {
  *    datatype:'datatype',
  *    list: 'list'
  *  }
  *}
  *</pre>
  *@param obj {String_or_Vmp_JList} list or datatype for new list
  *@throws in case of failure
  *@return {Vmp_JData} json data
*/
function vmp_japi_list(obj="")
{
  let jdata=new Vmp_JData();
  if(obj instanceof Vmp_JList)
  {  
    jdata.new_data(obj,vmp_japi_type_table(obj.jtype_));
    jdata.set_text("datatype",obj.jtype_);
    jdata.set_list("list",obj);
  }
  else if(vmp_isstring(obj))
  {
    jdata.new_data(obj,vmp_japi_type_table(obj));
    jdata.set_text("datatype",obj);
    jdata.new_list("list",obj);
  }
  else
    vmp_except("vmp_japi_list(obj) bad type");
  return jdata; 
}
/**
  *Create a json data {'datatype'} or "{}" for generic table
  *<pre>
  *{
  * jtype: {'datatype'}, 
  * jbody:
  * {
  *   datatype:'datatype',
  *   table:'table'
  * }
  *</pre>
  *@param table {String_or_Vmp_JList} table object or datatype for new table
  *@return {Vmp_JData} json data
*/
function vmp_japi_table(table="")
{
  let jdata=new Vmp_JData();
  if(obj instanceof Vmp_JTable)
  {  
    jdata.new_data(obj,vmp_japi_type_table(obj.jtype_));
    jdata.set_text("datatype",obj.jtype_);
    jdata.set_list("table",obj);
  }
  else if(vmp_isstring(obj))
  {
    jdata.new_data(obj,vmp_japi_type_table(obj));
    jdata.set_text("datatype",obj);
    jdata.new_list("table",obj);
  }
  else
    vmp_except("vmp_japi_table(obj) bad type");
  return jdata; 
}
/**
  *Create a json data vmp_japi_type("status") used to request information about your status at a process
 *<pre>
 *{ 
 *  jtype: vmp_japi_type("status"),
 *  jbody: 
 *  {
 *  }
 *}
 *  </pre>
 *@return {Vmp_JData} json data
*/
function vmp_japi_status()
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("status"));
  return ret;
}
/**
  *Create a json data login
  *@param user {String} user field(read data jdata.get_text("user"))
  *@param password {String} password field(read data jdata.get_text("password"))
  *@param md5_password {Boolean} if true password is encrypted in md5(read data jdata.get_bool("md5_password"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_login(user,password,md5_password=false) 
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("login"));
  ret.set_text("user",user);
  ret.set_text("password",password);
  ret.set_bool("md5_password",md5_password);
  return ret;
}
/**
  *Create a json data vmp_japi_type("passwd") used for change user password
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("passwd"),
  *  jbody: 
  *  {
  *    oldpassword:'oldpassword',
  *    password:'password',
  *    md5_password:bool
  *  }
  *}
  *</pre>
  *@param oldpassword {String} old password(read data jdata.get_text("oldpassword"))
  *@param password {String} new password(read data jdata.get_text("password"))
  *@param md5_password {Boolean} if true the input password and old password parameter are encrypted with the md5 function,default false (read data jdata.get_bool("md5_password"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_passwd(oldpassword,password,md5_password=false)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("passwd"));
  ret.set_text("oldpassword",oldpassword);
  ret.set_text("password",password);
  ret.set_bool("md5_password",md5_password); 
  return ret;
}
/**
  *Create a json data vmp_japi_type("userlist") used for request list users
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("userlist"),
  *  jbody: 
  *  {
  *  }
  *}
  *</pre>
  *@return {Vmp_JData} json data
*/
function vmp_japi_userlist()
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("userlist"));
  return ret;
}
/**
  *Create a json data vmp_japi_type("useradd") used for add a new user
  *<pre>
  *{ 
  *  jtype:vmp_japi_type("useradd"),
  *  jbody: 
  *  {
  *    user:'user',
  *    password:'password',
  *    permits:permits,
  *    only_localhost:only_localhost
  *  }
  *}
  *</pre>
  *@param user {String} user (read data jdata.get_text("user"))
  *@param password password (read data jdata.get_text("password"))
  *@param permits permits user (read data jdata.get_integer_range("permits",0,Number.MAX))
  *@param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_useradd(user,password,permits,only_localhost,md5_password=false)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("useradd"));
  ret.set_text("user",user);
  ret.set_text("password",password);
  ret.set_integer_range("permits",0,Number.MAX,permits);
  ret.set_bool("only_localhost",only_localhost);
  ret.set_bool("md5_password",md5_password);
  return ret;
}
/**
  *Create a json data vmp_japi_type("usermod") used for Change user values
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("usermod"),
  *  jbody: 
  *  {
  *    user:'user',
  *    permits:permits,
  *    only_localhost:only_localhost
  *  }
  *}
  *</pre>
  *@param user user (read data json::jdata.get_text("user"))
  *@param permits permits user (read data json::jdata.get_integer_range("permits",0,Number.MAX))
  *@param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_usermod(user,permits,only_localhost)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("usermod"));
  ret.set_text("user",user);
  ret.set_integer_range("permits",0,Number.MAX,permits);
  ret.set_bool("only_localhost",only_localhost);
  return ret;
}
/**
  *Create a json data vmp_japi_type("userpasswd") used for Change user password
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("userpasswd"),
  *  jbody: 
  *  {
  *    user:'user',
  *    password:'password',
  *    md5_password:md5_password
  *  }
  *}
  *</pre>
  *@param user user (read data jdata.get_text("user"))
  *@param password new password(read data jdata.get_text("password"))
  *@param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data jdata.get_bool("md5_password"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_userpasswd(user,password,md5_password=false)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("userpasswd"));
  ret.set_text("user",user);
  ret.set_text("password",password);
  ret.set_bool("md5_password",md5_password);
  return ret;
}
/**
  *Create a json data vmp_japi_type("userdel") used for delete user
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("userdel"),
  *  jbody: 
  *  {
  *    user:'user'
  *  }
  *}
  *</pre>
  *@param user user (read jdata.get_text("user"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_userdel(user)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("userdel"));
  ret.set_text("user",user);
  return ret;
}
/**
  *Create a json data vmp_japi_type("peerlist") used for request list peers
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("peerlist"),
  *  jbody: 
  *  {
  *  }
  *}
  *</pre>
  *@param obj object where the json data is inserted
  *@return {Vmp_JData} json data
*/
function vmp_japi_peerlist()
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("peerlist"));
  return ret;
}
/** 
  *Create a json data vmp_japi_type("peeradd") used to add a peer node in the access table
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("peeradd"),
  *  jbody: 
  *  {
  *    fingerprint:'fingerprint',
  *    subjetc:'subject',
  *    permits:permits
  *  }
  *}
  *</pre>
  *@param fingerprint fingerprint peer (read data jdata.get_text("fingerprint"))
  *@param subject subject peer (read data jdata.get_text("subject"))
  *@param permits permits peer (read data jdata.get_integer_range("permits",0,Number.MAX))
  *@return {Vmp_JData} json data
*/
function vmp_japi_peeradd(fingerprint,subject,permits)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("peeradd"));
  ret.set_text("fingerprint",fingerprint);
  ret.set_text("subject",subject);
  ret.set_integer_range("permits",Vmp_jrp_min_permits,Vmp_jrp_max_permits,permits);
  return ret;
}
/**
  *Create a json data vmp_japi_type("peermod") used to modify a peer node in the access table
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("peermod"),
  *  jbody: 
  *  {
  *    fingerprint:'fingerprint',
  *    permits:permits
  *  }
  *}
  *</pre>
  *@param fingerprint fingerprint peer (read data jdata.get_text("fingerprint"))
  *@param permits permits peer (read data jdata.get_integer_range("permits",0,Number.Max))
  *@return {Vmp_JData} json data
*/
function vmp_japi_peermod(fingerprint,permits)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("peermod"));
  ret.set_text("fingerprint",fingerprint);
  ret.set_integer_range("permits",Vmp_jrp_min_permits,Vmp_jrp_max_permits,permits); 
  return ret; 
}
/**
  *Create a json data vmp_japi_type("peerdel") used to delete a peer node in the access table
  *<pre>
  *{ 
  *  jtype: vmp_japi_type("peerdel"),
  *  jbody: 
  *  {
  *    fingerprint:'fingerprint',
  *  }
  *}
  *</pre>
  *@param fingerprint fingerprint peer (read data jdata.get_text("fingerprint"))
  *@return {Vmp_JData} json data
*/
function vmp_japi_peerdel(fingerprint)
{
  let ret=new Vmp_JData();
  ret.new_data(vmp_japi_type("peerdel"));
  ret.set_text("fingerprint",fingerprint);
  return ret;  
}
