/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 11/11/2024
 */

#include <time.h>
#include <sys/time.h>

#ifndef VAMPIRIA_VMP_TIME_H

#define VAMPIRIA_VMP_TIME_H 1

namespace vampiria { namespace vmp { namespace time {

//!time base type(s.ms)
typedef vmp_real Time;

//! timeval type
typedef struct timeval *Timeval;

//! Alloc Timeval
/*!
    @return timeval struct
*/
vmp::time::Timeval timeval_new();

//!Convert the timeval format Time
/*!
    @param tval input timeval time
    \return time 
*/
vmp::time::Time timeval_to_time(vmp::time::Timeval tval);

//!Convert the Time format timeval(new dynamic timeval struct is allocated)
/*!
     @param time input time
     @return timeval converterd
*/
vmp::time::Timeval time_to_timeval(vmp::time::Time time);

//! Free Timeval struct and settin (*pval) = 0
/*!
    @param ptval pointer to timeval
*/
void timeval_free(vmp::time::Timeval *ptval);

//!Causes the calling thread to sleep for time
/*!
    @param time time to sleep thread
*/
void vsleep(vmp::time::Time time);

//!Returns the time as the number of seconds since the Epoch,1970-01-01 00:00:00 +0000 (UTC).
/*!
    return time,except error
*/
vmp_uint time_wrap();

//!Returns the time in string format
/*!
    @param time the number of seconds since the Epoch,1970-01-01 00:00:00 +0000 (UTC).if time=0 use present time
    @return current time string
*/
vmp::str ctime_wrap(vmp_uint time=0);

//!Returns the time in time representation, expressed in Coordinated Universal Time (UTC)
/*!
    @param time the number of seconds since the Epoch,1970-01-01 00:00:00 +0000 (UTC).if time=0 use present time
    @return current gmtime string
*/
vmp::str gmtime_wrap(vmp_uint time=0);

//!Represents local calendar time
class Localtime
{
    private:
        struct tm tm_;/*!<Internal usage*/
        time_t time_;/*!<Internal usage*/
        
        //!Internal usage
        vmp::vector<vmp_int> month_day(vmp_int year_v);
        
        //! Internal usage
        vmp_bool day_check(vmp_int year_v,vmp_int month_v,vmp_int day_v);

        //! Internal usage
        void croniter_expand(vmp::vector<vmp_uint> *ret,vmp::str expr,vmp_int min,vmp_int max);
        
        //! Internal usage
        vmp::vector<vmp_uint> croniter_expand_base(vmp::str expr,vmp_int min,vmp_int max);
        
        //! Internal usage
        vmp::vector<vmp_uint> croniter_expand_day_month(vmp::str expr,vmp_int year_v,vmp_int month_v);
        
        //! Internal usage
        vmp::vector<vmp_uint> croniter_expand_day_week(vmp::str expr,vmp_int year_v,vmp_int month_v);

        //! Internal usage
        vmp_uint croniter_value(vmp_uint value,vmp::vector<vmp_uint> accepted);
        
        //!Internal usage.if week_day=true use week days field,otherwise use month days
        vmp_uint croniter_calc(vmp::vector<vmp_uint> mins,vmp::vector<vmp_uint> hours,vmp::vector<vmp_uint> months,vmp::str days,vmp_bool week_days);
    public:
        //! A Constructor
        /*!
            @param time the number of seconds since the Epoch,1970-01-01 00:00:00 +0000 (UTC).if time=0 use present time
        */
        Localtime(vmp_uint time=0);
        
        //! A Constructor
        /*!
            @param year_v year of date(1970-2037).if value is -1 does not change    
            @param month_v month of year(1-12).if value is -1 does not change
            @param day_v day of month(1-31 depends on the month).if value is -1 does not change
            @param hour_v clock hours(0-23).if value is -1 does not change
            @param min_v clock minutes(0-59).if value is -1 does not change
            @param sec_v clock seconds(0-59).if value is -1 does not change
            @ref time_
        */
        Localtime(vmp_int year_v,vmp_int month_v,vmp_int day_v,vmp_int hour_v,vmp_int min_v,vmp_int sec_v);
        
        //! A Destructor
        ~Localtime();

        //! Set a new date time
        /*!
            @param time the number of seconds since the Epoch,1970-01-01 00:00:00 +0000 (UTC).if time=0 use present time
            @return time or except in case of failure
        */
        vmp_uint set(vmp_uint time=0);
        
        //! Change the date of the structure
        /*!
            @param year_v year of date(1970-2037).if value is -1 does not change    
            @param month_v month of year(1-12).if value is -1 does not change
            @param day_v day of month(1-31 depends on the month).if value is -1 does not change
            @param hour_v clock hours(0-23).if value is -1 does not change
            @param min_v clock minutes(0-59).if value is -1 does not change
            @param sec_v clock seconds(0-59).if value is -1 does not change
            @ref time_
            @return new time value or except in case of failure
        */
        vmp_uint set(vmp_int year_v,vmp_int month_v,vmp_int day_v,vmp_int hour_v,vmp_int min_v,vmp_int sec_v);
        
        //! Returns the number of days of the year set in localtime.
        /*!
            @return 365 normal years,366 leap year
        */
        vmp_uint get_year_days();
        
        //! Set the date from the day number of the year in input
        /*!
            @param year_v year of date(1970-2037)
            @param nday Number of day of year(1-365/366)
            @param hour_v clock hours(0-23).if value is -1 does not change
            @param min_v clock minutes(0-59).if value is -1 does not change
            @param sec_v clock seconds(0-59).if value is -1 does not change
            @ref time_
            @return new time value or except in case of failure
        */
        vmp_uint set_from_year_day(vmp_int year_v,vmp_uint nday,vmp_int hour_v,vmp_int min_v,vmp_int sec_v);
       
        //! Calculates the time of the next event from the time stored in the object and the input crontab. Stores the new time in the object
        /*!
            @param crontab string in crontab mode (minute hour day_of_the_month month day_of_the_week[0-6 sunday-saturday]) 
            @ref time_
            @return new time value or except in case of failure
        */
        vmp_uint next_cronizer(vmp::str crontab);
        
        //! Returns setting time
        /*!
            @ref time_
        */
        vmp_uint time();
        
        //! Returns the number of seconds after the minute, in the range 0 to 59.        
        /*!
            @ref tm_
        */
        vmp_uint sec();

        //! Returns the number of minutes after the hour, in the range 0 to 59.    
        /*!
            @ref tm_
        */
        vmp_uint min();

        //! Returns the number of hours past midnight, in the range 0 to 23.
        /*!
            @ref tm_
        */
        vmp_uint hour();

        //! Returns the day of the month, in the range 1 to 31
        /*!
            @ref tm_
        */
        vmp_uint mday();
        
        //! Returns days of the week in abbreviated form(Sun,Mon,Tue,Wed,Thu,Fri,Sat)
        /*!
            @ref tm_
        */
        vmp::str wday();
        
        //! Returns the number of days since Sunday, in the range 0=Sun to 6=Sat
        /*!
            @ref tm_
        */
        vmp_uint wday_n();

        //! Returns the month in the short form(Jan,Feb,Mar,Apr,May,June,July,Aug,Sept,Oct,Nov,Dec)
        /*!
            @ref tm_
        */
        vmp::str month();

        //! Returns The number of months since January, in the range 1 to 12
        /*!
            @ref tm_
        */
        vmp_uint month_n();

        //! The number of years
        /*!
            @ref tm_
        */
        vmp_uint year();
        
        //!Returns the time in string format
        /*!
            @return time string
        */
        vmp::str ctime();

        //!Returns the time in time representation, expressed in Coordinated Universal Time (UTC)
        /*!
            @return gmtime string
        */
        vmp::str gmtime();
        
        //! Return the day number of the year stored
        /*!
            @return the day number of the year stored
        */
        vmp_uint year_day();
        
        //! Returns true if it reads summer time
        /*!
            @return true if summer type or false if normal time or except if the information is not available
        */
        vmp_bool summertime();
};

//!Timer object utility
class Timer
{
    private:
        struct timeval tinit_;/*!< Init time*/    
    public:    
        //! A constructor        
        Timer();

        //! A destructor
        ~Timer();
        
        //!Set the initial time reference to the current time
        /*!
            @sa tinit_
        */
        void init();
        
        //! Returns the current time than the time init
        /*!
            @sa tinit_             
            @return current time
        */
        vmp::time::Time now();
};

}}}

#endif

