/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/05/2024
 */
 
#ifndef VAMPIRIA_VMP_LIB_FS_H

#define VAMPIRIA_VMP_LIB_FS_H 1

namespace vampiria { namespace vmp { namespace fs {

//!mode inode type
typedef mode_t Mode;

//!Delete a file system name
/*!
    @param path path name
    @return void or except in case of failure
*/
void unlink_wrap(vmp::str path);

//!Format the directory path string by adding / if it is not there
/*!
    @param dirpath string input
    @return directory path format
*/
vmp::str format_dirpath(vmp::str dirpath);

//!Create a system path by concatenating the input dir path with subpath
/*!
    @param dirpath directory path
    @param subdirectories or files
    @return concatenated path. If an empty string is subpath return formatted dir path
*/
vmp::str union_path(vmp::str dirpath,vmp::str subpath="");

//!changes the permissions to a file
/*!
    @param path file file
    @param mode specifies the mode permissions(0ugo permission filesystem) 
*/
void chmod_wrap(vmp::str path,vmp::fs::Mode mode);

//! Create a directory named pathname
/*!
    @param dirpath pathname
    @param mode specifies the mode for the new directory(0ugo permission filesystem) 
    @return void or except in case of failure
*/
void mkdir_wrap(vmp::str dirpath,vmp::fs::Mode mode);

//!Deletes a directory and its contents(recursive mode)
/*!
    @param dirpath directory path
    @return void or except in case of failure
*/
void rmdir_wrap(vmp::str dirpath);

//!Changes the current working directory
/*!
    @param path dierctory to working
    @return void or except in case of failure
*/ 
void chdir_wrap(vmp::str path);

//!Check if dirpath is a valid path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool isdir(vmp::str dirpath);

//!Check if dirpath is a valid read path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool isrdir(vmp::str dirpath);

//!Check if dirpath is a valid write path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool iswdir(vmp::str dirpath);

//!Check if dirpath is a valid exec path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool isxdir(vmp::str dirpath);

//!Return string containing an absolute pathname that is the current working directory of the calling  process.
/*!
    @return current working directory or except in case of failure
*/
vmp::str getcwd_wrap();

//!Listen directory
/*!
    @param dirpath directory to listen
    @return listen directory or except in case of failure
*/
vmp::vector<vmp::str> listendir(vmp::str dirpath);

//!Check to see if path is a file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool isfile(vmp::str path);

//!Check to see if path is a read file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool isrfile(vmp::str path);

//!Check to see if path is a write file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool iswfile(vmp::str path);

//!Check to see if path is a exec file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool isxfile(vmp::str path);

//! Getting a file extension from a filename
/*!
    @param path file path
    @return file extension or "" if there is no extension
*/
vmp::str file_ext(vmp::str path);

//! c mode_t structure wrapper,contains the type of file 
class Mode_t_wrap
{
    private:
        mode_t mode_;/*!<Internal usage*/
    public:
        //! A constructor
        Mode_t_wrap();
        
        //! A destructor
        ~Mode_t_wrap();
        
        //! set data
        void set(mode_t mode);
        
        //!is it a regular file?
        /*!
            @return true ok,otherwise false
        */
        vmp_bool isreg();
        
        //! is directory?
        /*!
            @return true ok,otherwise false
        */
        vmp_bool isdir();
        
        //! is character device?
        /*!
            @return true ok,otherwise false
        */
        vmp_bool ischr();
        
        //! is block device?
        /*!
            @return true ok,otherwise false
        */
        vmp_bool isblk();
        
        //! is FIFO (named pipe)?
        /*!
            @return true ok,otherwise false
        */
        vmp_bool isfifo();
        
        //! is symbolic link?
        /*!
            @return true ok,otherwise false
        */
        vmp_bool islnk();
        
        //! is socket? 
        /*!
            @return true ok,otherwise false
        */
        vmp_bool issock();
};

//! Wrapper stat e struct stat.Describes information about a file in the filesystem
class FileStat
{
    private:
        vmp::str filepath_;/*!<file path*/
        struct stat stat_;/*!< c struct*/
    public:
        //! A constructor
        FileStat();
        
        //! A constructor
        /*!
            @param filepath file path
        */
        FileStat(vmp::str filepath);
        
        //! A destructor
        ~FileStat();
        
        //! Reset class
        void reset();
        
        //! Set file
        /*!
            @param filepath filepath
        */
        void set(vmp::str filepath);

        //! Returns file path
        /*!
            @return file path
        */
        vmp::str filepath();
        
        //! Get id of device containing file
        /*!
            @param dev output dev
            @return void or except in case of failure
        */
        void dev(vmp::Dev_t_wrap *dev);
        
        //! Get inode number
        /*!
            @return inode
        */
        vmp_uint ino();
        
        //! Get mode file
        //! Get id of device containing file
        /*!
            @param output output mode
            @return void or except in case of failure
        */
        void mode(vmp::fs::Mode_t_wrap *mode);
        
        //! Get number of hard links
        /*!
            @return number of hard link
        */
        vmp_uint nlink();
        
        //! Get user id of owner
        /*!
            @return user id
        */
        vmp_uint uid();
        
        //! Get group id of owner
        /*!
            @return group id
        */
        vmp_uint gid();
        
        //! return device ID (if special file)
        /*!
            @param dev output dev
            @return void or except in case of failure
        */
        void rdev(vmp::Dev_t_wrap *dev);
        
        //! Get total size,in bytes
        /*!
            @return size
        */
        vmp_size size();
        
        //! Get blocksize for file system I/O
        /*!
            @return blocksize
        */    
        vmp_size blksize(); 
        
        //! Get number of 512B blocks allocated
        /*!
            @return blocks
        */
        vmp_size blocks();  
        
        //! Get time of last access
        /*!
            @return time
        */
        vmp_uint atime();   
        
        //! Get time of last modification
        /*!
            @return time
        */
        vmp_uint mtime();   
    
        //! Get time of last status change
        /*!
            @return time
        */
        vmp_uint ctime();
};

}}}

#endif

