/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 07/06/2024
 */

#include "config.h"
#include "vmp.h"

namespace vampiria { namespace vmp { namespace env {

vmp::str author()
{
    return VAUTHOR;
}

vmp::str email()
{
    return VEMAIL;
}

vmp::str webpage()
{
    return VWEBPAGE;
}

vmp::str license()
{
    return VLICENSE;
}

vmp::str version()
{
    return VVERSION;
}

vmp::str vampiria_path(vmp::str subpath)
{
    return vmp::fs::union_path(VPATH,subpath);
}

vmp::str vampiria_frm(vmp::str subpath)
{
    return vmp::fs::union_path(VFRMWK,subpath);   
}

vmp::str vampiria_web_root()
{
    return VWEB;
}

vmp::str vampiria_web(vmp::str libname,vmp::str subpath)
{
    vmp::str spath=vmp::fs::union_path(vmp::env::vampiria_web_root(),libname);
    if(!vmp::fs::isdir(spath))
        vmp::except("Software name '%s' Not Found",libname.c_str());
    return vmp::fs::union_path(spath,subpath);
}

vmp::str vampiria_bin(vmp::str name)
{
    vmp::str path=vmp::fs::union_path(VEXEC,name);
    if(!vmp::fs::isxfile(path))
        vmp::except("Vampiria executable '%s' Not Found",name.c_str());    
    return path;
}

vmp::str vampiria_lib(vmp::str name)
{
   vmp::str path;
   vmp::unicode::str_write(&path,"%s.so",name.c_str());
   path=vmp::fs::union_path(VLIB,path);
   if(!vmp::fs::isfile(path))
       vmp::except("Vampiria lib '%s' Not Found",name.c_str());
   return path;
}

vmp::str vampiria_cache(vmp::str subpath)
{
    return vmp::fs::union_path(VCACHE,subpath);
}

vmp::str vampiria_home(vmp::str subpath)
{
    return vmp::fs::union_path(VHOME,subpath);
}

vmp::str software_root()
{
    return VSOFTWARE;
}

vmp::str software_path(vmp::str name,vmp::str subpath)
{
    vmp::str spath=vmp::fs::union_path(vmp::env::software_root(),name);
    if(!vmp::fs::isdir(spath))
        vmp::except("Software name '%s' Not Found",name.c_str());
    return vmp::fs::union_path(spath,subpath);
}

vmp_bool software_isinstalled(vmp::str dirpath)
{
    if(!vmp::fs::isdir(dirpath))
        vmp::except("Directory software '%s' Not Found",dirpath.c_str());
    vmp::str finst=vmp::fs::union_path(dirpath,VINSTALLEDF);
    return vmp::fs::isfile(finst);
}

vmp::str modules_root()
{
    return VMODULES;
}

vmp::str module_path(vmp::str name,vmp::str subpath)
{
    vmp::str mpath=vmp::fs::union_path(vmp::env::modules_root(),name); 
    if(!vmp::fs::isdir(mpath))
        vmp::except("Module name '%s' Not Found",name.c_str());
    return vmp::fs::union_path(mpath,subpath);
}

vmp::str module_main_path(vmp::str name)
{
    vmp::str mpath=vmp::env::module_path(name,VMODMAIN);
    if(!vmp::fs::isdir(mpath))
        vmp::except("Module name '%s' main directory Not Found",name.c_str());
    return mpath; 
}

vmp::str module_web_path(vmp::str name,vmp::str subpath)
{
    vmp::str s=vmp::fs::union_path(VMODULEJS,subpath);
    vmp::str mpath=vmp::env::module_path(name,s);
    if(!vmp::fs::isdir(mpath))
        vmp::except("Module name '%s' js directory %s Not Found",name.c_str(),s.c_str());
    return mpath; 
}

vmp::str module_manifest(vmp::str name)
{
    vmp::str manifest=vmp::fs::union_path(vmp::env::module_main_path(name),VMANIFEST);
    if(!vmp::fs::isfile(manifest))
        vmp::except("Module name '%s' manifest Not Found",name.c_str());
    return manifest;
}

vmp_bool module_isinstalled(vmp::str name)
{
    vmp::str mpath=vmp::env::module_main_path(name);
    vmp::env::module_manifest(name);
    return vmp::env::software_isinstalled(mpath);
}

vmp::str pkg_path(vmp::str name,vmp::str subpath)
{
    vmp::str ppath=vmp::fs::union_path(VPKG,name);
    if(!vmp::fs::isdir(ppath))
        vmp::except("Pkg name '%s' Not Found",name.c_str());
    return vmp::fs::union_path(ppath,subpath);
}

vmp::str vpkg_path(vmp::str name,vmp::str subpath)
{
    vmp::str vppath=vmp::fs::union_path(VVPKG,name);
    if(!vmp::fs::isdir(vppath))
        vmp::except("Vpkg name '%s' Not Found",name.c_str());
    return vmp::fs::union_path(vppath,subpath);
}

vmp::str frm_dtd(vmp::str dtd)
{
    vmp::str path=vmp::fs::union_path(FDTD,dtd); 
    if(!vmp::fs::isfile(path))
        vmp::except("Framework dtd '%s' Not Found",dtd.c_str());
    return path;
}

vmp::str python_exe()
{
    return VPYTHON;
}

vmp::str python_env()
{
    return VPYENV;
}

vmp::str python_moduleimp()
{
    return VPYMODULEIMP;
}

vmp::str python_main()
{
    return VPYMAIN;
}

vmp::str python_software_main(vmp::str dirsoftware)
{
    vmp::str main=vmp::fs::union_path(dirsoftware,VPYMAINAME);
    if(!vmp::fs::isfile(main))
        vmp::except("Python Software in directory '%s' Not Found",dirsoftware.c_str());
    return main;
}

vmp_bool python_files_check(vmp::str path)
{
    vmp::vector<vmp::str> v=vmp::unicode::str_split(path,"/");
    if(v.size() > 0)
    {
        v=vmp::unicode::str_regex(v[v.size()-1],"^__.*__.*",true);
        if(v.size() == 1)    
          return true;
    }
    return false;
}

vmp::str xml_hdr()
{
    return VXMLHDR;
}

vmp::str format_path(vmp::str path)
{
    if(path.size() > 0)
    {
        if(path[0] == '\\')
            return vmp::unicode::str_sub(path,1,path.size()-1);
        else if(vmp::unicode::str_regex_matching(path,"\\$\\{.*\\}\\/.*"))
        {
            vmp::vector<vmp::str> split=vmp::unicode::str_split(path,"/");
            vmp::str subpath="";
            if(split.size() > 1)
            {
                for(vmp_index i=1;i<split.size();i++)
                {
                    if(i==1)
                        vmp::unicode::str_write(&subpath,"%s",split[i].c_str());
                    else
                        vmp::unicode::str_cwrite(&subpath,"/%s",split[i].c_str());
                }
                if(path[path.size()-1] == '/')
                    vmp::unicode::str_cwrite(&subpath,"/");
            }
            vmp::str sub=vmp::unicode::str_sub(split[0],2,split[0].size()-3);
            if(sub == "")
                return vmp::env::vampiria_path(subpath);
            else if (sub == "bin")    
                return vmp::fs::union_path(VEXEC,subpath);
            else if (sub == "lib")    
                return vmp::fs::union_path(VLIB,subpath);
            else if (sub == "cache")    
                return vmp::env::vampiria_cache(subpath);
            else if (sub == "home")    
                return vmp::env::vampiria_home(subpath);
            else if (sub == "software")    
                return vmp::fs::union_path(vmp::env::software_root(),subpath);
            else if (sub == "modules")
                return vmp::fs::union_path(vmp::env::modules_root(),subpath);
            else if (sub == "pkg")    
                return vmp::fs::union_path(VPKG,subpath);
            else if (sub == "vpkg")    
                return vmp::fs::union_path(VVPKG,subpath);
            else if (sub == "frm")    
                return vmp::env::vampiria_frm(subpath);
            else
                vmp::except("vmp::env::format_path(vmp::str path=%s) bad value",path.c_str());
            
        }
        else
            return path;    
    }
    return "";
}

}}}



