/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 13/08/2024
 */

#include "rawnet.h"

namespace vampiria { namespace rawnet {

EventRawNet::EventRawNet():event::EventThreadBuf()
{
    rawnet_=0;
    evt_rawnet_reset();    
}

EventRawNet::~EventRawNet()
{
    evt_rawnet_reset();
}

void EventRawNet::evt_rawnet_reset()
{
    rawnet::close_rawnet(&rawnet_);
    ifname_="";
    filter_="";
    isread_=true;
    recv_=0;
    linktype_=-1;
    linktype_str_="";
}

vmp::str EventRawNet::evt_rawnet_iface()
{
    return ifname_;
}
        
vmp::str EventRawNet::evt_rawnet_filter()
{
    if(!isread_)
       vmp::except_s("rawnet::EventRawNet not filter in writeonly mode");
    return filter_;
}

vmp::str EventRawNet::evt_rawnet_mode()
{
    if(isread_)
        return "readwrite";
    return "writeonly";
}

vmp_int EventRawNet::evt_rawnet_linktype()
{
    return linktype_;
}
        
vmp::str EventRawNet::evt_rawnet_linktype_str()
{
    return linktype_str_;
}

void rawnet_read_ev(vmp::Buf *buf,void *arg)
{
    event::Cell *cell=(event::Cell *)arg;
    event::Manager *manager=cell->get_manager();
    rawnet::EventRawNet *event=cell->event<rawnet::EventRawNet>();
    manager->lock();
    vmp::Buf *sbuf=event->bufs_.get();
    sbuf->write_buf(buf);
    event->evt_threadbuf_send(sbuf);
    manager->unlock();
}

void rawnet_cancel_run(event::Cell *cell)
{
    rawnet::EventRawNet *event=cell->event<rawnet::EventRawNet>();
    rawnet::breakloop_rawnet(event->rawnet_);
}

void rawnet_read_run(event::Cell *cell)
{
    rawnet::EventRawNet *event=cell->event<rawnet::EventRawNet>();
    rawnet::loop_rawnet(event->rawnet_,rawnet::rawnet_read_ev,(void *)cell);
}

event::Cell *EventRawNet::evt_rawnet_rw_new(event::UI *ui,vmp::str ifname,vmp::str filter,event::EVTCBBUF recv,event::EVTCB close)
{
    rawnet::rawnet_t handle=rawnet::open_live(ifname);
    try
    {
        rawnet::filter_rawnet(handle,filter);
        rawnet_=handle;
        ifname_=ifname;
        filter_=filter;
        if(recv == 0)
            recv_=event::empty_buf_ev;
        else
            recv_=recv;
        isread_=true;
        linktype_=rawnet::datalink_rawnet(rawnet_);
        linktype_str_=rawnet::linktype_str(linktype_);
    }
    catch(vmp::exception &x)
    {
        rawnet::close_rawnet(&handle);
        evt_rawnet_reset();
        vmp::except_s(x.what());
    }
    return evt_threadbuf_new(ui,recv,rawnet::rawnet_read_run,rawnet_cancel_run,close);    
}

event::Cell *EventRawNet::evt_rawnet_w_new(event::UI *ui,vmp::str ifname,event::EVTCB close)
{
    rawnet::rawnet_t handle=rawnet::open_live(ifname);
    try
    {
        rawnet_=handle;
        ifname_=ifname;
        filter_="";
        recv_=event::empty_buf_ev;
        isread_=false;
        linktype_=rawnet::datalink_rawnet(rawnet_);
        linktype_str_=rawnet::linktype_str(linktype_);
    }
    catch(vmp::exception &x)
    {
        rawnet::close_rawnet(&handle);
        evt_rawnet_reset();
        vmp::except_s(x.what());
    }
    return evt_new(ui,-1,0,close);
}

void EventRawNet::evt_rawnet_loop()
{
    if(isread_)
        evt_threadbuf_start();
}

void EventRawNet::evt_rawnet_inject(vmp::Buf *buf)
{
    event::Manager *manager=cell_->get_manager();
    if(manager->cell_update(cell_))
        rawnet::inject_rawnet(rawnet_,buf);
}

void EventRawNet::evt_rawnet_close()
{
    if(isread_)
        evt_threadbuf_close();
    else
        evt_close();
    
}

void EventRawNet::evt_rawnet_free()
{
    if(isread_)
        evt_threadbuf_free();
    else
        evt_free();
    evt_rawnet_reset();
}

}}

