/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/08/2024
 */

#include <pcap/pcap.h>

#include "vmp.h"
#include "packet.h"
#include "event.h"

#include "rawnet/lib.h"

#ifndef VAMPIRIA_RAWNET_H

#define VAMPIRIA_RAWNET_H 1

namespace vampiria { namespace rawnet {

//!Dumper for writing packets to a file
class Dumper
{
    private:
        rawnet::dumper_t dumper_;/*!<Internal usage*/
        rawnet::rawnet_t handle_;/*!<Internal usage*/
    public:
        //! A constructor
        Dumper();

        //! A destructor
        ~Dumper();

        //!Open dumper section
        /*!
            @param wfile output file
            @param linktype (see pcap_datalink())
            @return throw exception on error
        */
        void open(vmp::str wfile,vmp_int linktype);

        //!Writes the buffered packet to the dump file
        /*!
            @param buf input packet
        */           
        void dump(vmp::Buf *buf);
        
        //!Close dumper section
        void close();
};

//!Read packages from a dump file
class Offline
{
    private:
        vmp::vector<vmp::Buf *> data_;/*!<data Internal usage*/
        vmp_int dlink_;/*!<Internal usage*/
    public:
        //! A constructor
        Offline();

        //! A destructor
        ~Offline();

        //! Reset data
        void reset();

        //! It reads packets from a dump file and stores them in the buffer vector data
        /*!
            @param file input file
            @param filter (see pcap_filter)
            @sa data_
            @sa dlink_
            @return link type file dump (see see pcap_datalink()). if error throw exception
        */
        vmp_int read(vmp::str file,vmp::str filter);

        //! Returns size packet stored in internal data
        /*!
            @sa data_
            @return data size
        */
        vmp_size data_size();

        //! Internal usage
        void add(vmp::Buf *buf);

        //! Return the package in internal data with index i
        /*!
            @param i index data
            @return the data in index or 0 if out range index 
        */
        vmp::Buf *get(vmp_index i);
};

//!It takes a dump file as input, reads it and filters the packets read. Those that are not discarded are written to the output file
/*!
    @param finput file input
    @param foutput file output
    @param filter (see pcap_filter)
    @return throw exception on error
*/
void filter_file(vmp::str finput,vmp::str foutput,vmp::str filter);

//!It takes an integer linktype as input and transforms it into string format("pcap_{linktype}")
/*!
    @param linktype base link type
    @return generic format string
*/
vmp::str linktype_str_generic(vmp_int linktype);

//!It takes an integer linktype as input and transforms it into string format.If the package is implemented in packet in the framework it returns the valid name, in other cases it returns a generic name(see raw::pkg::linktype_str_generic)
/*!
    @param linktype base link type
    @return linktype string
*/
vmp::str linktype_str(vmp_int linktype);

}}

#endif

#include "rawnet/rawevent.h"
#include "rawnet/rawui.h"
#include "rawnet/arpui.h"

