/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/09/2024
 */

#ifndef VAMPIRIA_PACKET_WEBSOCKET_HELPER_H

#define VAMPIRIA_PACKET_WEBSOCKET_HELPER_H 1

namespace vampiria { namespace packet { namespace websocket {

//! Helper for managing framing connections
class FramingHelper
{
    private:
        vmp::utils::Storage<packet::websocket::FramingExtData> storage_;/*<storage extension data*/
        vmp::utils::Storage<packet::websocket::WebSocket_Framing_P> packets_;/*<storage frame*/
        vmp::utils::Storage<vmp::Buf> bufs_;/*<storage buffer*/
        vmp::vector<FramingExtData *> exts_;/*!<extensions list*/
        vmp::Buf pingdata_;/*!<Data used from last ping to send*/
        packet::websocket::FramingExtManager *manager_;/*!<extension manager associated with the connection*/
        vmp::queue<vmp::Buf *> squeue_;/*!<Internal usage*/
        
        vmp::Buf recv_;/*<Internal usage*/
        vmp::queue<vmp::Buf *> tqueue_;/*!<temporary queue for receiving packets used if the recv buffer is full*/
        vmp::str ptype_;/*!<Packet type to read*/
        vmp::vector<packet::websocket::WebSocket_Framing_P *> frames_;/*!<Reading frames*/
        //!Internal usage
        vmp_bool check_init();
        
        //!Internal usage
        vmp_uint parse_error(vmp_uint code,vmp_uint *framecode,vmp::str *frameerr,packet::websocket::WebSocket_Framing_P *p);
        //!Internal usage
        void unmask(packet::websocket::WebSocket_Framing_P *p);
   public:
        //! A constructor
        FramingHelper();
        
        //! A destructor
        ~FramingHelper();
        
        //! init connection
        void init(packet::websocket::FramingExtManager *manager);
        
        //! reset connection
        void reset();
        
        //! Adds an extension
        /*!
            @param args extensions arguments list args[0] == "name"
            @param server if server == true use in server mode,otherwise in client mode
            @return in server mode always returns true,if the extension is not accepted it does not enter it in the list.
                    in client mode if the extension is not accepted return false
        */
        vmp_bool add_ext(vmp::vector<vmp::str> args,vmp_bool server=true);
        
        //! Add a list of extensions(see add_ext())
        /*!
            @param accepted string accepted
            @param server if server == true use in server mode,otherwise in client mode
            @return in server mode always returns true,if the extension is not accepted it does not enter it in the list.
                    in client mode if the extension is not accepted return false
        */
        vmp_bool add_exts(vmp::str accepted,vmp_bool server=true);
        
        //!Return extensions info string
        vmp::str exts_info();
        
        //!Parse packet
        /*!
            @param ibuf input buffer(if ibuf == 0 read buffer in the queue
            @param framecode in case error code return
            @param frameerr in case error string return
            @param out output data buffer
            @param maxframesize Maximum size of a frame
            @param maxframes maximum number of frames for data
            @return operation code or 0x100 in case of failure,for code 0x01 and 0x02 out buffer is setting,0x08 framecode and framerr setting
        */
        vmp_uint parse(vmp::Buf *ibuf,vmp_uint *framecode,vmp::str *frameerr,vmp::Buf *out,vmp_size maxframesize,vmp_size maxframes);
        
        //! Queue a package to send dividing in frames
        /*!
            @param buf message send
            @param opcode operation code
            @param maxframesize Maximum size of a frame
            @param maxframes maximum number of frames for data
        */
        void send_packet(vmp::Buf *buf,vmp::str opcode,vmp_size maxframesize,vmp_size maxframes);
        
        //! Create ping packet
        /*! ping packet
            
            @param bodylen payload size
        */
        void send_ping(vmp_size bodylen);
        
        //! Returns the next package in the queue to send
        /*! 
            @return buffer to send
        */
        vmp::Buf *next_send();
        
        //! Create frame close
        /*!
            @param code exit code
            @param reason exit reason string
        */
        vmp::Buf *frame_close(vmp_uint code,vmp::str reason);
        
        void free_packet(packet::websocket::WebSocket_Framing_P *p);
        
        //! free buffer
        void free_buf(vmp::Buf *buf);
};

}}}

#endif

