/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 03/09/2024
 */

#ifndef VAMPIRIA_PACKET_HTTP_PROTOCOL_H

#define VAMPIRIA_PACKET_HTTP_PROTOCOL_H 1

namespace vampiria { namespace packet { namespace http {

const vmp_uint HttpCode_Continue=100; /*!< Indicates that the initial part of a request has been received and has not yet been rejected by the server.*/
const vmp_uint HttpCode_SwitchingProtocols=101;/*!< Indicates that the server understands and is willing to comply with the client's request, via the Upgrade header field, for a change in the application protocol being used on this connection.*/
const vmp_uint HttpCode_Processing=102; /*!< Is an interim response used to inform the client that the server has accepted the complete request, but has not yet completed it.*/
const vmp_uint HttpCode_EarlyHints=103; /*!< Indicates to the client that the server is likely to send a final response with the header fields included in the informational response.*/
const vmp_uint HttpCode_OK= 200; /*!< Indicates that the request has succeeded.*/
const vmp_uint HttpCode_Created=201; /*!< Indicates that the request has been fulfilled and has resulted in one or more new resources being created.*/
const vmp_uint HttpCode_Accepted= 202;/*!< Indicates that the request has been accepted for processing, but the processing has not been completed.*/
const vmp_uint HttpCode_NonAuthoritativeInformation=203;/*!< Indicates that the request was successful but the enclosed payload has been modified from that of the origin server's 200 (OK) response by a transforming proxy.*/
const vmp_uint HttpCode_NoContent=204;/*!< Indicates that the server has successfully fulfilled the request and that there is no additional content to send in the response payload body.*/
const vmp_uint HttpCode_ResetContent=205;/*!< Indicates that the server has fulfilled the request and desires that the user agent reset the \"document view\", which caused the request to be sent, to its original state as received from the origin server.*/
const vmp_uint HttpCode_PartialContent=206;/*!< Indicates that the server is successfully fulfilling a range request for the target resource by transferring one or more parts of the selected representation that correspond to the satisfiable ranges found in the requests's Range header field.*/
const vmp_uint HttpCode_MultiStatus=207;/*!< Provides status for multiple independent operations.*/
const vmp_uint HttpCode_AlreadyReported= 208;/*!< Used inside a DAV:propstat response element to avoid enumerating the internal members of multiple bindings to the same collection repeatedly.[RFC 5842]*/
const vmp_uint HttpCode_IMUsed= 226;/*!< The server has fulfilled a GET request for the resource, and the response is a representation of the result of one or more instance-manipulations applied to the current instance.*/
const vmp_uint HttpCode_MultipleChoices=300;/*!< Indicates that the target resource has more than one representation, each with its own more specific identifier, and information about the alternatives is being provided so that the user (or user agent) can select a preferred representation by redirecting its request to one or more of those identifiers.*/
const vmp_uint HttpCode_MovedPermanently=301;/*!< Indicates that the target resource has been assigned a new permanent URI and any future references to this resource ought to use one of the enclosed URIs.*/
const vmp_uint HttpCode_Found=302;/*!< Indicates that the target resource resides temporarily under a different URI.*/
const vmp_uint HttpCode_SeeOther=303;/*!< Indicates that the server is redirecting the user agent to a different resource, as indicated by a URI in the Location header field, that is intended to provide an indirect response to the original request.*/
const vmp_uint HttpCode_NotModified=304;/*!< Indicates that a conditional GET request has been received and would have resulted in a 200 (OK) response if it were not for the fact that the condition has evaluated to false.*/
const vmp_uint HttpCode_UseProxy=305;/*!< \deprecated \parblock Due to security concerns regarding in-band configuration of a proxy.\endparblock The requested resource MUST be accessed through the proxy given by the Location field.*/
const vmp_uint HttpCode_TemporaryRedirect=307;/*!< Indicates that the target resource resides temporarily under a different URI and the user agent MUST NOT change the request method if it performs an automatic redirection to that URI.*/
const vmp_uint HttpCode_PermanentRedirect=308;/*!< The target resource has been assigned a new permanent URI and any future references to this resource ought to use one of the enclosed URIs. [...] This status code is similar to 301 Moved Permanently (Section 7.3.2 of rfc7231), except that it does not allow rewriting the request method from POST to GET.*/
const vmp_uint HttpCode_BadRequest=400;/*!< Indicates that the server cannot or will not process the request because the received syntax is invalid, nonsensical, or exceeds some limitation on what the server is willing to process.*/
const vmp_uint HttpCode_Unauthorized=401;/*!< Indicates that the request has not been applied because it lacks valid authentication credentials for the target resource. */
const vmp_uint HttpCode_PaymentRequired=402;/*!< *Reserved* */
const vmp_uint HttpCode_Forbidden=403; /*!< Indicates that the server understood the request but refuses to authorize it.*/
const vmp_uint HttpCode_NotFound=404;/*!< Indicates that the origin server did not find a current representation for the target resource or is not willing to disclose that one exists.*/
const vmp_uint HttpCode_MethodNotAllowed=405;/*!< Indicates that the method specified in the request-line is known by the origin server but not supported by the target resource.*/
const vmp_uint HttpCode_NotAcceptable=406;/*!< Indicates that the target resource does not have a current representation that would be acceptable to the user agent, according to the proactive negotiation header fields received in the request, and the server is unwilling to supply a default representation.*/
const vmp_uint HttpCode_ProxyAuthenticationRequired=407;/*!< Is similar to 401 (Unauthorized), but indicates that the client needs to authenticate itself in order to use a proxy.*/
const vmp_uint HttpCode_RequestTimeout=408;/*!< Indicates that the server did not receive a complete request message within the time that it was prepared to wait.*/
const vmp_uint HttpCode_Conflict=409;/*!< Indicates that the request could not be completed due to a conflict with the current state of the resource.*/
const vmp_uint HttpCode_Gone=410;/*!< Indicates that access to the target resource is no longer available at the origin server and that this condition is likely to be permanent.*/
const vmp_uint HttpCode_LengthRequired=411;/*!< Indicates that the server refuses to accept the request without a defined Content-Length.*/
const vmp_uint HttpCode_PreconditionFailed=412;/*!< Indicates that one or more preconditions given in the request header fields evaluated to false when tested on the server.*/
const vmp_uint HttpCode_PayloadTooLarge=413;/*!< Indicates that the server is refusing to process a request because the request payload is larger than the server is willing or able to process. */
const vmp_uint HttpCode_URITooLong=414;/*!< Indicates that the server is refusing to service the request because the request-target is longer than the server is willing to interpret.*/
const vmp_uint HttpCode_UnsupportedMediaType=415;/*!< Indicates that the origin server is refusing to service the request because the payload is in a format not supported by the target resource for this method.*/
const vmp_uint HttpCode_RangeNotSatisfiable=416;/*!< Indicates that none of the ranges in the request's Range header field overlap the current extent of the selected resource or that the set of ranges requested has been rejected due to invalid ranges or an excessive request of small or overlapping ranges.*/
const vmp_uint HttpCode_ExpectationFailed=417;/*!< Indicates that the expectation given in the request's Expect header field could not be met by at least one of the inbound servers.*/
const vmp_uint HttpCode_ImATeapot=418; /*!< Any attempt to brew coffee with a teapot should result in the error code 418 I'm a teapot.*/
const vmp_uint HttpCode_UnprocessableEntity=422;/*!< Means the server understands the content type of the request entity (hence a 415(Unsupported Media Type) status code is inappropriate), and the syntax of the request entity is correct (thus a 400 (Bad Request) status code is inappropriate) but was unable to process the contained instructions.*/
const vmp_uint HttpCode_Locked=423;/*!< Means the source or destination resource of a method is locked.*/
const vmp_uint HttpCode_FailedDependency=424; /*!< Means that the method could not be performed on the resource because the requested action depended on another action and that action failed.*/
const vmp_uint HttpCode_UpgradeRequired= 426;/*!< Indicates that the server refuses to perform the request using the current protocol but might be willing to do so after the client upgrades to a different protocol.*/
const vmp_uint HttpCode_PreconditionRequired= 428;/*!< Indicates that the origin server requires the request to be conditional.*/
const vmp_uint HttpCode_TooManyRequests=429;/*!< Indicates that the user has sent too many requests in a given amount of time (\"rate limiting\").*/
const vmp_uint HttpCode_RequestHeaderFieldsTooLarge=431;/*!< Indicates that the server is unwilling to process the request because its header fields are too large. */
const vmp_uint HttpCode_UnavailableForLegalReasons=451;/*!< This status code indicates that the server is denying access to the resource in response to a legal demand.*/
const vmp_uint HttpCode_InternalServerError=500;/*!< Indicates that the server encountered an unexpected condition that prevented it from fulfilling the request.*/
const vmp_uint HttpCode_NotImplemented=501;/*!< Indicates that the server does not support the functionality required to fulfill the request.*/
const vmp_uint HttpCode_BadGateway=502;/*!< Indicates that the server, while acting as a gateway or proxy, received an invalid response from an inbound server it accessed while attempting to fulfill the request.*/
const vmp_uint HttpCode_ServiceUnavailable=503;/*!< Indicates that the server is currently unable to handle the request due to a temporary overload or scheduled maintenance, which will likely be alleviated after some delay.*/
const vmp_uint HttpCode_GatewayTimeout=504;/*!< Indicates that the server, while acting as a gateway or proxy, did not receive a timely response from an upstream server it needed to access in order to complete the request.*/
const vmp_uint HttpCode_HTTPVersionNotSupported=505;/*!< Indicates that the server does not support, or refuses to support, the protocol version that was used in the request message.*/
const vmp_uint HttpCode_VariantAlsoNegotiates=506;/*!< Indicates that the server has an internal configuration error: the chosen variant resource is configured to engage in transparent content negotiation itself, and is therefore not a proper end point in the negotiation process.*/
const vmp_uint HttpCode_InsufficientStorage=507;/*!< Means the method could not be performed on the resource because the server is unable to store the representation needed to successfully complete the request.*/
const vmp_uint HttpCode_LoopDetected=508;/*!< Indicates that the server terminated an operation because it encountered an infinite loop while processing a request with "Depth: infinity". [RFC 5842]*/
const vmp_uint HttpCode_NotExtended=510; /*!< The policy for accessing the resource has not been met in the request. [RFC 2774]*/
const vmp_uint HttpCode_NetworkAuthenticationRequired=511;/*!< Indicates that the client needs to authenticate to gain network access.*/

//!Returns the standard http reason phrase for a http status code.
/*!
    @param code http code
    @return standard http reason phrase or "" if no standard phrase for the given code is known
*/
vmp::str HttpCode_reason(vmp_uint code);

//! Code in input is informational code?
/*!
    @param code http code
    @return true is true if the given code is an informational code,othewrwise false
*/
vmp_bool HttpCode_isInformational(vmp_uint code);

//! Code in input is successful code?
/*!
    @param code http code
    @return true is true if the given code is a successful code,othewrwise false
*/
vmp_bool HttpCode_isSuccessful(vmp_uint code);

//! Code in input is redirectional code?
/*!
    @param code http code
    @return true is true if the given code is a redirectional code,othewrwise false
*/
vmp_bool HttpCode_isRedirection(vmp_uint code);

//! Code in input is a client error code?
/*!
    @param code http code
    @return true is true if the given code is a client error code,othewrwise false
*/
vmp_bool HttpCode_isClientError(vmp_uint code);

//! Code in input is a server error code?
/*!
    @param code http code
    @return true is true if the given code is a server error code,othewrwise false
*/
vmp_bool HttpCode_isServerError(vmp_uint code);

//! Code in input is any type error code?
/*!
    @param code http code
    @return true is true if the given code is anytype error code,othewrwise false
*/
vmp_bool HttpCode_isError(vmp_uint code);

//! Encodes the url in input,enters hexadecimal values for special characters
/*!
    @param url input url
    @return encoded string
*/
vmp::str uri_encode(vmp::str url);
//! Decode the URL in input
/*!
    @param url encoded url
    @return standard string
*/

vmp::str uri_decode(vmp::str url);

//! Utility to work with http urls
class UriHelper
{
    private:
        vmp::str regex_scheme;/*!<Internal usage*/
        vmp::str regex_host;/*!<Internal usage*/
        vmp::str regex_path;/*!<Internal usage*/
        vmp::str regex_userinfo;/*!<Internal usage*/
        vmp::str regex_params;/*!<Internal usage*/
        vmp::str regex_urlipv6;/*!<Internal usage*/
    
        vmp::str type_;/*!<url type (absolute path="abs",only path="path",authority server = "authority")*/
        vmp::str scheme_;/*!<url scheme (http,https...)*/
        vmp::str user_;/*!<url user(if exist)*/
        vmp::str password_;/*!<url user(if exist)*/
        vmp::str host_;/*!<url host*/
        vmp_index port_;/*!<url host port*/
        vmp::str path_;/*!<url path*/
        vmp::PairList<vmp::str,vmp::str> query_;/*!<url parameters*/
        vmp::PairList<vmp::str,vmp::str> fragment_;/*!<url fragment values*/
        
        //! Internal usage print function
        void print_authority(vmp::str *ret);
        
        //! Internal usage print function
        void print_path(vmp::str *ret);
        
        //! Internal usage print function
        void print_params(vmp::str *ret);
        
    public:
        //! A constructor(empty url)
        UriHelper();
        
        //! A constructor
        /*!
            @param uri string url
        */
        UriHelper(vmp::str uri);
        
        //! A destructor
        ~UriHelper();
        
        //! Reset object
        void reset();
        
        //! Reads url string in the object
        /*!
            @param uri input uri string
        */
        void read(vmp::str uri);
        
        //! init absolute url type="abs" scheme://host:port/path
        /*!
            @param scheme scheme value
            @param host host value
            @param port port value
            @param path url path
        */
        void init_abs(vmp::str scheme,vmp::str host,vmp_uint port,vmp::str path);
        
        //! init autority url type="authority" host:port
        /*!
            @param host host value
            @param port port value
        */
        void init_authority(vmp::str host,vmp_uint port);
        
        //! set init path type="path" path
        /*!
            @param path value
        */
        void init_path(vmp::str path);
        
        //! set user and password in url (type="abs" or authotity)
        /*!
            @param user user value
            @param password password value
        */
        void set_userinfo(vmp::str user,vmp::str password);
        
        //! Returns url type
        /*!
            @sa type_
            @return type string
        */
        vmp::str type();
        
        //! Returns url scheme
        /*!
            @sa scheme_
            @return scheme string
        */
        vmp::str scheme();
        
        //! Returns url user
        /*!
            @sa user_
            @return user string
        */
        vmp::str user();
        
        //! Returns url password
        /*!
            @sa password_
            @return password string
        */
        vmp::str password();
        
        //! Returns url host
        /*!
            @sa host_
            @return host string
        */
        vmp::str host();
        
        //! Returns url port
        /*!
            @sa port_
            @return port string
        */
        vmp_index port();
        
        //! Returns url path
        /*!
            @sa path_
            @return path string
        */
        vmp::str path();
        
        //! Returns url query list
        /*!
            @sa query_
            @return query pairlist
        */
        vmp::PairList<vmp::str,vmp::str> *query();
        
        //! Returns fragment fragment list
        /*!
            @sa fragment_
            @return fragment pairlist
        */
        vmp::PairList<vmp::str,vmp::str> *fragment();
        
        //! Returns the url string associated with the object
        /*!
            @return url string or except in case of failure
        */
        vmp::str print();
        
        //! Search in a packet chain the http request header is set to url field
        /*!
            @param packet input packet
            @return void or except in case of failure
        */
        void set_requesturi(packet::Packet *packet);
};

//! Utility to work with http headers
class HeadersHelper
{
    private:
        vmp::str regex_name;/*!<internal usage*/
        vmp::str regex_value;/*!<internal usage*/
        vmp::PairList<vmp::str,vmp::str> headers_;/*!<headers list*/
    public:
        //! A constructor
        HeadersHelper();
        
        //! A constructor
        /*!
            @param headers headers in string mode
        */
        HeadersHelper(vmp::str headers);
        
        //! A destructor
        ~HeadersHelper();
        
        //! Returns headers list
        /*!
            @return header list
        */
        vmp::PairList<vmp::str,vmp::str> *headers();
        
        //! Reset object
        void reset();
        
        //! Reads header
        /*!
            @param headers headers string
            @return voi or except in case of failure
        */
        void read(vmp::str headers);
        
        //! Returns the headers string associated with the object
        /*!
            @return headers string or except in case of failure
        */
        vmp::str print();
        
        //! Search in a packet chain the http request header is set to header field
        /*!
            @param packet input packet
            @return void or except in case of failure
        */
        void set_requestheaders(packet::Packet *packet);
        
        //! Search in a packet chain the http response header is set to header field
        /*!
            @param packet input packet
            @return void or except in case of failure
        */
        void set_responseheaders(packet::Packet *packet);
};

}}}

#endif

