/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 24/10/2024
 */

#include "net.h"

namespace vampiria { namespace net {

UdpUI::UdpUI(event::Manager *manager):event::UI(manager)
{
    set_event(0,0,0,0);
}

UdpUI::~UdpUI()
{
}

vmp::str UdpUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp_int type=conn->evt_connection_type();
    vmp::str strtype=conn->evt_connection_strtype();
    net::Address *local=conn->evt_connection_localaddr(),*peer;
    
    if((type == net::CONN_UDP4) || (type == net::CONN_UDP6))
        vmp::unicode::str_write(&ret,"%s:[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if(type == CONN_UDPCONNECT)
    {
        peer=conn->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                           peer->host().c_str(),peer->service().c_str());
    }
    else if(type == net::CONN_UDPPROXY)
    {
        peer=conn->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[proxy]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}

void UdpUI::close_event(event::Cell *cell)
{
    net::EventConnection *conn=cell->event<net::EventConnection>();   
    conn->evt_connection_close();
}

void UdpUI::free_ref(event::Cell *cell)
{
    net::EventConnection *conn=cell->event<net::EventConnection>();
    conn->evt_connection_free();
    cref_.free(conn);
}

void UdpUI::set_event(net::EVTCBRECVFROM recv,event::EVTCB connect,event::EVTCBBUF crecv,event::EVTCB cevent,vmp_size maxpsize)
{
    recv_=recv;
    connect_=connect;
    crecv_=crecv;
    cevent_=cevent;
    maxpsize_=maxpsize;
}

event::Cell *UdpUI::new_bind(net::Address *local)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_udpbind(this,local,maxpsize_,recv_,cevent_);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("net::UdpUI::new_bind() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *UdpUI::new_udp4()
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_udp4(this,maxpsize_,recv_,cevent_);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("net::UdpUI::new_udp4() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *UdpUI::new_udp6()
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_udp6(this,maxpsize_,recv_,cevent_);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("net::UdpUI::new_udp6() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *UdpUI::new_client(net::Address *peer,net::ProxyChain *proxy)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_udp(this,peer,maxpsize_,connect_,crecv_,cevent_,proxy);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("net::UdpUI::new_client() '%s'",x.what());
    }
    manager_->unlock();
    conn->evt_connection_udp_connect(cell);
    return cell;
}

void UdpUI::sendTo(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    conn->evt_connection_sendTo(buf,peer);
    manager->unlock();
}

void UdpUI::send(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    conn->evt_connection_send(buf);
    manager->unlock();
}

vmp_size UdpUI::maxpsize(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp_size ret=conn->evt_connection_udpmaxsize();
    manager->unlock();
    return ret;   
}

net::Address *UdpUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::Address *ret=conn->evt_connection_localaddr();
    manager->unlock();
    return ret;   
}

net::Address *UdpUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::Address *ret=conn->evt_connection_peeraddr();
    manager->unlock();
    return ret;   
}

net::ProxyChain *UdpUI::proxychain(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::ProxyChain *ret=conn->evt_connection_proxychain();
    manager->unlock();
    return ret;
}

vmp_int UdpUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp_int ret=conn->evt_connection_type();
    manager->unlock();
    return ret;
}

vmp::str UdpUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp::str ret=conn->evt_connection_strtype();
    manager->unlock();
    return ret;
}

vmp_byte UdpUI::lastfrag(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp_byte ret=conn->evt_connection_udplastfrag();
    manager->unlock();
    return ret;
}

}}


