/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 31/10/2024
*/
 
#include "net.h"

namespace vampiria { namespace net {

vmp_int proxy_udp_send(net::EventConnection *udp,vmp::Buf *buf,net::Address *msgaddr,net::Address *peer,vmp::str *msg)
{
    vmp::Buf out,send;
    vmp_size maxsize=udp->udpmaxsize_-30;
    buf->index();
    if(buf->size() == 0)
    {
        (*msg)="send empty package ";
        return 0;
    }
    if(buf->size() <= maxsize)
    {
        net::socks5_udp(&send,0,msgaddr,buf);
        return net::socket_sendTo(udp->fd_,&send,peer);
    }
    else if(buf->size() > (maxsize*127))
    {    
        (*msg)="send package too large";
        return 0;
    }
    else
    {
        vmp_byte frag=0x00;
        while(buf->eob())
        {
            out.reset();
            send.reset();
            if(buf->size_reading() <= maxsize)
            {    
                buf->read_buf(&out,buf->size_reading());
                frag=0x80;    
            }
            else
            {      
                buf->read_buf(&out,maxsize);
                frag++;
            }        
            net::socks5_udp(&send,frag,msgaddr,&out);
            if(net::socket_sendTo(udp->fd_,&send,peer) == -1)
                return -1;
         }
    }
    return buf->size();
}

vmp_int proxychain_connect(vmp::str type,net::Address *addr)
{
    vmp_int iptype;
    vmp_int socket;
    try
    {
        iptype=addr->iptype();     
    }
    catch(vmp::exception &x)
    {
        iptype=PF_UNSPEC;
    }
    if((iptype == PF_UNSPEC) || (addr->service() == "0"))
        vmp::except("Proxy(%s,%s:%s) bad address",type.c_str(),addr->host().c_str(),addr->service().c_str());
    socket=net::socket_stream(iptype,false);
    if((socket == -1) || (net::socket_connect(socket,addr) == -1))
        vmp::except("Proxy(%s,%s:%s) %s",type.c_str(),addr->host().c_str(),addr->service().c_str(),vmp::value_errno().c_str());
    return socket;    
}

void proxychain_error(event::Cell *cell,vmp::str level,vmp_int retcode,vmp::str msg,vmp_uint nchain)
{
    event::Manager *manager=cell->get_manager();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    net::Proxy *proxy=evt->proxy_->get(nchain);
    vmp::str err;
    net::Address *addr=proxy->address();
    vmp::unicode::str_write(&err,"Proxy(%s,%s:%s) %s",proxy->type().c_str(),addr->host().c_str(),addr->service().c_str(),msg.c_str());
    manager->cell_close_err_spec(cell,level,retcode,err);
}

void proxychain_tcp_connect(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp::str err;
    manager->lock();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    try
    {
        vmp_int ret=net::socket_connect_check(evt->fd_);
        if(ret == 0)
        {
            if(net::socket_addrlocal(evt->fd_,&(evt->local_)) == -1)
                net::proxychain_error(cell,"net::EventConnection",vmp::get_errno(),vmp::value_errno());
            else
            {
                cell->read_=net::proxychain_tcp_recv;
                cell->writing_=false;   
                manager->cell_update(cell);
                evt->proxystatus_="init";
                evt->proxywait_=cell->timewait_;
                manager->cell_timewait(cell,evt->proxy_->get_timeout());
                evt->proxy_->next(evt);
            }
        }
        else if(ret == -1)
            net::proxychain_error(cell,"net::EventConnection",vmp::get_errno(),vmp::value_errno());
    }
    catch(vmp::exception &x)
    {
         net::proxychain_error(cell,"net::EventConnection",0,x.what());
    }
    manager->unlock();
}

void proxychain_tcp_recv(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    vmp::Buf buf;
    manager->lock();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    try
    {
        vmp_int ret=net::socket_recv(evt->fd_,&buf);
        if(ret == 0)
            net::proxychain_error(cell,"net::EventConnection",0,"connection close");
        else if(ret > 0)
        {
            while(1)
            {    
                exec=evt->proxy_->next(evt,&buf);
                if(exec)
                {
                    cell->read_=net::connection_tcp_recv;
                    manager->cell_timewait(cell,evt->proxywait_);
                    exec=manager->cell_update(cell);
                    break;
                }
                if(buf.eob())
                    break;
                buf.cut(buf.get_index(),buf.size_reading());
            }
        }    
        else if(ret == -1)
            net::proxychain_error(cell,"net::EventConnection",vmp::get_errno(),vmp::value_errno());
            
    }
    catch(vmp::exception &x)
    {
        net::proxychain_error(cell,"net::EventConnection",0,x.what());
    }
    manager->unlock();
    if(exec)
    {
        evt->connect_(cell);
        if(!buf.eob())
        {
             buf.cut(buf.get_index(),buf.size_reading());     
             evt->recv_(cell,&buf);
        }
    }
    buf.reset();
}

void proxychain_udp_recv(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    vmp_bool exec=false;
    manager->lock();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    net::EventConnection *master=evt->master_->event<net::EventConnection>();
    if(buf->size() == 0)
        manager->cell_close_err_spec(master->cell_,"net::EventConnection",0,"recv empty package");
    else if(buf->size() > evt->udpmaxsize_)
        manager->cell_close_err_spec(master->cell_,"net::EventConnection",0,"recv package too large");
    else if((evt->peer_.ip() == peer->ip()) && (evt->peer_.service() == peer->service()))
        exec=true;
    manager->unlock();
    if(exec)
    {
        net::Address address;
        vmp::Buf data;
        buf->index();
        net::socks5_udp_get(buf,&(master->udplastfrag_),&address,&data);
        data.index();
        master->recv_(master->cell_,&data);
    }
}

void proxychain_tcp_recv_empty(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp::Buf buf;
    manager->lock();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    try
    {
        vmp_int ret=net::socket_recv(evt->fd_,&buf);
        if(ret == 0)
            manager->cell_close_ok_spec(cell,"net::EventConnection");
        else if(ret > 0)
            net::proxychain_error(cell,"net::EventConnection",0,"Udp associate bad message");
        else if(ret == -1)
            net::proxychain_error(cell,"net::EventConnection",vmp::get_errno(),vmp::value_errno());
    }
    catch(vmp::exception &x)
    {
        net::proxychain_error(cell,"net::EventConnection",0,x.what());
    }
    manager->unlock();
    buf.reset();
}

void proxychain_udp_send(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    vmp::Buf out,send;
    vmp::str msg;
    net::EventConnection *udp=manager->cell_searchsub(cell,"udpassociate")->event<net::EventConnection>();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    vmp_int ret=net::proxy_udp_send(udp,buf,&evt->peer_,&udp->peer_,&msg);
    if(ret == 0)
        net::proxychain_error(cell,"net::EventConnection",0,msg);
    else if (ret == -1)
        net::proxychain_error(cell,"net::EventConnection",vmp::get_errno(),vmp::value_errno());
}

void proxychain_close(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    net::EventConnection *evt=cell->event<net::EventConnection>();
    manager->lock();
    if(evt->proxy_->logger_ != 0)
    {
        if(cell->ret_ == event::SUCCESS)
            evt->proxy_->logger_->write(vmp::utils::LOG_INFO,"%s(cellid=%u) close ok",cell->closelevel_.c_str(),cell->id());   
        else if(cell->ret_ == event::TIMEOUT)
            evt->proxy_->logger_->write(vmp::utils::LOG_ERR,"%s(cellid=%u) Timeout",cell->closelevel_.c_str(),cell->id());    
        else
            evt->proxy_->logger_->write(vmp::utils::LOG_ERR,"%s(cellid=%u) error(%d,'%s')",cell->closelevel_.c_str(),cell->id(),cell->errcode_,cell->err_.c_str());
    }
    manager->unlock();
    evt->proxyclose_(cell);  
}

}}

