/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 05/11/2024
 */

#ifndef VAMPIRIA_NET_CONNECTION_EVENTSOCKS_H

#define VAMPIRIA_NET_CONNECTION_EVENTSOCKS_H 1

namespace vampiria { namespace net {

//!Common class for Socks connection data
class SocksCommon:public event::UI
{
    public:
        //!A Constructor
        SocksCommon(event::Manager *manager);
        
        //! A Destructor
        ~SocksCommon();
        
        vmp::utils::Storage<net::EventConnection> cref_;/*!<Storage event*/
        
        //! Virtual function
        void close_event(event::Cell *cell);
        
        //! Virtual function
        void free_ref(event::Cell *cell);
        
        net::EVTCBACCEPT accept_;/*!<function performed when connection is accepted*/
        event::EVTCB lclose_;/*!<function performed when close listen socket*/
        event::EVTCB close_;/*!<function performed when close server connection*/
        
        vmp::time::Time ctimeout_;/*!<Connection timeout(default 5.0)*/
        
        vmp::vector<vmp_byte> auths_;/*!<Accepted authentication methods*/
        net::SOCKSCBNOAUTH noauth_;/*!<Function performed in the initial phase for unauthenticated proxy connections*/
        net::SOCKSCBUSERPASS userpass_;/*!<Function performed in the initial phase for user and password authentication */
        
        net::SOCKSCBREQUEST cmdconnect_;/*!<function performed when it receives a command request*/
        event::EVTCBBUF     crecv_;/*!<function performed when a request connect recv package*/
        
        net::SOCKSCBREQUEST cmdbind_;/*!<function performed when it receives a bind request*/
        event::EVTCBBUF     brecv_;/*!<function performed when a request bind recv package*/
        
        vmp_size udpmaxsize_;/*!<UDP max packet size*/
        net::SOCKSCBREQUEST cmdudpassociate_;/*!<function performed when it receives a udpassociate request*/
        net::EVTCBRECVFROM udprecv_;/*!<function performed when a request udpassociate recv package*/
        
        //! Set tcp connection event(if value == 0 not change callback)
        /*!
            @param accept function performed when connection is accepted
            @param lclose function performed when close listen socket
            @param close function performed when close server connection
        */
        void set_tcp_event(net::EVTCBACCEPT accept,event::EVTCB lclose,event::EVTCB close);
        
        //! Set timeout value(if value == 0 not change value)
        /*!
            @param ctimeout connection timeout
        */
        void set_timeout(vmp::time::Time ctimeout);
        
        //! Set authentication methods(if value == 0 is disabled method)
        /*!
            @param noauth call callback at connection time for anonymous user verification
            @param userpass call callback at connection time to authenticate users with user and password
        */
        void set_auths(net::SOCKSCBNOAUTH noauth,net::SOCKSCBUSERPASS userpass);

        //! Set the connenction event for command connect(if value == 0 not change callback)
        /*!
            @param cmdconnect function performed when it receives a command request
            @param crecv function performed when a request connect recv package
        */
        void set_connect_event(net::SOCKSCBREQUEST cmdconnect,event::EVTCBBUF crecv);
        
        //! Set the connenction event for command connect(if value == 0 not change callback)
        /*!
            @param cmdbind function performed when it receives a cmd bind request
            @param brecv function performed when a request bind recv package
        */
        void set_bind_event(net::SOCKSCBREQUEST cmdbind,event::EVTCBBUF brecv);
        
        //! Set the connenction eventand data for command udpassociate(if value == 0 not change)
        /*!
            @param cmdudpassociate callback called when it receives a udpassociate request
            @param udprecv function performed when it receives a udpassociate request
            @param udpmaxsize UDP max packet size
        */
        void set_udpassociate_event(net::SOCKSCBREQUEST cmdudpassociate,net::EVTCBRECVFROM udprecv,vmp_size udpmaxsize);
};

const vmp_byte evtsocks_status_init=0x00;/*!<Connection status EventSocks init.*/
const vmp_byte evtsocks_status_request=0x01;/*!<Connection status EventSocks request.*/
const vmp_byte evtsocks_status_bind=0x02;/*!<Connection status EventSocks bind,for status listen in bind connection.*/
const vmp_byte evtsocks_status_established=0x03;/*!<Connection status EventSocks established.*/
const vmp_byte evtsocks_status_close=0x04;/*!<Connection status EventSocks close connection.*/

//!Event socket event
class EventSocks:public net::EventConnection
{
    private:
        //! Reset Socks event
        void evt_socks_reset();
    public:
        //! A Constructor
        EventSocks();
        
        //! A Destructor
        ~EventSocks();
        
        net::SocksCommon *common_;/*!<Common class for Socks connection data*/
        vmp::time::Time ctime_;/*!<Internal usage for connection timeout*/
        vmp_byte cstatus_;/*!<Connection status*/
        vmp::str userid_;/*!<connection identifier*/
        vmp_uint permits_;/*!<connection permits*/
        vmp_byte command_;/*!<command request*/
        net::Address address_;/*!<address in request*/
        
        //!Create a new socks listen server
        /*!
            @param  ui user interface (see Class UI)
            @param local address to listen server connection
            @param backlog the maximum length to which the  queue  of pending  connections
            @param common common class
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_socks_listen(event::UI *ui,net::Address *local,vmp_uint backlog,net::SocksCommon *common);
        
        //! Sends a reply error and closes the connection
        /*!
            @param result result code socks5 error
            @param msg message string(if == 0 use net::msg_result(cd))
        */
        void evt_socks_reply_error(vmp_byte result,vmp::str msg="");
        
        //! Call when a connect request goes well
        void evt_socks_reply_connect();
        
        //! Call when a bind request is listening to
        /*!
            @param listen listen address bind request
        */
        void evt_socks_reply_bind_listen(net::Address *listen);
        
        //! Call when a bind request receives a connection
        /*!
            @param connect connect address bind request
        */
        void evt_socks_reply_bind_connect(net::Address *connect);
        
        //! Call when a udp associate request goes well
        void evt_socks_reply_udpassociate();
        
        //! Returns Connection status
        /*!
            @sa cstatus_
            @return connection status
        */
        vmp_byte evt_socks_cstatus();
        
        //! Returns connection identifier
        /*!
            @sa userid_
            @return connection identifier or "" if authentication anonymous
        */
        vmp::str evt_socks_userid();
        
        //! Returns connection permits
        /*!
            @sa permits_
            @return connection permits 
        */
        vmp_uint evt_socks_permits();
        
        //! Returns the command received from the socket session
        /*!
            @sa command_
            @return command socks or 0x00 in case of failure
        */
        vmp_byte evt_socks_command();
        
        //! Returns the command received from the socket session in string mode
        /*!
            @sa command_
            @return string command socks
        */
        vmp::str evt_socks_command_str();
        
        //! Returns the address in request
        /*!
            @sa address_
            @return address in request
        */
        net::Address *evt_socks_address();
        
        //! Returns the bind address of a UDP connection(request udp associate)
        /*!
            @return bind address UDP connection or except in case of failure
        */
        net::Address *evt_socks_udpbind();
        
        //! Close event
        void evt_socks_close();
       
        //! Free event
        void evt_socks_free();
};

}}

#endif

