/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 05/04/2024
 */

#include "modules.h"

namespace vampiria { namespace modules {

ModDescTag::ModDescTag():xml::Tag()
{
    set_priority(20);
}
        
ModDescTag::~ModDescTag()
{
}

        
void ModDescTag::build()
{
    xml::Tag *tag=get_parent();
    vmp::str pname=tag->get_name();
    if(pname == "module")
    {
         modules::ModRootTag *mtag=(ModRootTag *) tag;
         mtag->description_=get_text();
    }
    else if(pname == "input")
    {
         modules::ModInputTag *itag=(ModInputTag *) tag;
         itag->description_=get_text();
    }
    else if(pname == "opt_p")
    {
         modules::ModParTag *ptag=(ModParTag *) tag;
         ptag->description_=get_text();
    }
}

ModParTag::ModParTag():xml::Tag()
{
    set_priority(15);
    description_="";
}
        
ModParTag::~ModParTag()
{
    description_="";
}
 
void ModParTag::build()
{
    modules::ModInputTag *ptag=(modules::ModInputTag *) get_parent();
    build_impl();
    ptag->par_.push_back(this);
}

void ModParTag::check(vmp::str par)
{
    check_impl(par);
}

ModParGeneric::ModParGeneric():ModParTag()
{
}
       
ModParGeneric::~ModParGeneric()
{
}

void ModParGeneric::build_impl()
{
    description_=get_text();
}
       
void ModParGeneric::check_impl(vmp::str par)
{
}

ModOptDataTag::ModOptDataTag():xml::Tag()
{
    set_priority(20);
}
     
ModOptDataTag::~ModOptDataTag()
{
}

void ModOptDataTag::build()
{
    modules::ModParOpt *ptag=(modules::ModParOpt *)get_parent();
    ptag->opt_.push_back(get_text());
}

ModParOpt::ModParOpt():ModParTag()
{
}
       
ModParOpt::~ModParOpt()
{
}

void ModParOpt::build_impl()
{
    for(vmp_index i=0;i<opt_.size();i++)
    {
        if(i==0)
            vmp::unicode::str_cwrite(&description_,"[Accepted value:(%s",opt_[i].c_str());
        else
            vmp::unicode::str_cwrite(&description_,"|%s",opt_[i].c_str());
    }
    vmp::unicode::str_cwrite(&description_,")]");
}
       
void ModParOpt::check_impl(vmp::str par)
{
    for(vmp_index i=0;i<opt_.size();i++)
        if(par == opt_[i])
           return;
    vmp::str error;
    for(vmp_index i=0;i<opt_.size();i++)
    {
        if(i==0)
            vmp::unicode::str_write(&error,"Accepted value:(%s",opt_[i].c_str());
        else
            vmp::unicode::str_cwrite(&error,"|%s",opt_[i].c_str());
    }
    vmp::unicode::str_cwrite(&error,")");
    vmp::except_s(error);
}

ModParInteger::ModParInteger():ModParTag()
{
    min_=vmp::INTMIN;
    max_=vmp::INTMAX;
}
       
ModParInteger::~ModParInteger()
{
}

void ModParInteger::build_impl()
{
    vmp::str smin,smax;
    smin=get_attr("min");
    smax=get_attr("max");
    if(smin != "")
    {
        try
        {
            min_=vmp::unicode::str_todigit(smin);
        }
        catch(vmp::exception &x)
        {
            parser_error("Invalid attribute min='%s' must be an integer",smin.c_str());
        }
    }
    if(smax != "")
    {
        try
        {
            max_=vmp::unicode::str_todigit(smax);
        }
        catch(vmp::exception &x)
        {
            parser_error("Invalid attribute max='%s' must be an integer",smax.c_str());
        }
    }
    if(min_ > max_)
        parser_error("Invalid attributes min='%d' and max='%d' min must be <= max",min_,max_);
    vmp::unicode::str_write(&description_,"%s (integer between '%d' and '%d')",get_text().c_str(),min_,max_);
}
       
void ModParInteger::check_impl(vmp::str par)
{
    try
    {
        vmp::unicode::str_todigit_range(par,min_,max_);
    }
    catch(vmp::exception &x)
    {
        vmp::except("Accepted Integer between '%d' and '%d'",min_,max_);
    }
}


ModParReal::ModParReal():ModParTag()
{
    min_=(vmp_real)vmp::INTMIN;
    max_=(vmp_real)vmp::INTMAX;
}
       
ModParReal::~ModParReal()
{
}

void ModParReal::build_impl()
{
    vmp::str smin,smax;
    smin=get_attr("min");
    smax=get_attr("max");
    if(smin != "")
    {
        try
        {
            min_=vmp::unicode::str_toreal_range(smin,(vmp_real)vmp::INTMIN,(vmp_real)vmp::INTMAX);
        }
        catch(vmp::exception &x)
        {
            parser_error("Invalid attribute min='%s' must be an real between '%f' and '%f'",smin.c_str(),(vmp_real)vmp::INTMIN,(vmp_real)vmp::INTMAX);
        }
    }
    if(smax != "")
    {
        try
        {
            max_=vmp::unicode::str_toreal_range(smax,(vmp_real)vmp::INTMIN,(vmp_real)vmp::INTMAX);
        }
        catch(vmp::exception &x)
        {
            parser_error("Invalid attribute max='%s' must be an real between '%f' and '%f'",smin.c_str(),(vmp_real)vmp::INTMIN,(vmp_real)vmp::INTMAX);
        }
    }
    if(min_ > max_)
        parser_error("Invalid attributes min='%f' max='%f' min must be <= max",min_,max_);
    vmp::unicode::str_write(&description_,"%s (double between '%f' and '%f')",get_text().c_str(),min_,max_);   
}
       
void ModParReal::check_impl(vmp::str par)
{
    try
    {
        vmp::unicode::str_toreal_range(par,min_,max_);
    }
    catch(vmp::exception &x)
    {
        vmp::except("Accepted Real between '%f' and '%f'",min_,max_);
    }
}

ModParHex::ModParHex():ModParTag()
{
}
       
ModParHex::~ModParHex()
{
}

void ModParHex::build_impl()
{
    vmp::unicode::str_write(&description_,"%s(hexadecimal value)",get_text().c_str());
}
       
void ModParHex::check_impl(vmp::str par)
{
    if(!vmp::unicode::str_istype(par,"xdigit"))
        vmp::except("Accepted Hexadecimal");
}

ModParIp::ModParIp():ModParTag()
{
}
       
ModParIp::~ModParIp()
{
}

void ModParIp::build_impl()
{
    vmp::str name=get_name();
    if(name == "ip4_p")
        vmp::unicode::str_write(&description_,"%s(ipv4 value)",get_text().c_str());
    else if(name == "ip6_p")
        vmp::unicode::str_write(&description_,"%s(ipv6 value)",get_text().c_str());
    else
        vmp::unicode::str_write(&description_,"%s(ip value)",get_text().c_str());
}
       
void ModParIp::check_impl(vmp::str par)
{
    vmp::str name=get_name();
    if(name == "ip4_p")
    {
        if(!net::is_ipv4_raw(par))
           vmp::except("Accepted ipv4 address");
    }
    else if(name == "ip6_p")
    {    
        if(!net::is_ipv6_raw(par))
           vmp::except("Accepted ipv6 address");
    }
    else
    {
        if(!net::is_ip_raw(par))
           vmp::except("Accepted ip address");    
    }
}

ModParMac::ModParMac():ModParTag()
{
}
       
ModParMac::~ModParMac()
{
}

void ModParMac::build_impl()
{
    vmp::unicode::str_write(&description_,"%s(mac address value)",get_text().c_str());
}
       
void ModParMac::check_impl(vmp::str par)
{
    if(!net::is_macaddress_raw(par))
       vmp::except("Accepted mac address");
}

ModVarTag::ModVarTag():xml::Tag()
{
    set_priority(10);
}
        
ModVarTag::~ModVarTag()
{
}

void ModVarTag::build()
{
    modules::ModInputTag *ptag=(modules::ModInputTag *) get_parent();
    ptag->var_=true;
    ptag->vardesc_=get_text();
}

ModInputTag::ModInputTag():xml::Tag()
{
    set_priority(5);
    name_="";
    required_="";
    minpar_=0;
    description_="";
    var_=false;
    vardesc_="";
}
        
ModInputTag::~ModInputTag()
{
    name_="";
    required_="";
    description_="";
    vardesc_="";
}

void ModInputTag::destroy_impl()
{
    par_.clear();
}

void ModInputTag::build()
{
    name_=get_attr("name");
    required_=get_attr("required");
    vmp::str minpar;
    try
    {
        minpar=get_attr("minpar");
        if(minpar == "")
            minpar_=par_.size();
        else
            minpar_=vmp::unicode::str_todigit_range(minpar,0,par_.size());
    }
    catch(vmp::exception &x)
    {
        vmp::except("Invalid attribute 'minpar=%s' must be an integer between 0 and the number of par",minpar.c_str());
    }
    modules::ModRootTag *ptag=(ModRootTag *) get_parent();
    for(vmp_index i=0;i<ptag->input_.size();i++)
        if(name_ == ptag->input_[i]->name_)
            parser_error("Duplicate input name '%s' prev line %u",name_.c_str(),ptag->input_[i]->get_line());
    ptag->input_.push_back(this);
}

vmp::str ModInputTag::required_format()
{
    if(required_ == "o")
       return "one";
    else if (required_ == "op")
       return "one or plus";
    else if (required_ == "zo")
       return "zero or one";
    else if (required_ == "zp")
       return "zero or one or plus";
    return "";
}

ModRootTag::ModRootTag()
{
    description_="";
    run_=true;
}
        
ModRootTag::~ModRootTag()
{
}
 
void ModRootTag::destroy_impl()
{
    input_.clear();
}
       
void ModRootTag::build()
{
    vmp::str run=get_attr("run");
    if(run == "no")
        run_=false;
}

ModInfo::ModInfo(vmp::str manifest)
{
    parser_.add_tag<modules::ModRootTag>("module");
    parser_.add_tag<modules::ModDescTag>("description");
    parser_.add_tag<modules::ModInputTag>("input");
    parser_.add_tag<modules::ModParGeneric>("generic_p");
    parser_.add_tag<modules::ModParOpt>("opt_p");
    parser_.add_tag<modules::ModOptDataTag>("optdata_p");
    parser_.add_tag<modules::ModParInteger>("integer_p");
    parser_.add_tag<modules::ModParReal>("real_p");
    parser_.add_tag<modules::ModParHex>("hex_p");
    parser_.add_tag<modules::ModParIp>("ip_p");
    parser_.add_tag<modules::ModParIp>("ip4_p");
    parser_.add_tag<modules::ModParIp>("ip6_p");
    parser_.add_tag<modules::ModParMac>("mac_p");
    parser_.add_tag<modules::ModVarTag>("var");
    parser_.build(manifest,vmp::env::frm_dtd(modules::dtdname_));
}
      
ModInfo::~ModInfo()
{
    parser_.destroy();
}

modules::ModRootTag *ModInfo::get_root()
{
    return (modules::ModRootTag *) parser_.get_root();
}

vmp_bool ModInfo::executable()
{
    modules::ModRootTag *root=get_root();
    return root->run_;
}

}}

