/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 05/12/2024
*/

#ifndef VAMPIRIA_JSON_JAPI_H

#define VAMPIRIA_JSON_JAPI_H 1

namespace vampiria { namespace json {

//! Returns the full name of the data type json(example json::'jtype')
/*!
    @param jtype short name data type json
    @return full name data type json 
*/
vmp::str japi(vmp::str jtype);

//! Returns json type list [subtype] or []
/*!
    @param jtype json type in list("" list not typed)
    @return list type
*/
vmp::str japi_type_list(vmp::str jtype="");

//! Returns json type teble {subtype} or {}
/*!
    @param jtype json type in table("" table not typed)
    @return table type
*/
vmp::str japi_type_table(vmp::str jtype="");

//! Create a json data json::japi("void")
/*!
    <pre>
    { 
       jtype: json::japi("void"),
       jbody: {}
    }
    </pre>
    @param obj object where the json data is inserted
    @return void or except in case of failure
*/
void japi_void(json::JsonObj *obj);

//! Create a json data json::japi("[datatype]") or json::japi("[]") for generic list
/*!
    <pre>
    {
        jtype: json::japi("[datatype]"),
        jbody:
        {
            datatype:'datatype',
            list: 'list'(empty in this case)
        }
    }
    </pre>
    @param obj object where the json data is inserted
    @param datatype jdata types in the list(read data with json::jdata.get_text("datatype"),read data list with json::jdata.get_list("list"))
    @return void or except in case of failure
*/
void japi_list(json::JsonObj *obj,vmp::str datatype="");

//! Create a json data ['datatype'] or "[]" for generic list
/*!
    <pre>
    {
        jtype: [datatype],
        jbody:
        {
            datatype:'datatype',
            list: 'list'
        }
    }
    </pre>
    @param obj object where the json data is inserted
    @param list jdata types in the list(read data list with json::jdata.get_list("list"),read datatype with json::jdata.get_text("datatype"))
    @return void or except in case of failure
*/
void japi_list(json::JsonObj *obj,json::JList *list);

//! Create a json data {'datatype'} or "{}" for generic table
/*!
    <pre>
    {
        jtype: {'datatype'}, 
        jbody:
        {
            datatype:'datatype',
            table:'table'(empty in this case)
        }
    }
    </pre>
    @param obj object where the json data is inserted
    @param datatype jdata types in the table(read data with json::jdata.get_text("datatype"),read data table with json::jdata.get_table("table"))
    @return void or except in case of failure
*/
void japi_table(json::JsonObj *obj,vmp::str datatype="");

//! Create a json data json::japi("{datatype}") or json::japi("{}") for generic table
/*!
    <pre>
    {
        jtype: json::japi({'datatype'}), 
        jbody:
        {
            datatype:'datatype',
            table:'table'
        }
    }
    </pre>
    @param obj object where the json data is inserted
    @param table jdata types in the table(read data table with json::jdata.get_table("table"),read datatype with json::jdata.get_text("datatype"))
    @return void or except in case of failure
*/
void japi_table(json::JsonObj *obj,json::JTable *table);

//! Create a json data json::japi("oper") used for request operations between threads
/*!
    <pre>
    { 
       jtype: json::japi("oper"),
       jbody: 
       {
           id:'id',
           oper:'oper',
           bin:'binary data',
           input:{json data object}
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param id identity comunication (read data json::jdata.get_text("id"))
    @param input json data input of the operation (read data json::jdata.get_subtype("input"))
    @param bin binary data(read jdata.get_text("bin"))
    @return void or except in case of failure
*/
void japi_oper(json::JsonObj *obj,vmp::str id,json::JsonObj *input,vmp::Buf *bin=0);

//! Create a json data json::japi("operkill") used request to terminate an operation
/*!
    <pre>
    { 
       jtype: json::japi("operkill"),
       jbody: 
       {
           id:'id',
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param id identity comunication(equal to that of the request)(read data json::jdata.get_text("id"))
    @return void or except in case of failurevoid japi_login(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_bool md5_password)
*/
void japi_oper_kill(json::JsonObj *obj,vmp::str id);

//! Create a json data json::japi("operdata") used for response an operations request
/*!
    <pre>
    { 
       jtype: json::japi("operdata"),
       jbody: 
       {
           id:'id',
           bin:'binary data',
           data:{json data object}
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param id identity comunication(equal to that of the request)(read data json::jdata.get_text("id"))
    @param data json data output response(read data json::jdata.get_subtype("data"))
    @param bin binary data(read jdata.get_text("bin"))
    @return void or except in case of failure
*/
void japi_oper_data(json::JsonObj *obj,vmp::str id,json::JsonObj *data,vmp::Buf *bin=0);

//! Create a json data json::japi("operpush") used to send data related to the request from the person who created the request
/*!
    <pre>
    { 
       jtype: json::japi("operpush"),
       jbody: 
       {
           id:'id',
           bin:'binary data'
           data:{json data object}
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param id identity comunication(equal to that of the request)(read data json::jdata.get_text("id"))
    @param data json data pushed(read data json::jdata.get_subtype("data"))
    @param bin binary data(read jdata.get_text("bin"))
    @return void or except in case of failure
*/
void japi_oper_push(json::JsonObj *obj,vmp::str id,json::JsonObj *data,vmp::Buf *bin=0);

//! Create a json data json::japi("operclose") used to close an operations request
/*!
    <pre>
    { 
       jtype: json::japi("operclose"),
       jbody: 
       {
           id:'id',
           code:'code',
           msg:'msg'
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param id identity comunication(equal to that of the request)(read data json::jdata.get_text("id"))
    @param code operation exit code (read data json::jdata.get_integer("code"))
    @param msg  operation exit message (read data json::jdata.get_text("msg"))
    @return void or except in case of failure
*/
void japi_oper_close(json::JsonObj *obj,vmp::str id,vmp_int code,vmp::str msg);

//! Create a json data json::japi("status") used to request information about your status at a process
/*!
    <pre>
    { 
       jtype: json::japi("status"),
       jbody: 
       {
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @return void or except in case of failure
*/
void japi_status(json::JsonObj *obj);

//! Create a json data json::japi("statusupdate") used to inform that a status variable has changed in the process
/*!
    <pre>
    { 
       jtype: json::japi("statusupdate"),
       jbody: 
       {
           key:'key',
           data:{json data object}
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param key key of the updated object (read data json::jdata.get_text("key"))
    @param data value of the updated object (read data json::jdata.get_subtype("data"))
    @return void or except in case of failure
*/
void japi_status_update(json::JsonObj *obj,vmp::str key,json::JsonObj *data);

//! Create a json data json::japi("statuscancel") used to inform that a status variable has been deleted
/*!
    <pre>
    { 
       jtype: json::japi("statuscancel"),
       jbody: 
       {
           key:'key',
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param key key of the deleted object (read data json::jdata.get_text("key"))
    @return void or except in case of failure
*/
void japi_status_cancel(json::JsonObj *obj,vmp::str key);

//! Create a json data json::japi("login") used for authentication token
/*!
    <pre>
    { 
       jtype: json::japi("login"),
       jbody: 
       {
           login:'login',
           password:'password',
           md5_password:bool
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user name(read data json::jdata.get_text("user"))
    @param password password(read data json::jdata.get_text("password"))
    @param md5_password if true the input password parameter is encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_login(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_bool md5_password=false);

//! Create an authentication token,a jdata json::japi("login") encoded in base64
/*!
    @param user user name
    @param password password
    @param md5_password  if true the input password parameter is encrypted with the md5 function,default false
    @return string token or except in case of failure
*/
vmp::str japi_login_b64_e(vmp::str user,vmp::str password,vmp_bool md5_password=false);

//! Decode a authentication token and Read obj a jdata json::japi("login")
/*!
    @param json parser where the json data is inserted
    @param etoken input b64 token
    @return void or except in case of failure
*/
void japi_login_b64_d(json::Json *json,vmp::str etoken);

//! Create a json data json::japi("passwd") used for change user password
/*!
    <pre>
    { 
       jtype: json::japi("passwd"),
       jbody: 
       {
           oldpassword:'oldpassword',
           password:'password',
           md5_password:bool
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param oldpassword old password(read data json::jdata.get_text("oldpassword"))
    @param password new password(read data json::jdata.get_text("password"))
    @param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_passwd(json::JsonObj *obj,vmp::str oldpassword,vmp::str password,vmp_bool md5_password=false);

//! Create a json data json::japi("userlist") used for request list users
/*!
    <pre>
    { 
       jtype: json::japi("userlist"),
       jbody: 
       {
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @return void or except in case of failure
*/
void japi_userlist(json::JsonObj *obj);

//! Create a json data json::japi("userinfo") used for send userinfo data
/*!
    <pre>
    { 
       jtype: json::japi("userinfo"),
       jbody: 
       {
           user:'user',
           permits:permits,
           only_localhost:only_localhost
           
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param permits permits user (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
    @return void or except in case of failure
*/
void japi_userinfo(json::JsonObj *obj,vmp::str user,vmp_uint permits,vmp_bool only_localhost);

//! Create a json data json::japi("useradd") used for add a new user
/*!
    <pre>
    { 
       jtype: json::japi("useradd"),
       jbody: 
       {
           user:'user',
           password:'password',
           permits:permits,
           only_localhost:only_localhost
           
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param password password (read data json::jdata.get_text("password"))
    @param permits permits user (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
    @param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_useradd(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_uint permits,vmp_bool only_localhost,vmp_bool md5_password=false);

//! Create a json data json::japi("usermod") used for Change user values
/*!
    <pre>
    { 
       jtype: json::japi("usermod"),
       jbody: 
       {
           user:'user',
           permits:permits,
           only_localhost:only_localhost
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param permits permits user (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
    @return void or except in case of failure
*/
void japi_usermod(json::JsonObj *obj,vmp::str user,vmp_uint permits,vmp_bool only_localhost);

//! Create a json data json::japi("userpasswd") used for Change user password
/*!
    <pre>
    { 
       jtype: json::japi("userpasswd"),
       jbody: 
       {
           user:'user',
           password:'password',
           md5_password:md5_password
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param password new password(read data json::jdata.get_text("password"))
    @param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_userpasswd(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_bool md5_password=false);

//! Create a json data json::japi("userdel") used for delete user
/*!
    <pre>
    { 
       jtype: json::japi("userdel"),
       jbody: 
       {
           user:'user'
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @return void or except in case of failure
*/
void japi_userdel(json::JsonObj *obj,vmp::str user);

//! Create a json data json::japi("peerlist") used for request list peers
/*!
    <pre>
    { 
       jtype: json::japi("peerlist"),
       jbody: 
       {
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @return void or except in case of failure
*/
void japi_peerlist(json::JsonObj *obj);

//! Create a json data json::japi("peerinfo") used for send userinfo data
/*!
    <pre>
    { 
       jtype: json::japi("peerinfo"),
       jbody: 
       {
           fingerprint:'fingerprint',
           subjetc:'subject',
           permits:permits
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param fingerprint fingerprint peer (read data json::jdata.get_text("fingerprint"))
    @param subject subject peer (read data json::jdata.get_text("subject"))
    @param permits permits peer (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @return void or except in case of failure
*/
void japi_peerinfo(json::JsonObj *obj,vmp::str fingerprint,vmp::str subject,vmp_uint permits);

//! Create a json data json::japi("peeradd") used to add a peer node in the access table
/*!
    <pre>
    { 
       jtype: json::japi("peeradd"),
       jbody: 
       {
           fingerprint:'fingerprint',
           subjetc:'subject',
           permits:permits
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param fingerprint fingerprint peer (read data json::jdata.get_text("fingerprint"))
    @param subject subject peer (read data json::jdata.get_text("subject"))
    @param permits permits peer (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @return void or except in case of failure
*/
void japi_peeradd(json::JsonObj *obj,vmp::str fingerprint,vmp::str subject,vmp_uint permits);

//! Create a json data json::japi("peermod") used to modify a peer node in the access table
/*!
    <pre>
    { 
       jtype: json::japi("peermod"),
       jbody: 
       {
           fingerprint:'fingerprint',
           permits:permits
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param fingerprint fingerprint peer (read data json::jdata.get_text("fingerprint"))
    @param permits permits peer (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @return void or except in case of failure
*/
void japi_peermod(json::JsonObj *obj,vmp::str fingerprint,vmp_uint permits);

//! Create a json data json::japi("peerdel") used to delete a peer node in the access table
/*!
    <pre>
    { 
       jtype: json::japi("peerdel"),
       jbody: 
       {
           fingerprint:'fingerprint',
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param fingerprint fingerprint peer (read data json::jdata.get_text("fingerprint"))
    @return void or except in case of failure
*/
void japi_peerdel(json::JsonObj *obj,vmp::str fingerprint);

}}

#endif

