/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/11/2024
*/

#include "jrp.h"

namespace vampiria { namespace jrp {

void killwaitcb(void *ref)
{
    jrp::JrpReq *req=(jrp::JrpReq *) ref;
    event::Cell *cell=req->cell();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    req->kill_end();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    jrp::EventJrp *jrp=(jrp::EventJrp *)ssl->sub_;
    jrp->evtsub_jrp_close(jrp::status_kill_notcomplete);
    manager->unlock();
}

JrpReq::JrpReq()
{
    count_=0;
    cell_=0;
    kill_=0;
    common_=0;
    reset();
}

JrpReq::~JrpReq()
{
    reset();
}

jrp::JrpCommon *JrpReq::common()
{
    return common_;
}

event::Manager *JrpReq::get_manager()
{
    if(cell_ == 0)
        vmp::except_s("jrp::JrpReq::get_manager() event not active");
    return cell_->get_manager();
}

void JrpReq::setevent(jrp::JrpCommon *common,event::Cell *cell,vmp_int rid,json::JsonObj *jdata,vmp::str type)
{
    try
    {
        common_=common;
        cell_=cell;
        event::Manager *manager=cell_->get_manager();
        manager->cell_alloc(cell_);
        rid_=rid;
        type_=type;
        jdata_.json_new_obj(jdata);
        jrp::EventJrp *jrp=(jrp::EventJrp *)cell_->event<crypto::EventSsl>()->sub_;
        vmp::unicode::str_write(&key_,"%s/%s/%u",jrp->sessiontype_.c_str(),jrp->sessionid_.c_str(),rid_);
    }
    catch(vmp::exception &x)
    {
        reset();
        vmp::except_s(x.what());
    }
    count_=1;
}

void JrpReq::kill_end()
{
    if(kill_ != 0)
    {
        event::Manager *manager=kill_->get_manager();
        manager->cell_close(kill_,event::SUCCESS);
        manager->cell_release(kill_);
        kill_=0;
    }
}

void JrpReq::reset()
{
    rid_=0;
    key_="";
    jdata_.json_new();
    type_="";
    kill_end();
    if(cell_ != 0)
    {
        event::Manager *manager=cell_->get_manager();
        manager->cell_release(cell_);
        cell_=0;
    }
    isclosed_=false;
    status_=jrp::status_ok;
    msg_="";
}

vmp::str JrpReq::type()
{
    return type_;
}

event::Cell *JrpReq::cell()
{
    return cell_;
}

vmp_index JrpReq::rid()
{
    return rid_;
}

vmp::str JrpReq::key()
{
    return key_;
}

json::Json *JrpReq::jdata_json()
{
    return &jdata_;
}

json::JsonObj *JrpReq::jdata_root()
{
    return jdata_.root();
}

vmp::str JrpReq::jdata_type()
{
    json::JData jdata;
    jdata.set(jdata_.root());
    return jdata.jtype();
}

vmp_int JrpReq::status()
{
    if(isclosed_)
        return status_;
    return 1;
}

vmp::str JrpReq::msg()
{
    if(isclosed_)
        return msg_;
    return "open";
}

vmp_index JrpReq::alloc_internal()
{
    if(cell_ == 0)
        vmp::except_s("jrp::JrpReq::alloc_internal() the request is free");
    return ++count_;
}

vmp_index JrpReq::alloc()
{
    vmp_index ret;
    event::Manager *manager=get_manager();
    manager->lock();
    try
    {
        ret=alloc_internal();
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->unlock();
    return ret;
}

vmp_index JrpReq::release_internal()
{
    if(count_ == 0)
        vmp::except_s("jrp::JrpReq::release_internal() the request is free");
    if(count_ == 1)
    {    
        if(isclosed_)
        {
            reset();
            count_=0;
            common_->free_request(this);
            common_=0;
        }
        else
            vmp::except_s("jrp::JrpReq::release_internal() the request cannot be released because it is not closed");
    }
    else
        --count_;
    return count_;
}

vmp_index JrpReq::release()
{
    vmp_index ret;
    event::Manager *manager=get_manager();
    manager->lock();
    try
    {
        ret=release_internal();
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->unlock();
    return ret;
}

void JrpReq::push(json::JsonObj *outputs,vmp::Buf *payload)
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "request")
        {
            manager->unlock();
            vmp::except("jrp::JrpReq::push() type '%s' operation denied",type_.c_str());
        }
        jrp::push(&buf,rid_,outputs,payload);
        crypto::EventSsl *ssl=cell_->event<crypto::EventSsl>();
        ssl->evt_ssl_send(&buf);
    }
    manager->unlock();
}

void JrpReq::response(json::JsonObj *outputs,vmp::Buf *payload)
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "managed")
        {
            manager->unlock();
            vmp::except("jrp::JrpReq::response() type '%s' operation denied",type_.c_str());
        }
        jrp::response(&buf,rid_,outputs,payload);
        crypto::EventSsl *ssl=cell_->event<crypto::EventSsl>();
        ssl->evt_ssl_send(&buf);
    }
    manager->unlock();
}

void JrpReq::kill()
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "request")
        {
            manager->unlock();
            vmp::except("jrp::JrpReq::kill() type '%s' operation denied",type_.c_str());
        }
        jrp::kill(&buf,rid_);
        crypto::EventSsl *ssl=cell_->event<crypto::EventSsl>();
        ssl->evt_ssl_send(&buf);
        kill_=ssl->common_->routine(jrp::killwaitcb,0,(void *) this,key_,common_->killwait_);
        manager->cell_alloc(kill_);
    }
    manager->unlock();
}

void JrpReq::close(vmp_int status,vmp::str msg)
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "managed")
        {
            manager->unlock();
            vmp::except("jrp::JrpReq::close() type '%s' operation denied",type_.c_str());
        }
        isclosed_=true;
        status_=status;
        msg_=msg;
        jrp::close(&buf,rid_,status_,msg_);
        crypto::EventSsl *ssl=cell_->event<crypto::EventSsl>();
        ssl->evt_ssl_send(&buf);
        kill_end();
        release_internal();
        jrp::EventJrp *jrp=(jrp::EventJrp *)ssl->sub_;
        jrp::JrpReq *tmp;
        jrp->srvreq_.cancel(rid_,&tmp);
        release_internal();
    }
    manager->unlock();
}

void JrpReq::request_oper(json::JsonObj *oper,vmp::Buf *bin)
{
    json::japi_oper(oper,key_,jdata_.root(),bin);
}

void JrpReq::push_oper(json::JsonObj *oper,json::JsonObj *jdata,vmp::Buf *bin)
{
    json::japi_oper(oper,key_,jdata,bin);
}

void JrpReq::kill_oper(json::JsonObj *oper)
{
    japi_oper_kill(oper,key_);
}

void JrpReq::send_from_oper(json::JsonObj *oper)
{
    json::JData jdata;
    jdata.set(oper);
    vmp::Buf payload;
    vmp::str otype=jdata.jtype();
    if(otype == json::japi("operdata"))
    {
        json::JsonObj outputs;
        jdata.get_subtype("data",&outputs);
        payload.write_str(jdata.get_text("bin"));
        payload.index();
        response(&outputs,&payload);
    }
    else if (otype == json::japi("operclose"))
    {
        vmp_int status=jdata.get_integer("code");
        vmp::str msg=jdata.get_text("msg");
        close(status,msg);
    }
    else
        vmp::except("JrpReq::send_from_oper(oper.jtype == '%s') bad jtype",otype.c_str());
}

void JrpReq::recv_close(vmp_int status,vmp::str msg)
{
    if(!isclosed_)
    {
        if(type_ == "")
            vmp::except("jrp::JrpReq::recv_close() type='%s' operation denied",type_.c_str());
        isclosed_=true;
        status_=status;
        if(msg == "")
            msg_=jrp::msg_status(status);
        else
            msg_=msg;
        kill_end();
        release_internal();
    }
}

}}

