/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 24/10/2024
 */

#include "event.h"

namespace vampiria { namespace event {

Manager::Manager()
{
    cellids_=1;
    timer_.init();
    self_=vmp::thread::pthread_self_wrap();
    init_=false;
    breakui_=new event::BreakloopUI(this);
    init();
}

Manager::~Manager()
{
    delete breakui_;
    breakui_=0;
}

void Manager::cell_freedata(event::Cell *cell)
{
    event::Cell *master=cell_master(cell);
    vmp::str sid=cell_subid(cell);
    cell_delsub(master,sid);
}

void Manager::init()
{
    lock();
    if(!init_)
    {
        timeout_=0.0;
        loop_=true;
        init_=true;
        breakui_->init();
        breakui_->cell_->priority_=event::BREAKLOOP_PRIORITY;
    }
    unlock();
}

void Manager::lock()
{
    mutex_.lock();
}

void Manager::unlock()
{
    mutex_.unlock();
}

vmp_bool Manager::forced_unlock()
{
    return mutex_.forced_unlock();
}

void Manager::recovery_lock(vmp_bool status)
{
    mutex_.recovery_lock(status);
}

void Manager::breakloop()
{
    if(self_ != vmp::thread::pthread_self_wrap())
       breakui_->active();       
}

event::Cell *Manager::cell_new(event::UI *ui,event::Event *event,vmp_int priority,event::EVTCB read,event::EVTCB close)
{
    if(!init_)
        vmp::except_s("event::Manager::cell_new() event Manager not initialized");
    event::Cell *cell=cellref_.get();
    if(cell->id_ == 0)
      cell->id_=cellids_++;
    cell->reset();
    cell->ui_=ui;
    cell->event_=event;
    event->cell_=cell;
    cell->manager_=this;
    cell_set_priority(cell,priority);
    cell->writing_=false;
    if(read == 0)
      cell->read_=event::empty_ev;
    else
      cell->read_=read;
    if(close == 0)
      cell->close_=event::empty_ev;
    else
      cell->close_=close;
    cell_alloc(cell);
    pcell_.push(cell);
    breakloop();
    return cell;  
}

vmp_bool Manager::cell_update(event::Cell *cell)
{
    if(cell == 0)
        return false;
    if(cell->state_ == event::CLOSE)
        return false;
    if(cell->state_ == event::CLOSEWAIT)
    {
        if(cell->nwait_ == 0)
            return false;
        cell->nwait_--;
    }
    if(cell->timewait_ != 0.0)
        cell->timeout_=time_now()+cell->timewait_;
    return true;void set_priority(vmp_int priority);
}

void Manager::cell_timewait(event::Cell *cell,vmp::time::Time timewait)
{
    if(cell != 0)
    {
        if(timewait > 0.0)
        {
            cell->timewait_=timewait;
            cell->timeout_=time_now()+cell->timewait_;
        }
        else
        {
            cell->timewait_=0.0;
            cell->timeout_=0.0;
        }
        breakloop();
    }
}

vmp::time::Time Manager::cell_get_timewait(event::Cell *cell)
{
    return cell->timewait_;
}

vmp::time::Time Manager::cell_get_timeout(event::Cell *cell)
{
    return cell->timeout_;
}

void Manager::cell_close(event::Cell *cell,event::CELLRET retcode,vmp::str errinfo)
{
    if((cell != 0) && (cell->state_ != event::CLOSE))
    {
        cell->ret_=retcode;
        cell->err_=errinfo;
        cell->state_=event::CLOSE;
        cell->ui_->close_event(cell);
        breakloop();
    }
}

void Manager::cell_close_ok_spec(event::Cell *cell,vmp::str closelevel)
{
    if((cell != 0) && (cell->state_ != event::CLOSE))
    {
        cell->closelevel_=closelevel;
        cell->ret_=event::SUCCESS;
        cell->state_=event::CLOSE;
        cell->ui_->close_event(cell);
        breakloop();
    }
}

void Manager::cell_close_err_spec(event::Cell *cell,vmp::str closelevel,vmp_int errcode,vmp::str errinfo)
{
    if((cell != 0) && (cell->state_ != event::CLOSE))
    {
        cell->closelevel_=closelevel;
        cell->errcode_=errcode;
        cell->ret_=event::ERROR;
        cell->err_=errinfo;
        cell->state_=event::CLOSE;
        cell->ui_->close_event(cell);
        breakloop();
    }
}

void Manager::cell_closewait(event::Cell *cell,vmp_uint n,vmp::str closelevel)
{
   if((cell != 0) && (cell->state_ == event::ACTIVE))
   {
       if(n == 0)
           cell_close_ok_spec(cell,closelevel);
       else
       {
           cell->state_=event::CLOSEWAIT;
           cell->closelevel_=closelevel;
           cell->nwait_=n;
       }
   }
}

vmp::time::Time Manager::time_now()
{
    return timer_.now();
}

void Manager::set_timeout(vmp::time::Time timeout)
{void set_priority(vmp_int priority);
    if(timeout > 0.0)
    {
       if((timeout_ == 0.0) || (timeout < timeout_))
          timeout_=timeout;  
    }
}

void Manager::cell_alloc(event::Cell *cell)
{
    if(cell != 0)
        cell->refalloc_++;
}

void Manager::cell_release(event::Cell *cell)
{
    if(cell != 0)
    {
        if(cell->refalloc_ > 1)
            cell->refalloc_--;
        else if (cell->refalloc_ == 1)
        {
            cell->refalloc_=0;
            cell->ui_->free_ref(cell);
            cellref_.free(cell);
        }
        else
            vmp::warning_s("event::Manager::cell_release() cell already released");
    }
}

void Manager::cell_set_priority(event::Cell *cell,vmp_uint priority)
{
    if(priority < event::MAX_CELL_PRIORITY)
       cell->priority_=priority;
    else
       cell->priority_=event::MAX_CELL_PRIORITY;
}

vmp_uint Manager::cell_get_priority(event::Cell *cell)
{
    return cell->priority_;
}

event::Cell *Manager::cell_master(event::Cell *cell)
{
    if(cell != 0)
        return cell->event_->master_;
    return 0;
}

vmp::str Manager::cell_subid(event::Cell *cell)
{
    if(cell != 0)
        return cell->event_->sid_;
    return "";
}

void Manager::cell_addsub(event::Cell *cell,vmp::str sid,event::Cell *sub)
{
    if((cell != 0) && (sub != 0))
    {
        cell->event_->evtsubs_.insert(sid,sub);
        sub->event_->sid_=sid;
        sub->event_->master_=cell;
        cell_alloc(cell);
        cell_alloc(sub);
    }
}

event::Cell *Manager::cell_searchsub(event::Cell *cell,vmp::str sid)
{
    event::Cell *ret=0;
    if(cell != 0)
        cell->event_->evtsubs_.search(sid,&ret);
    return ret;
}

void Manager::cell_delsub(event::Cell *cell,vmp::str sid)
{
    if(cell != 0)
    {
        event::Cell *ret=0;
        try
        {
            cell->event_->evtsubs_.cancel(sid,&ret);
            ret->event_->master_=0;
            ret->event_->sid_="";
            cell_release(ret);
            cell_release(cell);
        }
        catch(vmp::exception &x)
        {
        }
    }
}

void Manager::stop()
{
    loop_=false;
    breakui_->active();
}

void Manager::loop()
{
    if(!init_)
        vmp::except_s("event::Manager::loop() event Manager not initialized");
    vmp::Select_FD select;
    event::Cell *cell;
    vmp::vector<event::Cell *> vread;
    vmp::vector<event::Cell *> vclose;
    vmp::time::Time now,stimeout;
    vmp_size s;
    while(1)
    {
        lock();
        select.reset();
        vread.clear();
        vclose.clear();
        while(!pcell_.empty())
        {
            cell=pcell_.top();
            pcell_.pop();
            if(((cell->state_ == event::CLOSEWAIT) && (cell->nwait_ == 0)) || (!loop_))
                cell_close(cell,event::SUCCESS);   
            if(cell->state_ == event::CLOSE)
                vclose.push_back(cell);
            else
            {
                vread.push_back(cell);
                select.fd_add(cell->event_->fd_);
                if(cell->writing_)
                    select.fd_add_w(cell->event_->fd_);
            }
        }
        s=vclose.size();
        unlock();
        for(vmp_index i=0;i<s;i++)
        {
            cell=vclose[i];
            try
            {
                cell->close_(cell);
            }
            catch(vmp::exception &x)
            {
                forced_unlock();
                vmp::error("event::Manager::loop() close(%s) (%s)",cell->identity().c_str(),x.what());
            }
            lock();
            cell_freedata(cell);
            cell_release(cell);
            unlock();
        }
        lock();
        for(vmp_index i=0;i<vread.size();i++)
            set_timeout(vread[i]->timeout_);
        now=timer_.now();
        if(timeout_ > 0.0)
        {
           if(timeout_ <= now)
             stimeout=-1.0;
           else
             stimeout=timeout_-now;
        }
        else
           stimeout=0.0;
        timeout_=0.0;
        if((s > 0) || stimeout == -1.0)
        {
            stimeout=0.0;
            breakui_->active();
        }
        unlock();
        if(vread.size() > 0)
        {    
            if(loop_)
                select.fd_monitor(stimeout);
        }
        else
            break;
        lock();
        now=timer_.now();
        unlock();
        for(vmp_index i=0;i<vread.size();i++)
        {    
            cell=vread[i];
            lock();
            vmp_bool rbool=cell->event_->active_;
            unlock();
            if(rbool)
            {
                try
                {
                   cell->read_(cell);
                }
                catch(vmp::exception &x)
                {
                   forced_unlock();
                   cell->close_err(x.what());
                }
            }
            lock();
            if((cell->timeout_) != 0.0 && (cell->timeout_<=now))
                cell_close(cell,event::TIMEOUT);
            pcell_.push(cell);
            unlock();
        }
    }
    init_=false;
    unlock();
}

}}

