/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 14/01/2023
 */

#include "event.h"

namespace vampiria { namespace event {

ThreadEv::ThreadEv()
{
}
        
ThreadEv::~ThreadEv()
{
}

void ThreadEv::cancel_impl()
{
    cancel_(cell_);
}

void ThreadEv::run()
{
    event::Manager *manager=cell_->get_manager();
    try
    {
        run_(cell_);
    }
    catch(vmp::exception &x)
    {
        manager->lock();
        event_->evt_pipe_closew();
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->lock();
    event_->evt_pipe_closew();
    manager->unlock();
}

EventThread::EventThread():event::EventPipe()
{
    thread_.event_=(event::EventPipe *)this;
}

EventThread::~EventThread()
{
    thread_.event_=0; 
}

void thread_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    EventThread *event=cell->event<EventThread>();
    manager->lock();
    vmp_byte byte;
    vmp_int ret;
    try
    {
        if((ret=vmp::fd_read_bytes(event->fd_,&byte,1)) == 0)
        {
            if(event->cancel_)
            {
                event->cancel_=false;
                event->thread_.stop();
                manager->cell_close(cell,event::SUCCESS);
            }
        }
        else if(ret == -1)
        {
            if(!((errno == EAGAIN) || (errno == EWOULDBLOCK))) 
            {    
                vmp::str error;
                vmp::unicode::str_write(&error,"event::threadbuf_ev() %s",vmp::value_errno().c_str());
                manager->cell_close(cell,event::ERROR,error);
            }
        }
        else
            exec=manager->cell_update(cell);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->active_(cell);
}

void thread_cancel_default(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::thread::pthread_cancel_wrap(cell->event<event::EventThread>()->thread_.pth());
    manager->unlock();
}

void thread_close(event::Cell *cell)
{
    event::EventThread *th=cell->event<event::EventThread>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    if(th->cancel_)
    {
        th->cancel_=false;
        manager->unlock();
        th->thread_.cancel();
        th->thread_.stop();
    }
    else
        manager->unlock();
    th->close_(cell);    
}

event::Cell *EventThread::evt_thread_new(event::UI *ui,event::EVTCB active,event::EVTCB run,event::EVTCB cancel,event::EVTCB close)
{
    cancel_=false;
    event::Cell *cell;
    try
    {
        cell=evt_pipe_new(ui,thread_ev,thread_close);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s(x.what());
    }
    if(active == 0)
        active_=event::empty_ev;
    else
        active_=active;
    if(run == 0)
        thread_.run_=event::empty_ev;
    else
        thread_.run_=run;
    if(cancel == 0)
        thread_.cancel_=event::thread_cancel_default;
    else
        thread_.cancel_=cancel;
    if(close == 0)
        close_=event::empty_ev;
    else
        close_=close;
    thread_.cell_=cell;
    event::Manager *manager=cell->get_manager();
    manager->cell_alloc(cell);
    return cell;
}

void EventThread::evt_thread_active()
{
    try
    {
        vmp_byte b=0x00;
        vmp::fd_write_bytes(wfd_,&b,1);
    }
    catch(vmp::exception &x)
    {
    }
}

void EventThread::evt_thread_start()
{
    try
    {
        cancel_=true;
        thread_.start();
    }
    catch(vmp::exception &x)
    {
        cancel_=false;
        cell_->get_manager()->cell_close(cell_,event::ERROR,x.what());
    }
}

void EventThread::evt_thread_close()
{
    evt_pipe_close();
}

void EventThread::evt_thread_free()
{
    event::Manager *manager=cell_->get_manager();
    thread_.cell_=0;
    manager->cell_release(cell_);
    evt_pipe_free();
}

EventThreadBuf::EventThreadBuf():event::EventPipe()
{
    thread_.event_=(event::EventPipe *)this;
}

EventThreadBuf::~EventThreadBuf()
{
    thread_.event_=0;
}

void threadbuf_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    vmp::Buf *buf=0;
    event::Manager *manager=cell->get_manager();
    EventThreadBuf *event=cell->event<EventThreadBuf>();
    manager->lock();
    vmp_byte byte;
    vmp_int ret;
    try
    {
        if((ret=vmp::fd_read_bytes(event->fd_,&byte,1)) == 0)
        {
            if(event->cancel_)
            {
                event->cancel_=false;
                event->thread_.stop();
                manager->cell_close(cell,event::SUCCESS);
            }
        }
        else if(ret == -1)
        {
            if(!((errno == EAGAIN) || (errno == EWOULDBLOCK))) 
            {    
                vmp::str error;
                vmp::unicode::str_write(&error,"event::threadbuf_ev() %s",vmp::value_errno().c_str());
                manager->cell_close(cell,event::ERROR,error);
            }
        }
        else if(event->queue_.size() > 0)
        {
            exec=manager->cell_update(cell);
            buf=event->queue_.front();
            event->queue_.pop();
        }
        else
            manager->cell_close(cell,event::ERROR,"event::threadbuf_ev() empty queue message");
        
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->recv_(cell,buf);
    if(buf != 0)
    {
        buf->reset();
        manager->lock();
        event->bufs_.free(buf);
        manager->unlock();
    }
}

void threadbuf_cancel_default(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::thread::pthread_cancel_wrap(cell->event<event::EventThreadBuf>()->thread_.pth());
    manager->unlock();
}

void threadbuf_close(event::Cell *cell)
{
    event::EventThreadBuf *th=cell->event<event::EventThreadBuf>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    if(th->cancel_)
    {
        th->cancel_=false;
        manager->unlock();
        th->thread_.cancel();
        th->thread_.stop();
    }
    else
        manager->unlock();
    th->close_(cell);  
}

event::Cell *EventThreadBuf::evt_threadbuf_new(event::UI *ui,event::EVTCBBUF recv,event::EVTCB run,event::EVTCB cancel,event::EVTCB close)
{
    cancel_=false;
    event::Cell *cell;
    try
    {
        cell=evt_pipe_new(ui,threadbuf_ev,threadbuf_close);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s(x.what());
    }
    if(recv == 0)
        recv_=event::empty_buf_ev;
    else
        recv_=recv;
    if(run == 0)
        thread_.run_=event::empty_ev;
    else
        thread_.run_=run;
    if(cancel == 0)
        thread_.cancel_=event::threadbuf_cancel_default;
    else
        thread_.cancel_=cancel;
    if(close == 0)
        close_=event::empty_ev;
    else
        close_=close;
    thread_.cell_=cell;
    event::Manager *manager=cell->get_manager();
    manager->cell_alloc(cell);
    return cell;
}

void EventThreadBuf::evt_threadbuf_send(vmp::Buf *buf)
{
    try
    {
        vmp_byte b=0x00;
        vmp::fd_write_bytes(wfd_,&b,1);
        queue_.push(buf);
    }
    catch(vmp::exception &x)
    {
        buf->reset();
        bufs_.free(buf);
    }
}

void EventThreadBuf::evt_threadbuf_start()
{
    try
    {
        cancel_=true;
        thread_.start();
    }
    catch(vmp::exception &x)
    {
        cancel_=false;
        cell_->get_manager()->cell_close(cell_,event::ERROR,x.what());
    }
}

void EventThreadBuf::evt_threadbuf_close()
{
    evt_pipe_close();
}

void EventThreadBuf::evt_threadbuf_free()
{
    vmp::Buf *buf;
    while(queue_.size() != 0)
    {
        buf=queue_.front();
        queue_.pop();
        buf->reset();
        bufs_.free(buf);
    }
    event::Manager *manager=cell_->get_manager();
    thread_.cell_=0;
    manager->cell_release(cell_);
    evt_pipe_free();
}

}}

