/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/10/2024
 */
 
#ifndef VAMPIRIA_CRYPTO_EVENTWSS_H

#define VAMPIRIA_CRYPTO_EVENTWSS_H 1

namespace vampiria { namespace crypto {

typedef void (*WSSAUTHCB)(event::Cell *cell,json::JData *auth);/*!< Websocket authentication callback*/

//!Class shared by websocket events used for connection table and callbacks to execute.
class WssCommon:public crypto::SslCommon
{
    public:
        //! A constructor
        /*!
            @param manager event timer manager
        */
        WssCommon(event::Manager *manager);
        
        //! A destructor
        ~WssCommon();
        
        vmp::Table<vmp::str,event::Cell *> usertable_;/*!<Connection table*/
        packet::PacketHelper helper_;/*!< Packet Helper internal usage */
        
        vmp::Table<vmp::str,vmp::str> mimes_;/*!<Internal usage*/
        vmp::vector<vmp::str> http_versions;/*!< Internal usage*/
        vmp::vector<vmp_uint> wss_versions;/*!<Internal usage*/
        
        vmp::str servername_;/*!<Server name*/
        vmp_size maxsizeuri_;/*!<Max size request uri*/
        vmp_size maxsizeheader_;/*!<Max size http header*/
        vmp_size maxsizepayload_;/*!<Max size request payload*/
        
        crypto::Ctx_Peer_Web *ctx_;/*!<Context local peer associated*/
        vmp::str webroot_;/*!<root directory of the web program*/
        
        net::EVTCBACCEPT wss_tcpaccept_;/*!<Tcp server accept callback.Called when a connection tcp is accepted*/
        crypto::WSSAUTHCB  wss_auth_;/*!<Function performed to receive authentication*/
        event::EVTCB wss_session_;/*!<called when session init connection*/
        
        //! init object
        /*!
            @param ctx ssl context peer web associated associated with connections  
            @param webroot root directory of the web program
            @param ctimeout connection timeout(setting if value >=0)
        */
        void set_wss_init(crypto::Ctx_Peer_Web *ctx,vmp::str webroot,vmp::time::Time ctimeout);
        
        //! Set the parameters of the http connection
        /*!
            @param servername http server name(if value == "" not setting)
            @param maxsizeuri max size request uri(if value == 0 not setting)
            @param maxsizeheader max size http header(if value == 0 not setting)
            @param maxsizepayload Max size request payload(if value == 0 not setting)
        */
        void set_wss_http(vmp::str servername,vmp_size maxsizeuri,vmp_size maxsizeheader,vmp_size maxsizepayload);
        
        //! Set wss event (if input value = 0 doesn't know anything)
        /*!
            @param tcpaccept called when tcp server accept connection
            @param session called when session init in server connection
            @param lclose called when the listen close connection
            @param sclose called when the server close connection
        */
        void set_wss_event(net::EVTCBACCEPT tcpaccept,event::EVTCB session,event::EVTCB lclose,event::EVTCB sclose);
    
};

//!A webscoket event ssl
class EventWss:public crypto::EventSsl
{
    protected:
        //! Reset websocket event
        void evt_wss_reset();
    public:
        //! A Constructor
        EventWss();

        //! A Destructor
        ~EventWss();
        
        vmp_bool isopensession_;/*!<Internal usage.Session is open?*/
        
        vmp::vector<vmp::str> hosts_;/*!<Hosts list accepted*/
        vmp::str hostsp_;/*!<Host port event*/
        vmp_bool localhost_;/*!<If true the connections are local,otherwise are remote*/
        vmp::vector<vmp::str> origins_;/*!<Web socket origins list only host*/
        
        vmp::Buf httprequest_;/*!<http request received to start the session. If the session is transformed into websocket it is not initialized*/
        vmp::Buf httpresponse_;/*!<http response send*/
        vmp_uint httpcode_;/*!<Http request code*/
        vmp::str httpversion_;/*!<Http request header version*/
        vmp::str httpmethod_;/*!<Http request header method*/ 
        vmp::str path_;/*!<Http request uri path*/
        vmp::str httpbody_;  /*!<Http request header body*/
        vmp::str host_;/*!<Web header host*/
        vmp::str hostp_;/*!<Web header host port*/
        vmp::str subprotocols_;/*!<Web header subprotocols*/
        vmp::str wsorigin_;/*!<Web socket header origin*/
        vmp::str wskey_;/*!<Web socket header key*/
        vmp::str wsupgrade_;/*!<Web socket header upgrade*/
        vmp::str wsconnection_;/*!<Web socket header connection*/
        vmp_uint wsversion_;/*!<Web socket header version*/
        vmp::str cookies_;/*!<Http cookie*/
        
        //! Internal usage
        vmp::str wss_utils_body(vmp::str body);
        
        //! Internal usage
        void wss_utils_web(packet::Packet **ret);
        
        //! Internal usage
        packet::Packet *wss_utils_response_text(vmp_uint code,vmp::str body="");
        
        //! Internal usage
        void wss_utils_response_except(packet::Packet **ret,vmp_uint code,vmp::str body="");
        
        //! Internal usage
        void wss_utils_badversion(packet::Packet **ret);
        
        //! Internal usage
        void wss_utils_upgrade_except(packet::Packet **ret);
        
        //!Function called when authentication was performed and continue handshake(monitor mode)(internal usage)
        /*!
            @param user user login
            @param permits permissions assigned to the user.permits == 0 indicates access denied and check the subprotocols headers(if failed response BadRequest),
            if the user is already logged in with another client HttpCode_Forbidden 
        */
        void wss_utils_continue(vmp::str user,vmp_uint permits);
        
        //!Create a new event wss server(host valid ip address)
        /*!
            @param ui user interface (see Class UI)
            @param local address to listen server connection
            @param common shared p2p class
            @param backlog the maximum length to which the  queue  of pending  connections
            @param hosts hosts accepted list
            @param origins Alternative sources (e.g. 'localhost') the port set is local
            @param localhost If true the connections to this interface are local,otherwise are remote
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_wss_listen(event::UI *ui,net::Address *local,crypto::WssCommon *common,vmp_uint backlog,vmp::vector<vmp::str> hosts,vmp::vector<vmp::str> origins,vmp_bool localhost);
        
        //! Close event
        void evt_wss_close();
       
        //! Free event
        void evt_wss_free();
};

}}

#endif

