/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 17/10/2024
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

EventSslSub::EventSslSub()
{
    cell_=0;
}

EventSslSub::~EventSslSub()
{
}

void EventSslSub::evtsub_ssl_reset()
{
    sessiontype_="";
    sessionid_="";
    sessionpermits_=0;
    recv_=event::empty_buf_ev;
    close_=event::empty_ev;
    if(cell_ != 0)
    {
       event::Manager *manager=cell_->get_manager();
       manager->cell_release(cell_);
       cell_=0;
    }
}

void EventSslSub::evtsub_ssl_recv(vmp::Buf *buf)
{
    recv_(cell_,buf);
}

void EventSslSub::evtsub_ssl_close()
{
    evtsub_ssl_close_impl();
}

void EventSslSub::evtsub_ssl_free()
{
    evtsub_ssl_free_impl();
    evtsub_ssl_reset();
}
void timercb_empty(void *ref)
{
}

void routine_event(event::Cell *cell)
{
    crypto::SslTimerRef *routine=cell->event<crypto::SslTimerRef>();
    routine->event_(routine->ref_);
}

void routine_event_close(event::Cell *cell)
{
    crypto::SslTimerRef *routine=cell->event<crypto::SslTimerRef>();
    routine->close_(routine->ref_);
}

SslTimerRef::SslTimerRef():event::EventTimer()
{
    reset();   
}

SslTimerRef::~SslTimerRef()
{
}

void SslTimerRef::reset()
{
    event_=crypto::timercb_empty;
    close_=crypto::timercb_empty;
    ref_=0;
    identity_="";
}

SslCommon::SslCommon(event::Manager *manager):event::UI(manager)
{
    vmp::except_check_pointer((void *)manager,"crypto::SslCommon(manager=null)");
    manager_=manager;
    ctimeout_=3.0;
    ckeyupdate_=0;
    skeyupdate_=0;
    tcpconnect_=event::empty_ev;
    tcpaccept_=net::empty_accept_ev;
    sslconnect_=event::empty_ev;
    sslaccept_=event::empty_ev;
    crecv_=event::empty_buf_ev;
    srecv_=event::empty_buf_ev;
    lclose_=event::empty_ev;
    cclose_=event::empty_ev;
    sclose_=event::empty_ev;
    
    //framing section
    framing_maxsize_=1048576;
    framing_maxn_=17;
    framing_rtimeout_=60.0;
    framing_ptimeout_=5.0;
    framing_pinglenbody_=120;
    framing_recv_text_=crypto::framing_recv_default;
    framing_recv_bin_=crypto::framing_recv_default;
}

SslCommon::~SslCommon()
{
    ctx_.reset();
}

void SslCommon::set_ssl_init(crypto::Ctx *ctx,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)ctx,"crypto::SslCommon(ctx=null)");
    vmp::except_check_pointer((void *)ctx->ctx_,"crypto::SslCommon(ctx.ctx_=null)");
    ctx_.copy(ctx);
    if(ctimeout > 0)
        ctimeout_=ctimeout;
}

void SslCommon::set_ssl_keyupdate(vmp_uint ckeyupdate,vmp_uint skeyupdate)
{
    ckeyupdate_=ckeyupdate;
    skeyupdate_=skeyupdate;
}

void SslCommon::set_ssl_client_event(event::EVTCB tcpconnect,event::EVTCB sslconnect,event::EVTCBBUF crecv,event::EVTCB cclose)
{
    if(tcpconnect != 0)
        tcpconnect_=tcpconnect;
    if(sslconnect != 0)
        sslconnect_=sslconnect;
    if(crecv != 0)
        crecv_=crecv;
    if(cclose != 0)
        cclose_=cclose;
}

void SslCommon::set_ssl_server_event(net::EVTCBACCEPT tcpaccept,event::EVTCB sslaccept,event::EVTCBBUF srecv,event::EVTCB lclose,event::EVTCB sclose)
{
    if(tcpaccept != 0)
        tcpaccept_=tcpaccept;
    if(sslaccept != 0)
        sslaccept_=sslaccept;
    if(srecv != 0)
        srecv_=srecv;
    if(lclose != 0)
        lclose_=lclose;
    if(sclose != 0)
        sclose_=sclose;
}

void SslCommon::set_ssl_framing_config(vmp_size maxframesize,vmp_size maxnframes,vmp::time::Time rtimeout,vmp::time::Time ptimeout,vmp_size pinglenbody)
{
    vmp::time::Time rtmp=framing_rtimeout_,ptmp=framing_ptimeout_;
    if(rtimeout > 0.0)
        rtmp=rtimeout;
    if(ptimeout > 0.0)
        ptmp=ptimeout;
    if(rtmp > ptmp)
    {
        framing_rtimeout_=rtmp;
        framing_ptimeout_=ptmp;
    }
    else
        vmp::except_s("crypto::SslCommon::set_ssl_framing_config(rtimeout <= ptimeout)");
    if(maxframesize != 0)
        framing_maxsize_=maxframesize;
    if(maxnframes != 0)
        framing_maxn_=maxnframes;
    if(pinglenbody != 0)
        framing_pinglenbody_=pinglenbody;
}

void SslCommon::set_ssl_framing_cb(event::EVTCBBUF recv_text,event::EVTCBBUF recv_bin)
{
    if(recv_text != 0)
        framing_recv_text_=recv_text;
    if(recv_bin != 0)
        framing_recv_bin_=recv_bin;
}

vmp::str SslCommon::identity(event::Cell *cell)
{
    return cell->event<crypto::SslTimerRef>()->identity_;
}
 
void SslCommon::close_event(event::Cell *cell)
{
    crypto::SslTimerRef *routine=cell->event<crypto::SslTimerRef>();
    routine->evt_timer_close();
}
        
void SslCommon::free_ref(event::Cell *cell)
{
    crypto::SslTimerRef *routine=cell->event<crypto::SslTimerRef>();
    routine->reset();
    routine->evt_timer_free();
    tref_.free(routine);
} 

event::Cell *SslCommon::routine(crypto::TIMERCB event,crypto::TIMERCB close,void *ref,vmp::str identity,vmp::time::Time timeval)
{
    crypto::SslTimerRef *routine=tref_.get();
    if(event != 0)
        routine->event_=event;
    if(close != 0)
        routine->close_=close;
    routine->ref_=ref;
    routine->identity_=identity;
    return routine->evt_timer_new(this,timeval,crypto::routine_event,crypto::routine_event_close);
}

void SslCommon::routine_active(event::Cell *cell,vmp::time::Time timeval)
{
    cell->event<crypto::SslTimerRef>()->evt_timer_active(timeval);
}

void SslCommon::routine_deactive(event::Cell *cell)
{
    cell->event<crypto::SslTimerRef>()->evt_timer_deactive();
}

vmp_bool SslCommon::routine_isactive(event::Cell *cell)
{
    return cell->event<crypto::SslTimerRef>()->evt_timer_isactive();
}

EventSsl::EventSsl():net::EventConnection()
{
    sub_=0;
    evt_ssl_reset(true);    
}
       
EventSsl::~EventSsl()
{
    evt_ssl_reset();
}

void EventSsl::evt_ssl_reset(vmp_bool init)
{
    common_=0;
    linktype_=crypto::LINK_NORMAL;
    framing_.reset();
    framingcode_=0;
    timeping_=0;
    if(!init)
        ssl_.reset();
    sslsend_=crypto::ssl_sslsend;
}

void EventSsl::evt_ssl_ctimeout(vmp_bool active)
{
    event::Manager *manager=cell_->get_manager();
    if(active)
    {
        ctimeout_=cell_->timewait_;
        manager->cell_timewait(cell_,common_->ctimeout_);
    }
    else
    {
        manager->cell_timewait(cell_,ctimeout_);
        ctimeout_=0;
    }
}

event::Cell *EventSsl::evt_ssl_client(event::UI *ui,net::Address *peer,crypto::SslCommon *common,net::ProxyChain *proxy)
{
    vmp::except_check_pointer((void *)common,"crypto::EventSsl::evt_ssl_client(common=null)");
    common_=common;
    ctimeout_=common->ctimeout_;
    return evt_connection_tcpclient(ui,peer,ssl_tcp_connect,common->crecv_,crypto::ssl_close_client,proxy);
}

event::Cell *EventSsl::evt_ssl_listen(event::UI *ui,net::Address *local,crypto::SslCommon *common,vmp_uint backlog)
{
    vmp::except_check_pointer((void *)common,"crypto::EventSsl::evt_ssl_listen(common=null)");
    common_=common;
    ctimeout_=common->ctimeout_;
    return evt_connection_tcplisten(ui,local,backlog,crypto::ssl_tcp_accept,common->lclose_,common->srecv_,crypto::ssl_close_server);
}

void EventSsl::evt_ssl_key_update()
{
    if(ssl_.key_update() == -1)
        vmp::except_s("crypto::EventSsl::evt_ssl_key_update() failed");
}

void EventSsl::evt_ssl_key_update_routine()
{
    vmp_int type=evt_ssl_type();
    event::Cell *cell;
    vmp::time::Time time=0;
    event::Manager *manager=cell_->get_manager();
    if((cell=manager->cell_searchsub(cell_,"keyupdate")) != 0)
        return;
    if((type == crypto::CONN_SSLCLIENT) && (common_->ckeyupdate_) != 0)
        time=(vmp::time::Time)(common_->ckeyupdate_*60);
    else if((type == crypto::CONN_SSLSERVER) && (common_->skeyupdate_) != 0)
        time=(vmp::time::Time)(common_->skeyupdate_*60);  
    if(time != 0)
    {
        vmp::str identity;
        vmp::unicode::str_write(&identity,"KeyUpdate(%s)",cell_->ui<event::UI>()->identity(cell_).c_str());
        cell=common_->routine(crypto::ssl_routine_key,0,(void *)this,identity,time);
        manager->cell_addsub(cell_,"keyupdate",cell);
    }
}

vmp_int EventSsl::evt_ssl_type()
{
    vmp_int ctype=evt_connection_type();
    switch(ctype)
    {
        case net::CONN_TCPLISTEN:
            return crypto::CONN_SSLLISTEN;
        case net::CONN_TCPSERVER:
            return crypto::CONN_SSLSERVER;
        case net::CONN_TCPCLIENT:
            return crypto::CONN_SSLCLIENT;
        case net::CONN_TCPPROXY:
            return crypto::CONN_SSLPROXY;
        default: 
            break;
    }
    return crypto::CONN_NONE;
}

vmp::str EventSsl::evt_ssl_strtype()
{
    switch(evt_ssl_type())
    {
        case crypto::CONN_NONE:
            return "NONE";
        case crypto::CONN_SSLLISTEN:
            return "SSLLISTEN";
        case crypto::CONN_SSLSERVER:
            return "SSLSERVER";
        case crypto::CONN_SSLCLIENT:
            return "SSLCLIENT";
        case crypto::CONN_SSLPROXY:
            return "SSLPROXY";
        default:
            vmp::except("crypto::EventSsl::evt_ssl_strtype() conntype_=%d bad value",conntype_);
    }
    return "";
}

void EventSsl::evt_ssl_local_x509(crypto::X509_Wrap *cout)
{
    return common_->ctx_.get_x509(cout); 
}

void EventSsl::evt_ssl_peer_x509(crypto::X509_Wrap *cout)
{
    return ssl_.get_peer_x509(cout);
}

void EventSsl::evt_ssl_framing_active()
{
    event::Manager *manager=cell_->get_manager();
    linktype_=crypto::LINK_WSFRAMING;
    vmp::str identity;
    vmp::unicode::str_write(&identity,"framing_ping(%s)",cell_->ui<event::UI>()->identity(cell_).c_str());
    event::Cell *cell=common_->routine(crypto::framing_ping,0,(void *)this,identity,common_->framing_rtimeout_);
    manager->cell_addsub(cell_,"framing_ping",cell);
    cell_->read_=crypto::framing_read;
    sslsend_=crypto::ssl_sslsend_framing;
}

void EventSsl::evt_ssl_framing_next()
{
    vmp::Buf *buf=framing_.next_send();
    if(buf != 0)
    {
        cell_->writing_=true;
        evt_connection_send(buf);
        framing_.free_buf(buf);
    }
    else if(cell_->writing_)
        cell_->writing_=false;
}

void EventSsl::evt_ssl_framing_text(vmp::Buf *buf)
{
    if(linktype_ == crypto::LINK_WSFRAMING)
    {
        framing_.send_packet(buf,"text",common_->framing_maxsize_,common_->framing_maxn_);
        if(!cell_->writing_)
            evt_ssl_framing_next();
    }
}

void EventSsl::evt_ssl_framing_bin(vmp::Buf *buf)
{
    if(linktype_ == crypto::LINK_WSFRAMING)
    {
        framing_.send_packet(buf,"bin",common_->framing_maxsize_,common_->framing_maxn_);
        if(!cell_->writing_)
            evt_ssl_framing_next();
    }
}

void EventSsl::evt_ssl_framing_close(vmp_uint code,vmp::str msg)
{
    if(linktype_ == crypto::LINK_WSFRAMING)
    {
        event::Manager *manager=cell_->get_manager();
        framingcode_=code;
        if(code == packet::websocket::WebSocketCode_Normal)
            manager->cell_close_ok_spec(cell_,"crypto::EventSslFraming");
        else
        {
            if(msg == "")
                manager->cell_close_err_spec(cell_,"crypto::EventSslFraming",framingcode_,packet::websocket::WebSocketCode_reason(framingcode_));
            else
                manager->cell_close_err_spec(cell_,"crypto::EventSslFraming",framingcode_,msg);
                
        }
    }
}

void EventSsl::evt_ssl_sub_active(crypto::EventSslSub *sub)
{
    event::Manager *manager=cell_->get_manager();
    if(sub_ != 0)
        manager->cell_close_err_spec(cell_,"crypto::EventSsl",0,"crypto::EventSsl::evt_ssl_sub_active() duplex setting subprotocols");
    else if(sub == 0)
        manager->cell_close_err_spec(cell_,"crypto::EventSsl",0,"crypto::EventSsl::evt_ssl_sub_active(sub=Null)");
    else
    {
        sub_=sub;
        sub_->cell_=cell_;
        manager->cell_alloc(cell_);
    }
}

void EventSsl::evt_ssl_send(vmp::Buf *buf)
{
    sslsend_(cell_,buf);
}

crypto::EventSslSub *EventSsl::evt_ssl_sub_reset()
{
    crypto::EventSslSub *ret;
    ret=sub_;
    if(sub_ != 0)
    {
        event::Manager *manager=cell_->get_manager();
        sub_->cell_=0;
        sub_=0;
        manager->cell_release(cell_);
    }
    return ret;
}

void EventSsl::evt_ssl_close()
{
    if(sub_ != 0)
        sub_->evtsub_ssl_close();
    try
    {
        if(linktype_ == crypto::LINK_WSFRAMING)
        {
             if(cell_->closelevel_ == "")
                 cell_->closelevel_="crypto::EventSslFraming"; 
             if(cell_->closelevel_ == "crypto::EventSslFraming")
             {
                 if(framingcode_ == 0)
                 {
                     if(cell_->ret_ == event::SUCCESS)
                         framingcode_=packet::websocket::WebSocketCode_Normal;   
                     else if(cell_->ret_ == event::ERROR)
                     {
                         framingcode_=packet::websocket::WebSocketCode_Unexpected;
                         cell_->errcode_=framingcode_;
                     }
                 }
                 if(framingcode_ != 0)
                 {
                     vmp::Buf *buf=framing_.frame_close(framingcode_,cell_->err_);
                     evt_connection_send(buf,true);
                     framing_.free_buf(buf);
                 }
             }
        }
        ssl_.close();
    }
    catch(vmp::exception &x)
    {
    }
    evt_connection_close();
}
       
void EventSsl::evt_ssl_free()
{
    if(sub_ != 0)
        sub_->evtsub_ssl_free();
    evt_ssl_reset();
    evt_connection_free();
}

}}

