/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 10/10/2024
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

vmp_bool ssl_connect(event::Manager *manager,event::Cell *cell,crypto::EventSsl *ssl)
{
    vmp_bool exec=false;
    vmp_int ret=ssl->ssl_.connect();
    if(ret == 0)
    {
        exec=manager->cell_update(cell);
        cell->read_=ssl_ssl_crecv;
        ssl->send_=crypto::ssl_send;
        ssl->evt_ssl_ctimeout(false);
    }
    else if(ret == -1)
        manager->cell_close_err_spec(cell,"crypto::EventSsl",ssl->ssl_.get_errno(),crypto::err_string());
    return exec;
}

vmp_bool ssl_accept(event::Manager *manager,event::Cell *cell,crypto::EventSsl *ssl)
{
    vmp_bool exec=false;
    vmp_int ret=ssl->ssl_.accept();
    if(ret == 0)
    {
        exec=manager->cell_update(cell);
        cell->read_=ssl_ssl_srecv;
        ssl->send_=crypto::ssl_send;
        ssl->evt_ssl_ctimeout(false);
    }
    else if(ret == -1)
        manager->cell_close_err_spec(cell,"crypto::EventSsl",ssl->ssl_.get_errno(),crypto::err_string());
    return exec;
}

void ssl_tcp_connect(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->common_->tcpconnect_(cell);
    manager->lock();
    try
    {
        ssl->ssl_.set(&(ssl->common_->ctx_),ssl->evt_connection_socket());
        ssl->evt_ssl_ctimeout(true); 
        cell->read_=crypto::ssl_ssl_connect;
        exec=ssl_connect(manager,cell,ssl);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(cell,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
    if(exec)
    {
        ssl->evt_ssl_key_update_routine();
        ssl->common_->sslconnect_(cell);
    }
}

void ssl_tcp_accept(event::Cell *cell,event::Cell *child)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    manager->lock();
    crypto::EventSsl *cssl=child->event<crypto::EventSsl>();
    cssl->common_=ssl->common_;
    cssl->framing_.init(&(cssl->common_->framing_ext_));
    manager->unlock();
    ssl->common_->tcpaccept_(cell,child);
    manager->lock();
    try
    {
        cssl->ssl_.set(&(cssl->common_->ctx_),cssl->evt_connection_socket());
        cssl->evt_ssl_ctimeout(true);
        child->read_=crypto::ssl_ssl_accept;
        exec=ssl_accept(manager,child,cssl);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(child,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
    if(exec)
    {
        cssl->evt_ssl_key_update_routine();
        cssl->common_->sslaccept_(child);
    }
}

void ssl_ssl_connect(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    try
    {
        exec=ssl_connect(manager,cell,ssl);   
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(cell,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
    if(exec)
    {
        ssl->evt_ssl_key_update_routine();
        ssl->common_->sslconnect_(cell);
    }
}

void ssl_ssl_accept(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    try
    {
        exec=ssl_accept(manager,cell,ssl);   
        
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(cell,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
    if(exec)
    {
        ssl->evt_ssl_key_update_routine();
        ssl->common_->sslaccept_(cell);
    }
}

void ssl_ssl_crecv(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    vmp::Buf buf;
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    try
    {
        vmp_int ret=ssl->ssl_.recv(&buf);
        if(ret == 0)
            manager->cell_close_ok_spec(cell,"crypto::EventSsl");
        else if(ret > 0)
            exec=manager->cell_update(cell);
        else if(ret == -1)
            manager->cell_close_err_spec(cell,"crypto::EventSsl",ssl->ssl_.get_errno(),crypto::err_string());
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(cell,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
    if(exec)
        ssl->common_->crecv_(cell,&buf);
    buf.reset();  
}

void ssl_ssl_srecv(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    vmp::Buf buf;
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    try
    {
        vmp_int ret=ssl->ssl_.recv(&buf);
        if(ret == 0)
            manager->cell_close_ok_spec(cell,"crypto::EventSsl");
        else if(ret > 0)
            exec=manager->cell_update(cell);
        else if(ret == -1)
            manager->cell_close_err_spec(cell,"crypto::EventSsl",ssl->ssl_.get_errno(),crypto::err_string());
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(cell,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
    if(exec)
        ssl->common_->srecv_(cell,&buf);
    buf.reset();  
}

void ssl_send(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    if(ssl->ssl_.send(buf) == -1)
        manager->cell_close_err_spec(cell,"crypto::EventSsl",vmp::get_errno(),vmp::value_errno());
}

void ssl_sslsend(event::Cell *cell,vmp::Buf *buf)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_connection_send(buf);   
}

void ssl_sslsend_framing(event::Cell *cell,vmp::Buf *buf)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_ssl_framing_bin(buf); 
}

void ssl_routine_key(void *ref)
{
    crypto::EventSsl *ssl=(crypto::EventSsl *)ref;
    event::Manager *manager=ssl->cell_->get_manager();
    manager->lock();
    try
    {
        ssl->evt_ssl_key_update();
    }
    catch(vmp::exception &x)
    {
        manager->cell_close_err_spec(ssl->cell_,"crypto::EventSsl",0,x.what());
    }
    manager->unlock();
}

void ssl_subprotocols_recv(event::Cell *cell,vmp::Buf *buf)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->sub_->evtsub_ssl_recv(buf);
}

void ssl_close_client(event::Cell *cell)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    if(ssl->sub_ != 0)
        ssl->sub_->close_(cell);
    ssl->common_->cclose_(cell);
}

void ssl_close_server(event::Cell *cell)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    if(ssl->sub_ != 0)
       ssl->sub_->close_(cell);
    ssl->common_->sclose_(cell);
}

}}

