#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 08/12/2024

import vmp
import vmp_event       as event
import vmp_net         as net
import vmp_plugin.base as base
import vmp_json        as json
import vmp_crypto      as crypto
from vmp_jrp_  import JrpUI     as JrpUI
from vmp_jrp_  import JrpCommon as JrpCommon

ptype_="client.jrp"

def session(cell):
  plugin=cell.getvar("plugin")
  plugin.peerstatus="connect"
  plugin.iblock=False
  plugin.print_connect()
  plugin.prompt()
  
def close(cell):
  plugin=cell.remvar("plugin")
  plugin.peer.release()
  plugin.peer=None
  plugin.peerstatus="close"
  plugin.exitlevel=cell.closelevel()
  plugin.exitcode=cell.errcode()
  if cell.ret() == event.CELLRET_ERROR:
    plugin.exitmsg=cell.str_error()
  elif cell.ret() == event.CELLRET_TIMEOUT:
    plugin.exitmsg="timeout"
  if plugin.iblock:
    vmp.output("Server Close (exitlevel={0},errocde={1},errmsg={2})".format(plugin.exitlevel,plugin.exitcode,plugin.exitmsg))
    plugin.iblock=False
    plugin.prompt();
  elif plugin.exitlevel == "crypto::EventSslFramingPeer":
    vmp.output("\nServer Close (exitlevel={0},errocde={1},errmsg={2})".format(plugin.exitlevel,plugin.exitcode,plugin.exitmsg))
    plugin.prompt();

def response_d(jreq,data,payload):
  cell=jreq.cell()
  plugin=cell.getvar("plugin")
  rid=jreq.rid()
  plugin.response_i(jreq,data)
  
def close_d(jreq):
  cell=jreq.cell()
  plugin=cell.getvar("plugin")
  plugin.close_i(jreq)
  rid=jreq.rid()
  request=plugin.requests[rid]
  request.status=jreq.status()
  request.msg=jreq.msg()
  request.jreq=None
  jreq.release()
  
def helper(plugin,args):
  cmds=dict(sorted(plugin.cmds.items()))
  if(len(args) == 0):
    vmp.output("Command list:")
    for cmd in cmds:
      vmp.output("{0:<20} - {1}".format(cmd,cmds[cmd].desc))
  else:
    cmd=args[0]
    if cmd in cmds:
      vmp.output("Command '{0}' - {1}".format(cmd,cmds[cmd].desc))
      for i in range(0,len(cmds[cmd].args)):
        vmp.output("  arg{0:<2} - {1}".format(i,cmds[cmd].args[i]))
      vmp.output("min arguments number: {0}".format(cmds[cmd].minargs))
    else:
      vmp.except_s("not found helper cmd {0}".format(args[0]))
  
def local(plugin,args):
  vmp.output("Subject:        [{0}]".format(plugin.subject))
  vmp.output("Finegrprint:    [{0}]".format(plugin.ctx.fingerprint()))
  vmp.output("Root Directory: [{0}]".format(plugin.rootpath))
  
def connect(plugin,args):
  if plugin.peerstatus == "close":
    try:
      plugin.exitlevel=""
      plugin.exitcode=None
      plugin.exitmsg=""
      plugin.iblock=True;
      addr=net.Address()
      if (plugin.proxychain == None) or (plugin.proxychain.size() == 0):
         addr.set(args[0],"{0}".format(args[1]))
      else:
         addr.set(args[0],"{0}".format(args[1]),True)
      plugin.peer=plugin.jrpui.new_client(addr,plugin.proxychain)
      plugin.peer.setvar("plugin",plugin)
      plugin.peer.alloc()
      plugin.requests={}
      plugin.vars={}
      plugin.peerstatus="connecting"
    except Exception as e:
      vmp.error("{0}".format(str(e)))
  else:
    vmp.error("Client already connected to a peer")
    vmp.time.vsleep(0.1)
  
def connectstatus(plugin,args):
  if plugin.peerstatus == "close":
    if plugin.exitcode == None:
      vmp.output("Disconnect")
    elif plugin.exitmsg == "timeout":
      vmp.output("Close Timeout")
    elif plugin.exitmsg == "":
      vmp.output("Close ok (exitlevel={0})".format(plugin.exitlevel))
    else:
      vmp.output("Close (exitlevel={0},errocde={1},errmsg={2})".format(plugin.exitlevel,plugin.exitcode,plugin.exitmsg))
  elif plugin.peerstatus == "connecting":
    vmp.output("connecting")
  else:
    plugin.print_connect()
    
def disconnect(plugin,args):
  if plugin.peer != None:
    plugin.iblock=True
    plugin.peer.close()
  else:
    vmp.error("Peer already disconnect")
    vmp.time.vsleep(0.1)

def requests(plugin,args):
  requests=dict(sorted(plugin.requests.items()))
  vmp.output("|  rid  |          type           | status  |  exit message")
  for req in requests:
    data=requests[req]
    if data.status == 'open':
      vmp.output("| {0:^5} |  {1:^20}   |  open   | {2:^30}".format(req,data.type,''))
    else:
      vmp.output("| {0:^5} |  {1:^20}   | {2:^6}  |   {3}".format(req,data.type,data.status,data.msg))
  
def kill(plugin,args):
  if args[0] in plugin.requests:
    data=plugin.requests[args[0]]
    if data.status == 'open':
      data.jreq.kill()
      vmp.output("Send Kill {0}".format(args[0]))
      return []
  vmp.output("Request {0} not active".format(args[0]))
  
def quit(plugin,args):
  plugin.manager.stop()

def varars(plugin,args):
  if len(args) == 0:
    vmp.output("Session list:")  
    for v in plugin.vars:
      vmp.output("{0}".format(v))  
  elif len(args) == 1:
    vmp.output("Var list session '{0}':".format(args[0]))
    if args[0] in plugin.vars: 
      for v in plugin.vars[args[0]]:
        vmp.output("{0:<20} - {1}".format(v,plugin.vars[args[0]][v].desc))   
  elif len(args) == 2:
    if args[0] in plugin.vars and args[1] in plugin.vars[args[0]]:
      vmp.output("{0}".format(plugin.vars[args[0]][args[1]].value))
    else:
      vmp.error("Var ({0},{1}) not found".format(args[0],args[1]))
      vmp.time.vsleep(0.1)

def status(plugin,args):
  try:
    j=json.Json()
    json.japi_status(j.root())
    plugin.send_request(j.root())
    vmp.output("Result in section var 'status'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def userlist(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('userinfo')
    j=json.Json()
    json.japi_userlist(j.root())
    plugin.send_request(j.root())
    vmp.output("Result in section var 'userinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def useradd(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('userinfo')
    j=json.Json()
    json.japi_useradd(j.root(),args[0],args[1],args[2],args[3])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'userinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def usermod(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('userinfo')
    j=json.Json()
    json.japi_usermod(j.root(),args[0],args[1],args[2])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'userinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def userpasswd(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('userinfo')
    j=json.Json()
    json.japi_userpasswd(j.root(),args[0],args[1])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'userinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)
    
def userdel(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('userinfo')
    j=json.Json()
    json.japi_userdel(j.root(),args[0])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'userinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def peerlist(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('peerinfo')
    j=json.Json()
    json.japi_peerlist(j.root())
    plugin.send_request(j.root())
    vmp.output("Result in section var 'peerinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def peeradd(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('peerinfo')
    j=json.Json()
    json.japi_peeradd(j.root(),args[0],args[1],args[2])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'peerinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)
    
def peermod(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('peerinfo')
    j=json.Json()
    json.japi_peermod(j.root(),args[0],args[1])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'peerinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)

def peerdel(plugin,args):
  try:
    plugin.busy()
    plugin.newsectionvar('peerinfo')
    j=json.Json()
    json.japi_peerdel(j.root(),args[0])
    plugin.send_request(j.root())
    vmp.output("Result in section var 'peerinfo'")
  except Exception as e:
    vmp.error(str(e))
    vmp.time.vsleep(0.1)
   
#Internal usage
class Requests:
  def __init__(self,jreq):
    self.jreq=jreq
    self.type=jreq.jdata_type()
    self.status="open"
    self.msg=''
    
class Var:
  def __init__(self,value,desc):
    self.value=value
    self.desc=desc

##Internal usage
class Cmd:
  def __init__(self,desc,minargs,args,cbfunc,prompt):
    self.desc=desc
    self.minargs=minargs
    self.args=args
    self.cbfunc=cbfunc
    self.prompt=prompt

  def check(self,args):
    retargs=[]
    l=len(args)
    if (l < self.minargs) or (l > len(self.args)):
      vmp.except_s("bad arguments number")
    i=0
    for arg in args:
      t=self.args[i]
      try:
        if t[1] == 'generic':
          retargs.append(args[i])
        elif t[1] == 'integer':
          minvalue=vmp.INTMIN
          maxvalue=vmp.INTMAX
          if(len(t) >= 3) and t[2] != None:
            minvalue=t[2]
          if(len(t) >= 4) and t[3] != None:
            maxvalue=t[3]
          v=vmp.unicode.str_todigit_range(args[i],minvalue,maxvalue)
          retargs.append(v)
        elif t[1] == 'real':
          minvalue=vmp.INTMIN
          maxvalue=vmp.INTMAX
          if(len(t) >= 3) and t[2] != None:
            minvalue=t[2]
          if(len(t) >= 4) and t[3] != None:
            maxvalue=t[3]
          v=vmp.unicode.str_toreal_range(args[i],minvalue,maxvalue)
          retargs.append(v)
        elif t[1] == 'boolean':
          if args[i] == 'true':
            retargs.append(True)
          elif args[i] == 'false':
            retargs.append(False)
          else:
            vmp.except_s("{0} not boolean type".format(args[i]))
        elif t[1] == 'hex':
          if not str_istype(args[i],"xdigit"):
            vmp.except_s("{0} not exadecimal type".format(args[i]))
          retargs.append(args[i])
        elif t[1] == 'ip':
          if not net.is_ip_raw(args[i]):
            vmp.except_s("{0} not ip type".format(args[i]))
          retargs.append(args[i])
        elif t[1] == 'ip4':
          if not net.is_ipv4_raw(args[i]):
            vmp.except_s("{0} not ipv4 type".format(args[i]))
        elif t[1] == 'ip6':
          if not net.is_ipv6_raw(args[i]):
            vmp.except_s("{0} not ipv6 type".format(args[i]))
          retargs.append(args[i])
        elif t[1] == 'mac':
          if not net.is_macaddress_raw(args[i]):
            vmp.except_s("{0} not mac type".format(args[i]))
          retargs.append(args[i])
        else:
          vmp.except_s("fatal error bad description tuple")
      except Exception as e:
        vmp.except_s("argn={0} {1}".format(i,str(e)))
      i=i+1
    return retargs
    
  def exec(self,plugin,args):
    retargs=self.check(args)
    return self.cbfunc(plugin,retargs)

  
##Plugin interface used for input and output operations.
class Plugin(base.Plugin):
  def __init__(self,data,options):
    base.Plugin.__init__(self,ptype_,data,6,options)
    self.logger=vmp.utils.Logger()
    self.manager=data[0]
    self.rootpath=data[1]
    self.subject=data[2]
    self.prompt_active=data[3]
    self.debug=data[4]
    self.proxychain=data[5]
    
    self.cmds={}
    
    self.add_command("helper","Helper command",0,[('command name','generic')],helper)
    self.add_command("local","Information about the identity of the local peer",0,[],local)
    self.add_command("connect","Connect to jrp peer",2,[('host address peer','generic'),('host service','integer',1,65535)],connect)
    self.add_command("connectstatus","Print connection client status",0,[],connectstatus)
    self.add_command("disconnect","Disconnect peer",0,[],disconnect)
    self.add_command("requests","List of requests sent",0,[],requests)
    self.add_command("kill","Send a kill message to a request ",1,[("request number",'integer',0,None)],kill)
    self.add_command("quit","Exit console",0,[],quit,False)
    self.add_command("status","Send a jrp status request",0,[],status)
    self.add_command("vars","Print section list if argument is not specified,print variable list if section is specified,print variable value if section variable is specified",0,[('session name','generic'),('variable name','generic')],varars)
    self.add_command("userlist","List of web users",0,[],userlist)
    self.add_command("useradd","Add a web user",4,[('web user','generic'),('user password','generic'),('user permits','integer',0,vmp.INTMAX),('if access is allowed only in localhost','boolean')],useradd)
    self.add_command("usermod","Change ownership of a web user",3,[('web user','generic'),('user permits','integer',0,vmp.INTMAX),('if access is allowed only in localhost','boolean')],usermod)
    self.add_command("userpasswd","Change password of a web user",2,[('web user','generic'),('user password','generic')],userpasswd)
    self.add_command("userdel","Delete a web user's password",1,[('web user','generic')],userdel)
    self.add_command("peerlist","List of peers",0,[],peerlist)
    self.add_command("peeradd","Add a peer",3,[('fingerprint peer','generic'),('subject peer','generic'),('user permits','integer',0,vmp.INTMAX)],peeradd)
    self.add_command("peermod","Change ownership of a peer",2,[('fingerprint peer','generic'),('user permits','integer',0,vmp.INTMAX)],peermod)
    self.add_command("peerdel","Delete a peer",1,[('fingerprint','generic')],peerdel)
    self.ctx=crypto.Ctx_Peer(vmp.fs.union_path(self.rootpath,"x509"),1,self.subject)
    
    if self.debug:
      self.logger.set(vmp.fs.union_path(self.rootpath,'client.log'),vmp.utils.LOG_DEBUG,True)
    else:
      self.logger.set(vmp.fs.union_path(self.rootpath,'client.log'),vmp.utils.LOG_INFO,True)
    
    self.jrpcommon=JrpCommon()
    self.jrpcommon.jrp_callback(None,None,None,None,response_d,close_d)
    self.jrpui=JrpUI(data[0],self.ctx,self.jrpcommon,self.logger)
    self.jrpui.set_p2p_client_event(None,session,close);
    
    self.peer=None
    self.peerstatus="close"
    self.iblock=False
    self.exitlevel=""
    self.exitcode=None
    self.exitmsg=""
    self.requests={}
    self.vars={}
    self.busystate=False
    
    self.prompt()
   
  def busy(self):
    if self.busystate:
      vmp.except_s("Request Manager is busy")
    self.busystate=True
  
  def unbusy(self):
    self.busystate=False
    
  def prompt(self):
    if self.prompt_active:
      vmp.output_raw("ragnu>");
  
  def print_connect(self):
    vmp.output("Connect      peer {0}".format(self.peer.identity()));
    vmp.output("Subject      peer {0}".format(self.jrpui.peer_subject(self.peer)))
    vmp.output("Fingerprint  peer  {0}".format(self.jrpui.peer_fingerprint(self.peer)))
    vmp.output("LocalPermits peer {0}".format(self.jrpui.permits(self.peer)))
    vmp.output("Broadcast Data peer:".format(self.jrpui.broadcastdata_peer(self.peer)))
    vmp.output("Session Data peer:")
    vinput=self.jrpui.all_input(self.peer)
    for v in vinput:
       vmp.output("  (input='{0}',push={1},response={2})".format(v,self.jrpui.search_push(self.peer,v),self.jrpui.search_response(self.peer,v)))
  
  ##Create section var
  #
  #@param name section name
  def newsectionvar(self,name):
    if name in self.vars:
      self.delsectionvar(name)
    self.vars[name]={}
  
  ##Remove an section from list
  #
  #@param name section name
  def delsectionvar(self,name):
    self.vars.pop(name)
    
  ## Set Variable value
  #
  #@param section section var
  #@param name variable name
  #@param value variable value
  #@param desc description field
  def setvar(self,section,name,value,desc):
    if not section in self.vars: 
      self.newsectionvar(section)
    if section in self.vars and name in self.vars[section]:
      self.vars[section][name].value=value
      self.vars[section][name].desc=desc
    else:
      self.vars[section][name]=Var(value,desc)
  
  ## Delete Variable value
  #
  #@param name variable name
  #@param section section var
  def delvar(self,section,name):
    if section in self.vars and name in self.vars[section]:
      self.vars[section].pop(name)
  
  ## Send a request jrp a peer connected
  #
  #@param jobj json object to send
  #@param payload payload buffer data
  #@return request id or or except in case of failure
  def send_request(self,jobj,payload=None):
    if self.peer != None:
      jreq=self.jrpui.request(self.peer,jobj,payload)
      rid=jreq.rid()
      self.requests[rid]=Requests(jreq)
      jreq.alloc()
      return rid
    vmp.except_s("Peer not connected")
  
  ## Push a value to request jrp
  #
  #@param rid request id
  #@param jobj json object to push
  #@param payload payload buffer data
  #@return void or except in case of failure
  def push_request(self,rid,jobj,payload=None):
     if self.peer == None:
       vmp.except_s("Peer not connected")
     if (rid in self.requests) and (self.requests[rid].status == "open"):
       self.requests[rid].push(jobj,payload)
       vmp.output("Send Push to request {0}".format(rid))
     else:
       vmp.except_s("Request {0} not open".format(rid))
  
  ##Send a broadcast message
  #
  #@param jobj json object to push
  #@param payload payload buffer data
  #@return void or except in case of failure
  def send_broadcast(self,rid,jobj,payload=None):
    self.jrpui.broadcast(self.peer,jobj,payload)
  
  ## Add command to plugin
  #
  #@param name command name
  #@param helper helper string
  #@param minargs minimum number of arguments
  #@param args list of tuple Tuple [(argument decription,argtype,...var arguments),...]
  #argtype :
  #'generic' - generic arguments
  #'integer' - integer value var arguments=(minvalue='',maxvalue='') ''=not assigned
  #'boolean' - boolean value
  #'real'    - integer value var arguments=(minvalue='',maxvalue='') ''=not assigned
  #'hex'     - hexadecimal value
  #'ip'      - ip value
  #'ip4'     - ipv4 value
  #'ip6'     - ipv6 value
  #'mac'     - mac address value 
  #@param callback function of type [list request id open request]  cbfunc(plugin,args,debug),return true if debug mode
  #@prompt if true activates prompt after execution
  def add_command(self,name,helper,minargs,args,cbfunc,prompt=True):
    self.cmds[name]=Cmd(helper,minargs,args,cbfunc,prompt)
  
  ## Run a command
  #
  #@param cmdline command line
  def command(self,cmdline):
    if (cmdline != '\n') and (not self.iblock):
      try:
        split=vmp.unicode.shlex_split(cmdline)
        if len(split) > 0:
          cmd=split[0]
          args=split[1:]
          if cmd in self.cmds:
            try:
              self.cmds[cmd].exec(self,args)
              if not self.cmds[cmd].prompt:
                return
            except Exception as e:
              vmp.output("Command '{0}'".format(str(e)))
          else:
            vmp.except_s("Command '{0}' not found".format(cmd)) 
      except Exception as e:
        vmp.error(str(e))
        vmp.time.vsleep(0.1)
    if not self.iblock:
      self.prompt()
  
  #Internal usage
  def response_i(self,jreq,jdata):
    jd=json.JData()
    jd.set(jdata)
    if jd.jtype() == json.japi("statusupdate"):
      key=jd.get_text("key")
      jobj=json.JsonObj()
      j=json.Json()
      jd.get_subtype("data",jobj)
      j.json_new_obj(jobj)
      self.setvar('status',key,j.json_str(),"status update var")
    elif jd.jtype() == json.japi("statuscancel"):
      key=jd.get_text("key")
      self.delvar('status',key)
    elif jd.jtype() == json.japi("peerinfo"):
      fng=jd.get_text("fingerprint")
      d={'fingerprint':fng,'subject':jd.get_text('subject'),'permits':jd.get_integer_range('permits',0,vmp.INTMAX)}
      value="{0}".format(d)
      self.setvar('peerinfo',fng,value,"peer info")
    elif jd.jtype() == json.japi("userinfo"):
      user=jd.get_text("user")
      if jd.get_bool("only_localhost"):
        o="True"
      else:
        o="False"
      d={'user':user,'permits':jd.get_integer_range('permits',0,vmp.INTMAX),'only_localhost':o}
      value="{0}".format(d)
      self.setvar('userinfo',user,value,"user '{0}' info".format(user))
    else:
      self.response(jreq,jdata)
    
  #Internal usage
  def close_i(self,jreq):
    jd=json.JData()
    jd.set(jreq.jdata_root())
    t=jd.jtype()
    if (t == json.japi("userlist")) or (t == json.japi("useradd")) or (t == json.japi("usermod")) or (t == json.japi("userpasswd")) or (t == json.japi("userdel")) or (t == json.japi("peerlist")) or (t == json.japi("peeradd")) or (t == json.japi("peermod")) or (t == json.japi("peerdel")):
      self.unbusy()
    self.close(jreq)
    
  ##Virtual function for the management of incoming responses
  #
  #@param jreq json request associated
  #@param jdata jsonObject data recv
  def response(self,jreq,djata):
    pass
    
  ##Virtual function for the management close response
  #
  #@param jreq json request associated
  def close(self,jreq):
    pass
       
##Plugin interface topic
class PluginTopic(base.PluginTopic):
  def __init__(self,modname,plugname):
    base.PluginTopic.__init__(self,ptype_,modname,plugname)
    self.set_desc("Plugin Interface used for creating jrp clients")
    self.add_data("An event manager used for JRP events")
    self.add_data("The root directory of the files used by the plugin")
    self.add_data("The subject for the peer client certificate")
    self.add_data("Print the video prompt?(boolean)")
    self.add_data("Run in debug mode?(boolean)")
    self.add_data("Proxychain connection data,if none not proxy used")
    self.add_function("busy","Occupy the request manager for operations that need to be done one at a time")
    self.add_function("unbusy","Occupy the request manager for operations that need to be done one at a time",[],"void or except in case the request manager is busy")
    self.add_function("prompt","Print the prompt on the screen")
    self.add_function("newsectionvar","Create new section var",[('name','variable name')])
    self.add_function("delsectionvar","Delete section var",[('name','variable name')])
    self.add_function("setvar","Set a variable",[('section','section var'),('name','variable name'),('name','variable value'),('desc','description field')])
    self.add_function("delvar","Delete Variable value",[('section','section var'),('name','variable name')])
    self.add_function("send_request","Send a request jrp a peer connected",[('jobj','json object to send'),('payload','Binary data to send')],'request id or or except in case of failure')
    self.add_function("push_request","Push a value to request jrp",[('rid','request id'),('jobj','json object to push'),('payload','Binary data to send')],'void or except in case of failure')
    self.add_function("send_broadcast","Send a broadcast message",[('jobj','json object to send'),('payload','Binary data to send')],'void or except in case of failure')
    self.add_function("add_command","Add command to plugin",[('name','command name'),('helper','helper string'),('minargs','minimum number of arguments'),('args','arguments list description (see doxygen doc in plugin type)'),('callback','callback function to run command (see doxygen doc in plugin type)'),('prompt','if true active prompt')])
    self.add_function("command","Send the plugin a command to run",[('cmdline','command string')])
    self.add_function("response","Virtual function for the management of incoming responses",[('jreq','json request associated'),('jdata','json object data recv')])
    self.add_function("close","Virtual function for the management close requests",[('jreq','json request associated')])
    
    
