/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 15/10/2024
 */
 
#include "crypto_.h"

VAMPIRIA_PYLIB_INIT(vmp_crypto_);

class PySslUI:public crypto::SslUI
{
   public:
        PySslUI(event::Manager *manager,crypto::Ctx *ctx);
        PySslUI(event::Manager *manager,crypto::Ctx *ctx,vmp::time::Time ctimeout);
        ~PySslUI();

        pylib::PyCall pytcpconnect_;
        pylib::PyCall pycsslconnect_;
        pylib::PyCall pycrecv_;
        pylib::PyCall pycclose_;

        pylib::PyCall pyaccept_;
        pylib::PyCall pyssslconnect_;
        pylib::PyCall pysrecv_;
        pylib::PyCall pylclose_;
        pylib::PyCall pysclose_;
};

PySslUI::PySslUI(event::Manager *manager,crypto::Ctx *ctx):crypto::SslUI(manager,ctx)
{
}

PySslUI::PySslUI(event::Manager *manager,crypto::Ctx *ctx,vmp::time::Time ctimeout):crypto::SslUI(manager,ctx,ctimeout)
{
}

PySslUI::~PySslUI()
{
}

struct stub_sslui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PySslUI *>(cell->ui<PySslUI>(),false);
        }
        return 0;
    }
};

struct SslUI_Constructor:public pylib::Constructor<PySslUI>
{
    PySslUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 2)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             crypto::Ctx *ctx=args.get<crypto::Ctx *>(1);
             return new PySslUI(manager,ctx);
         }
         else if(args.size() == 3)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             crypto::Ctx *ctx=args.get<crypto::Ctx *>(1);
             vmp::time::Time ctimeout=args.get<vmp::time::Time>(2);
             return new PySslUI(manager,ctx,ctimeout);
         }
         return 0;
    }
};

void ssl_pytcpconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pytcpconnect_.call(cargs);
}

void ssl_pycsslconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pycsslconnect_.call(cargs);
}

void ssl_pycrecv(event::Cell *cell,vmp::Buf *buf)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    sslui->pycrecv_.call(cargs);
}

void ssl_pycclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pycclose_.call(cargs);
}

struct stub_SslUI_set_event_client:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 4)
        {
            event::EVTCB      tcpconnect;
            event::EVTCB      sslconnect;
            event::EVTCBBUF   crecv;
            event::EVTCB      cclose;
            
            args.get_callable(0,instance->pytcpconnect_);
            args.get_callable(1,instance->pycsslconnect_);
            args.get_callable(2,instance->pycrecv_);
            args.get_callable(3,instance->pycclose_);
            
            if(pylib::is_none(instance->pytcpconnect_.ref()))
                tcpconnect=0;
            else
                tcpconnect=ssl_pytcpconnect;
            if(pylib::is_none(instance->pycsslconnect_.ref()))
                sslconnect=0;
            else 
                sslconnect=ssl_pycsslconnect;
            if(pylib::is_none(instance->pycrecv_.ref()))
                crecv=0;
            else 
                crecv=ssl_pycrecv;
            if(pylib::is_none(instance->pycclose_.ref()))
                cclose=0;
            else 
                cclose=ssl_pycclose;            
            instance->set_event_client(tcpconnect,sslconnect,crecv,cclose);
            return pylib::retnone();                      
        }
        return 0;
    }
};

void ssl_pyaccept(event::Cell *listen,event::Cell *server)
{
    PySslUI *sslui=listen->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(listen,false);
    cargs.add<event::Cell *>(server,false);
    sslui->pyaccept_.call(cargs);
}

void ssl_pyssslconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pyssslconnect_.call(cargs);
}

void ssl_pysrecv(event::Cell *cell,vmp::Buf *buf)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    sslui->pysrecv_.call(cargs);
}

void ssl_pylclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pylclose_.call(cargs);
}

void ssl_pysclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pysclose_.call(cargs);
}

struct stub_SslUI_set_event_server:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 5)
        {
            net::EVTCBACCEPT  accept;
            event::EVTCB      sslconnect;
            event::EVTCB      lclose; 
            event::EVTCBBUF   srecv;
            event::EVTCB      sclose;
            
            args.get_callable(0,instance->pyaccept_);
            args.get_callable(1,instance->pyssslconnect_);
            args.get_callable(2,instance->pysrecv_);
            args.get_callable(3,instance->pylclose_);
            args.get_callable(4,instance->pysclose_);
            
            if(pylib::is_none(instance->pyaccept_.ref()))
                accept=0;
            else 
                accept=ssl_pyaccept;
            if(pylib::is_none(instance->pyssslconnect_.ref()))
                sslconnect=0;
            else 
                sslconnect=ssl_pyssslconnect;
            if(pylib::is_none(instance->pysrecv_.ref()))
                srecv=0;
            else 
                srecv=ssl_pysrecv;
            if(pylib::is_none(instance->pylclose_.ref()))
                lclose=0;
            else 
                lclose=ssl_pylclose;
            if(pylib::is_none(instance->pysclose_.ref()))
                sclose=0;
            else 
                sclose=ssl_pysclose;
            instance->set_event_server(accept,sslconnect,srecv,lclose,sclose);
            return pylib::retnone();                      
        }
        return 0;
    }
};

struct stub_SslUI_new_client:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            net::Address *peer=args.get<net::Address *>(0);
            return pylib::Converter<event::Cell *>(instance->new_client(peer),false);
        }
        else if(args.size() == 2)
        {
            net::Address *peer=args.get<net::Address *>(0);
            net::ProxyChain *proxy=0;
            if(!args.is_none(1))
                proxy=args.get<net::ProxyChain *>(1);
            return pylib::Converter<event::Cell *>(instance->new_client(peer,proxy),false);
        }
        return 0;
    }
};

void crypto_init(pylib::Module module)
{
    module.def_("init",crypto::init);
    module.def_("end",crypto::end);

    //Wrapper cell->ui()
    module.def_with_stub("sslui",stub_sslui());
    
    pylib::Class<PySslUI> sslui=module.class_with_constructor<PySslUI>("SslUI",SslUI_Constructor());
    sslui.def_with_stub("set_event_client",stub_SslUI_set_event_client());
    sslui.def_with_stub("set_event_server",stub_SslUI_set_event_server());
    sslui.def_("set_time_keyupdate",&PySslUI::set_time_keyupdate);
    sslui.def_with_stub("new_client",stub_SslUI_new_client());
    sslui.def_("new_listen",&PySslUI::new_listen,false);
    sslui.def_("send",&PySslUI::send);
    sslui.def_("key_update",&PySslUI::key_update);
    sslui.def_("localaddr",&PySslUI::localaddr,false);
    sslui.def_("peeraddr",&PySslUI::peeraddr,false);
    sslui.def_("proxychain",&PySslUI::proxychain,false);
    sslui.def_("connection_type",&PySslUI::connection_type);
    sslui.def_("connection_strtype",&PySslUI::connection_strtype);
    sslui.def_("get_local_x509",&PySslUI::get_local_x509);
    sslui.def_("get_peer_x509",&PySslUI::get_peer_x509);
}

void vampiria_pylib_init_module(pylib::Module init_module)
{
    crypto_init(init_module);
    crypto_hash(init_module);    
    crypto_pkey(init_module);
    crypto_x509(init_module);
    crypto_ssl(init_module);
    crypto_ctx(init_module);
}

 
 
