/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 27/08/2024
 */
/** 
  *Returns the value associated with the input css variable
  *@param name {String} variable name
  *@return the value associated with the input or undefined in case the variable not found
*/
function vmp_rootvar(name)
{
  let ret = getComputedStyle(document.documentElement).getPropertyValue(name);
  return (ret != '')?ret:undefined;
}
/**
  *@throws throws exception
  *@param msg {String} message error
*/
function vmp_except(msg){throw new Error(msg);}
/**
  *Wrapper of the javascript function document.createElement
  *@param tagname {string} tag name
  *@param type {string} tag type('div','table' etc.. for element derived)
  *@return tag element
*/
function vmp_wrap_createElement(tagname,type='')
{
  if(type === '')
    return document.createElement(tagname);
  else
    return document.createElement(type,{is:tagname});
}
/**
  *Wrapper of the javascript function customElements.define that creates custom tag
  *@param modname {String} module name(the name must have a hyphen in the middle)
  *@param modtype class type element
  *@param type {String}tag type(div,table etc.. derivative element)
*/
function vmp_wrap_define(modname,modtype,type='')
{
  try
  {
    if(type === '')
      customElements.define(modname,modtype);
    else
      customElements.define(modname,modtype,{extends:type});
  }
  catch(error)
  {
    vmp_except("vmp_define(name='" + modname + "') " + error);
  }  
}
/**
  *Wrapper of the javascript function document.createTextNode that creates text tag
  *@param message {String} message text
  *@return tag text
*/
function vmp_wrap_createTextNode(message){return document.createTextNode(message);}
/**
  *Wrapper of the javascript function alert that creates alert message
  *@param message {String} message text
*/
function vmp_wrap_alert(message){alert(message);}
/**
  *Wrapper of the javascript function prompt that creates prompt message
  *@param message {String} message text display
  *@param defvalue {String} defvalue default value
  *@return If the user clicks "OK", the input value is returned otherwise null is returned.

*/
function vmp_wrap_prompt(message,defvalue){return prompt(message,defvalue);}
/**
  *Wrapper of the javascript function confirm that creates prompt message
  *@param message {question} question display
  *@return true if the user clicked OK, otherwise false.
*/
function vmp_wrap_confirm(question){return confirm(question);}
/**
  *Wrapper of the javascript for create Map object
  *@return Map object.
*/
function vmp_wrap_map(){return new Map();}
/**
  *Wrapper of the javascript for create Set object
  *@return Set object.
*/
function vmp_wrap_set(){return new Set();}
/**
  *Wrapper of the javascript for create WeakMap object
  *@return WeakMap object.
*/
function vmp_wrap_weakMap(){return new WeakMap();}
/**
  *Wrapper of the javascript for create WeakMap object
  *@return WeakMap object.
*/
function vmp_wrap_weakSet(){return new WeakSet();}
/**
  *Wrapper of the javascript JSON.parse for constructing the JavaScript value or object described by the string.
  *@param text {String} the string to parse as JSON.
  *@throws in case of failure
  *@return The Object, Array, string, number, boolean, or null value corresponding to the given JSON text
*/
function vmp_wrap_json_parse(text){return JSON.parse(text);}
/**
  *Wrapper of the javascript JSON.stringify for converts a JavaScript value to a JSON string.
  *@param data {String} The value to convert to a JSON string.
  *@return {String} json string
*/
function vmp_wrap_json_stringify(data){return JSON.stringify(data);}
/**
  *Wrapper of the javascript btoa for create encodes a string in base-64.
  *@param text {String} input string.
  *@return {String} the base-64 encoded string
*/
function vmp_wrap_base64_encode(text){return btoa(text);}
/**
  *Wrapper of the javascript atob for create method decodes a base-64 encoded string
  *@param enc {String} the string to be decoded.
  *@return {String} the decoded string.
*/
function vmp_wrap_base64_decode(enc){return atob(enc);}
/**
  *Wrapper of the javascript text encoder
  *@param etype {String} encoder type(ex "utf-8").
  *@return {String} text enoder object.
*/
function vmp_wrap_TextDecoder(etype){return new TextDecoder(etype);}
/**
  *@param n {number} Max n accepted
  *@return returns a random number between 0(inclusive),and n(exclusive)
*/
function vmp_wrap_random(n)
{
  return Math.floor(Math.random()*n);
}
/** 
  *Converts a string of exadecimal values in the byte vector (2 characters 0-F 1 byte)
  *@param xstr string input
  *@throws throws exception
  *@return byte vector.
*/
function vmp_xstr_tobytes(xstr)
{
    let ret=new Array();
    let tmp;
    let b=0x00;
    if((xstr.length % 2) == 1)
      tmp=`0${xstr}`;
    else
      tmp=xstr;
    for(let i=0;i<tmp.length;i++)
    {
      switch(tmp[i])
      {
        case '0': b|=0x00;break;
        case '1': b|=0x01;break;
        case '2': b|=0x02;break;
        case '3': b|=0x03;break;
        case '4': b|=0x04;break;
        case '5': b|=0x05;break;
        case '6': b|=0x06;break;
        case '7': b|=0x07;break;
        case '8': b|=0x08;break;
        case '9': b|=0x09;break;
        case 'a': b|=0x0A;break;
        case 'A': b|=0x0A;break;
        case 'b': b|=0x0B;break;
        case 'B': b|=0x0B;break;
        case 'c': b|=0x0C;break;
        case 'C': b|=0x0C;break;
        case 'd': b|=0x0D;break;
        case 'D': b|=0x0D;break;
        case 'e': b|=0x0E;break;
        case 'E': b|=0x0E;break;
        case 'f': b|=0x0F;break;
        case 'F': b|=0x0F;break;
        default:vmp_except(`String ${xstr} is not hexadecimal`);
       }
       if((i%2) == 0)
         b=b<<4;
       else
       {
         ret.push(b);
         b=0x00;
       }    
    }
    return ret;
}
/** 
  *Converts a bytes vector to a string of hexadecimal values
  *@param bytes byte vector input
  *@return The converted string value.
*/
function vmp_bytes_toxstr(bytes)
{
  let ret="",tmp;
  for(let i=0;i<bytes.length;i++)
  {
    tmp=bytes[i].toString(16);
    if(tmp.length == 1)ret+=`0${tmp}`
    else ret+=tmp;
  }
  return ret;
}
/**
  *Converts a bytes vector to a string of hexadecimal values in human mode(the reverse function does not work)
  *@param bytes byte vector input
  *@param delimiter byte delimiter
  *@return The converted string value in human mode.
*/
function vmp_bytes_toxstr_hm(bytes,delimiter)
{
  let ret="",tmp;
  for(let i=0;i<bytes.length;i++)
  {
    tmp=bytes[i].toString(16);
    if(tmp.length == 1)tmp=`0${tmp}`;
    if(i == 0)
      ret+=tmp;
    else
      ret+=`${delimiter}${tmp}`;
  }
  return ret;
}
/**
  *Check if the input is an integer
  *@param n input verify
  *@param min {Number} Minimum value of the range
  *@param max {Number} Maxinum value of the range
  *.@return true if n is numeric between min and max,otherwise false
*/
function vmp_isint(n,min=Number.MIN,max=Number.MAX)
{
  if (!(typeof n === "number") || (n % 1 != 0) || n < min || n > max) return false;
  return true;
}
/**
  *Check if the input is a number
  *@param n input verify
  *@param min {Number} Minimum value of the range
  *@param max {Number} Maxinum value of the range
  *.@return true if n is number between min and max,otherwise false
*/
function vmp_isnumber(n,min=Number.MIN,max=Number.MAX)
{
  if (!(typeof n === "number") || n < min || n > max) return false;
  return true;
}
/**
  *Check if the input is a bool
  *@param nb input verify
  *.@return true if n is numeric between min and max,otherwise false
*/
function vmp_isbool(b)
{
  return (typeof b === "boolean");
}
/**
  *Check if the input is a string
  *@param str input verify
  *@return true if str is string,otherwise false
*/
function vmp_isstring(str){return (typeof str === "string");}
/**
  *Check if the input is an array
  *@param arr input verify
  *@return true if arr is array,otherwise false
*/
function vmp_isarray(arr){return (Array.isArray(arr));}
/**
  *Check if the input is a simplex object ({})
  *@param obj input verify
  *@return true if obj is simplex object,otherwise false
*/
function vmp_isobj(obj){return (Object.entries(obj).length != 0);}
/**Internal usage*/
const VMP_BUFSIZEMAX=Number.MAX_VALUE-1;
/**Internal usage*/
const VMP_BUF_SIZE_INIT=32;
/**Internal usage*/
const VMP_BUF_EOB_ERROR="End Of Buf Error";

/**Buffer type */
class Vmp_Buf
{
  /**
    *Create an buffer.
    *@constructor
  */
  constructor()
  {
    this.maxsize_=VMP_BUF_SIZE_INIT;
    this.mem_=new ArrayBuffer(this.maxsize_);
    this.reset();
  }
  /**Internal usage*/
  expand(size)
  {
    if(size > this.maxsize_)
    {
      do
      {
        if(this.maxsize_ > VMP_BUFSIZEMAX/2)this.maxsize_=VMP_BUFSIZEMAX;
        else this.maxsize_ *= 2;  
      }
      while(size > this.maxsize_);
      let old=this.pointer();
      this.mem_=new ArrayBuffer(this.maxsize_);
      let mem=this.pointer();
      for(let i=0;i<this.size_;i++)mem[i]=old[i];  
    }
    if(size > this.size_)
      this.size_=size;
  }
  /** 
    *New size buffer
    *@param size {number} buffer size                        
  */
  newsize(size)
  {
    if(size > this.size_)        
      this.expand(size);
    else if(size < this.size_)
    {        
      let view=this.pointer(size);
      for(let i=0;i<view.length;i++)view[i]=0x00;
      this.size_=size;
    }
  }
  /** 
    *Size buffer
    @return {number} buffer size                        
  */        
  size(){return this.size_;}
  /** 
    *@return {number}the number of bytes available for reading.(size()-index())  
  */
  size_reading(){return (size_-index_);}
  /**
    *Set index position
    *@param point {number} index pointer to setting. If point > size_ setting index_=size_,if point < 0 setting index_=0
  */
  index(point=0){this.index_=(point<this.size_)?point:this.size_;}
  /**
    *@return {number} current index(End of Buf index == size)
  */
  get_index(){return this.index_}
  /**
    *@return {bool} True if end of buf occurred
  */
  eob(){return (this.index_===this.size_)?true:false;}
  /**reset buffer*/
  reset()
  {
    let mem=this.pointer();
    for(let i=0;i<this.size_;i++)
      mem[i]=0x00;
    this.size_=0;
    this.index(); 
  }
  /**
    *Get array buffer pointer memory
    *@param point first byte pointed
    *@return {Uint8Array} an view Uint8Array
  */
  pointer(point=0){return (point<this.size_)?new Uint8Array(this.mem_,point,this.size_-point):null;}
  /**
    *Check if the buffer contents are equal
    *@param ibuf {Vmp_Buf} input buffer
    *@return true if the buffer contents are equal,otherwise false
  */
  equal(ibuf)
  {
    if(this.size_ === ibuf.size())
    {  
      let b=this.pointer(),bi=ibuf.pointer();
      for(let i=0;i<this.size_;i++)
        if(b[i] != bi[i])
          return false;
    }    
    return true;
  }
  /**
    *Writes the input byte array to the buffer at the this.index_ position and increments the this.index_
    *@param uint8array {Uint8Array} input array
    *@return {number} bytes written in the buffer
  */
  write(uint8array)
  {
    let size=uint8array.length;
    let bmax=VMP_BUFSIZEMAX-this.index_;
    if(bmax < size)
      size=bmax;
    let nindex=this.index_+size
    this.expand(nindex);
    let mem=this.pointer(this.index_);
    for(let i=0;i<size;i++)mem[i]=uint8array[i];
    this.index(nindex);
    return size;
  }
  /**
    *Reads size bytes from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {number} number of bytes to read    
    *@return {Uint8Array} byte reading
  */ 
  read(size)
  {
    let nindex=this.index_+size;
    if(nindex > this.size_)
      vmp_except(`${VMP_BUF_EOB_ERROR} read buf size ${size}`);
    let ret=new Uint8Array(this.mem_,this.index_,size);
    this.index(nindex);
    return ret;
  }
  /** 
    *Write input byte to buffer to the this.index_ position and increments the this.index_ value
    *@param value {bytes} input bytes
    *@return {number} bytes written in the buffer
  */                
  write_byte(byte)
  {
    let arr=new Array(1);
    arr[0]=byte;
    return this.write(new Uint8Array(arr));
  }
  /**
    *Reads size byte from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {number} number of bytes to read    
    *@return {byte} byte reading
  */ 
  read_byte()
  {
    let arr=this.read(1);
    return arr[0];
  }
  /** 
    *Write input array byte to buffer to the this.index_ position and increments the this.index_ value
    *@param value {bytes} input bytes
    *@return {number} bytes written in the buffer
  */                
  write_array(bytes){return this.write(new Uint8Array(bytes));}
  /**
    *Reads size byte from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {number} number of bytes to read    
    *@return {array[bytes]} bytes reading
  */ 
  read_array(size)
  {
    let arr=this.read(size);
    let ret=new Array();
    for(let byte of arr)
      ret.push(byte);
    return ret;
  }
  /**
    *Write arraybuffer content to buffer input to the this.index_ position and increments the this.index_ value
    *@param abuf {arrayBuffer} input arrayBuffer
    *@return {number} bytes reading
  */
  write_arraybuffer(abuf){return this.write(new Uint8Array(abuf));}
  /**
    *@return Return {arrayBuffer} a data buffer array from this.index_ size position and increments the this.index_ value
  */
  read_arraybuffer(size)
  {
    let arr=this.read(size);
    let ret=new ArrayBuffer(size);
    let v=new Uint8Array(ret);
    for(let i=0;i<arr.length;i++)
      v[i]=arr[i]
    return ret;
  }
  /** 
    *Write n random bytes in the buffer to the this.index_ position and increments the this.index_ value
    *@param n {number} bytes number
    *@return {number} the number of bytes written in the buffer 
  */
  random_bytes(n)
  {
    let arr=new Array();
    for(let i=0;i<n;i++)
      arr.push(vmp_wrap_random(256));
    return this.write(new Uint8Array(arr));
  }
  /**
    *Write size value in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {number} input size
    *@param dim number of bytes to write input(1-4)[b8,b7,b6,b5,b4,b3,b2,b1]
    *@return {number} the number of bytes written in the buffer
  */   
  write_size(value,dim)
  {
    let arr=new Array();
    if(dim < 0)
      dim =0;
    else if(dim > 8)
      dim=8;
    for(let i=0;i<dim;i++)
      arr.push((value >> ((dim-i-1)*8)) & 0xFF);
    return this.write(new Uint8Array(arr)); 
  }
  /**
    *Read size value from the buffer in the this.index_ position and increments the this.index_ value.
    *@param dim number of bytes to read(1-4)b8,b7,b6,b5,b4,b3,b2,b1]
    *@return {number} size reading
  */ 
  read_size(dim)
  {
    let ret=0;
    if(dim == 0)
      return ret;
    else if(dim > 8)
      dim=8;
    let arr=this.read(dim);
    for(let i=0;i<dim;i++)
      ret+=(arr[i]<<((dim-i-1)*8));
    return ret;
  }
  /**
    *Write string values in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {string} input string
    *@return {number} the number of bytes written in the buffer
  */                  
  write_str(value){return this.write(new Uint8Array(new TextEncoder().encode(value)));}
  /**
    *Read string values from the buffer in the this.index_ position and increments the this.index_ value
    *@param {number} size to read
    *@return {string} string value
  */  
  read_str(size){return new TextDecoder().decode(this.read(size));}
  /**
    *Write string of exadecimal values in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {exadecimal} input string
    *@return the number of bytes written in the buffer
  */                  
  write_xstr(value){return this.write(new Uint8Array(vmp_xstr_tobytes(value)));}
  /**
    *Read string of exadecimal values from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {number} size to read
    *@return {string exadecimal} string value
  */
  read_xstr(size){return vmp_bytes_toxstr(this.read(size));}
  /**
    *Read string of exadecimal in human mode values from the buffer in the index_ position and increments the index value.(The reverse function does not work)
    *@param size {number} size to read
    *@param delimiter {string} byte delimiter
    @return {string exadecimal} string value
  */
  read_xstr_hm(size,delimiter){return vmp_bytes_toxstr_hm(this.read(size),delimiter);}
  /**
    *Concatenating the contents of the input buffer to the end of the buffer.               
    *@param value {Vmp_Buf} input buffer
    *@return number the number of bytes written in the buffer
  */
  cat(value)
  {
    this.index(this.size_);
    let ret=this.write(value.pointer());
    this.index();
    return ret;
  }
  /**
    *Concatenating the contents of the input buffer atRegister a module in the project
 the beginning of the buffer.             
    *@param value {Vmp_Buf} input buffer
    *@return {number} the number of bytes written in the buffer
  */
  cathead(value)
  {
    let arr=new Array();
    for(let o of this.pointer())
      arr.push(o);
    this.reset();
    this.write(value.pointer());
    this.write(arr);
    this.index();
  }
  /**
    *Write buffer value in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {Vmp_Buf} input buffer
    *@return {number} the number of bytes written in the buffer
  */
  write_buf(value){return this.write(value.pointer());}
  /**
    *Read buffer value from the buffer in the this.index_ position and increments the this.index_ value
    *@param size number of bytes to read
    *return {Vmp_Buf} output buffer    
  */ 
  read_buf(size)
  {
    let ret=new Vmp_Buf();
    let tmp=this.read(size);
    ret.write(tmp);
    return ret;
  }
  /**
    *Resizes the buffer set this.index_ position 0
    *@param init first byte
    *@param len new buffer size
  */                
  cut(init,len)
  {
    let mtmp=new Uint8Array(this.mem_,init,len);
    let arr=new Array();
    for(let o of mtmp)
      arr.push(o);
    this.reset();
    return this.write(new Uint8Array(arr));
  }
};
/*
  *Class representing a timeout operation.
  *@class
  *@public
*/
class Vmp_Timeout
{
  /**
    *Create an timer.
    *@constructor
  */
  constructor(){this.id_=null;}
  /**
    *Start a timeout 
    *@param func {callback} the function to execute if the timeout is activated
    *@param timeout {number} timeout number
    *@param args function parameters
  */
  start(func,timeout,...args){this.stop();this.id_=setTimeout(func,timeout,args);}
  /**Stop the timeout*/
  stop(){if(this.id_!=null){clearTimeout(this.id_);this.id_=null;}}
};
/**
  *@return {string} author project
*/
function vmp_env_author(){return VAUTHOR;}
/**
  *@return {string} email author project
*/
function vmp_env_email(){return VEMAIL;}
/**
  *@return {string} web page project
*/
function vmp_env_webpage(){return VWEBPAGE;}
/**
  *@return {String} license project
*/
function vmp_env_license(){return VLICENSE;}
/**
  *@return {String} version project
*/
function vmp_env_version(){return VVERSION;}
/**
  *@param tag input tag
  *@return {Array} peer component from the tag
*/
function vmp_tag_peers(tag){return (tag.parentElement != null)?tag.children:null;}
/**
  *@param tag input tag
  *@return {Array} parents elements[parent,parent.parent etc..] from the tag
*/
function vmp_tag_parents(tag)
{
  let ret=[]
  let parent=tag.parentElement
  while(parent != null)
  {
    ret.push(parent);
    parent=tag.parentElement
  }
  return ret;
}
/**
  *Insert an item in the index position in the tag
  *@param tag input tag
  *@param child element to be inserted
  *@param index {number} index position
  *@return the appended node 
*/
function vmp_tag_childInsert(tag,child,index)
{
  let e=tag.children[index];
  return (e != null)?e.before(child):tag.appendChild(child);
}
/**
  *Replace an item in the index position,if index is out range it append at the end in the tag
  *@param tag input tag
  *@param child element to be inserted
  *@param index {number} index position
  *@return the replaced node 
*/
function vmp_tag_childReplace(tag,child,index)
{
  let e=tag.children[index];
  return (e != null)?e.replaceWith(child):tag.appendChild(child);
}
/**
  *Remove an item in the index position in the tag
  *@param tag input tag
  *@param index {number} index position
*/
function vmp_tag_childRemove(tag,index)
{
  let e=tag.children[index];
  if(e != null){e.remove()};
}
/**
  *Removes the first item in the tag
  *@param tag input tag
*/
function vmp_tag_childRemove(tag)
{
  if(tag.firstElementChild != null)
    tag.firstElementChild.remove();
}
/**
  *removes the last item in the tag
  *@name vmp_component#childRemoveBack
  *@public
  *@function   
*/
function vmp_tag_childRemoveBack(tag)
{
  if(tag.lastElementChild != null)
    tag.lastElementChild.remove();
}
/**
  *Remove all childs in the tag
  *@param tag input tag
*/
function vmp_tag_childRemoveAll(tag)
{
  while(tag.firstElementChild)
    tag.firstElementChild.remove();
}
/**
  *decorator who transforms a tag into a component
  *@param element tag element type
  *@return tag element 
*/
function vmp_component(element)
{
  /**
    *@name vmp_component#cmpid_
    *@property {String} component id
    *@public
  **/
  element.prototype.cmpid_='';
  /**
    *@name vmp_component#forward_
    *@property {String} forward address of requests associated with this component
    *@public
  **/
  element.prototype.forward_='';
  /**
    *@name vmp_component#ismodule_
    *@property {bool} Is it a module component?
    *@public
  **/
  element.prototype.ismodule_=false;
  /**
    *@name vmp_component#module_
    *@property {String} Component module,if null does not belong to any module
    *@public
  **/
  element.prototype.module_=null;
  /**
    *Initializes the component
    *@param id {String} component id
    *@param module {Vmp_Module} module where the component is inserted
  */
  element.prototype.init=function(id,module=null){this.cmpid_=id;this.module_=module;}
  /**
    *@name vmp_component#cmpid
    *@function
    *@public 
    *@return {String} id component remote node to which the module is associated
  */
  element.prototype.cmpid=function(){return (this.module_ === null)?this.cmpid_:this.module_.cmpid_;}
   /**
    *@name vmp_component#forward
    *@function
    *@public 
    *@return {String} forward address of requests associated with this component
  */
  element.prototype.forward=function()
  {
     if(this.forward_ == '')
       return (this.module_ === null)?'':this.module_.forward_;
     return forward_;
  }
  /**
    *@name vmp_component#peers
    *@function
    *@public 
    *@return {Array} peer component
  */
  element.prototype.peers = function(){return vmp_tag_peers(this);}
  /**
    *@name vmp_component#parents
    *@function
    *@public 
    *@return {Array} parents elements[parent,parent.parent etc..] 
  */
  element.prototype.parents=function(){return vmp_tag_parents(this);}
  /**
    *Insert an item in the index position
    *@name vmp_component#childInsert
    *@function
    *@public 
    *@param child element to be inserted
    *@param index {number} index position
    *@return the appended node 
  */
  element.prototype.childInsert=function(child,index){return vmp_tag_childInsert(this,child,index);}
  /**
    *Replace an item in the index position,if index is out range it append at the end
    *@name vmp_component#childReplace
    *@function
    *@public 
    *@param child element to be inserted
    *@param index {number} index position
    *@return the replaced node 
  */
  element.prototype.childReplace=function(child,index){return vmp_tag_childReplace(this,child,index);}
  /*
    *Remove an item in the index position
    *@name vmp_component#childRemove
    *@function
    *@public 
    *@param index {number} index position
  */
  element.prototype.childRemove=function(index){return vmp_tag_childRemove(this,index);}
  /**
    *Removes the first item
    *@name vmp_component#childRemovePop
    *@public
    *@function 
  */
  element.prototype.childRemovePop=function(){return vmp_tag_childRemovePop(this);}
  /**
    *removes the last item
    *@name vmp_component#childRemoveBack
    *@public
    *@function   
  */
    
  element.prototype.childRemoveBack=function(){return vmp_tag_childRemoveBack(this);}
  /**
    *Remove all childs
    *@name vmp_component#childRemoveAll
    *@function
    *@public 
  */
  element.prototype.childRemoveAll=function(){return vmp_tag_childRemoveAll(this);}
  /**
    *Send an http get
    *@name vmp_component#getremote node to which the module is associated
    *@function
    *@public
    *@param url {string} url address
    *@param id  operation id
    *@param timeout {number} timeout operation
  */
  element.prototype.get=function(url,id,timeout=5000)
  {
    let xhr = new XMLHttpRequest();
    let cmp=this;
    xhr.open("GET",url, true);
    xhr.timeout = timeout;
    xhr.onload = (e) => {
      if (xhr.readyState === 4) {cmp.get_recv(id,xhr.status,xhr.responseText);}
    };
    xhr.ontimeout = (e) => {cmp.get_recv(408,"Request Timeout");};
    xhr.onerror = (e) =>{cmp.get_recv(id,xhr.status,xhr.statusText);};
    xhr.send(null);
  }
  /**
    *Virtual function for recv get operation
    *@name vmp_component#get_recv
    *@public
    *@function 
    *@param id  operation id
    *@param status {number} operation status
    *@param text {string} operation text response
  */
  element.prototype.get_recv=function(id,status,text){}
  /**
    *Send an http postremote node to which the module is associated
    *@name vmp_component#post
    *@function 
    *@param url {string} url address
    *@param id  operation id
    *@param ctype {string} content-type header data
    *@param data {string}  input data to send
    *@param timeout {number} timeout operation
  */
  element.prototype.post=function(url,id,ctype,data,timeout=5000)
  {
    let xhr = new XMLHttpRequest();
    let cmp=this;
    xhr.open("POST",url, true);
    xhr.timeout = timoout;
    xhr.onload = (e) => {
      if (xhr.readyState === 4){cmp.post_recv(id,xhr.status,xhr.responseText);}
    };
    xhr.ontimeout = (e) => {cmp.post_recv(id,408,"Request Timeout");};
    xhr.onerror = (e) => {cmp.post_recv(id,xhr.status,xhr.statusText);};
    xhr.setRequestHeader('Content-type',ctype);
    xhr.send(data);
  }
  /**
    *Virtual function for recv post operation
    *@name vmp_component#post_recv
    *@public
    *@function 
    *@param id  operation id
    *@param status {number} operation status
    *@param text {string} operation text response
  */
  element.prototype.post_recv=function(id,status,text){}
  /** 
    *Virtual function for load component
    *@name vmp_component#load
    *@public
    *@function 
  */
  element.prototype.load=function(){}
  /** 
    *Virtual function for unload component
    *@name vmp_component#unload
    *@public
    *@function
  */
  element.prototype.unload=function(){}
  /**
    *Virtual function that receives a message generic
    *@name vmp_component#msg_recv
    *@param jdata {Vmp_JData} json data
    *@param buf {Vmp_Buf} Buffer data
    *@public
    *@function
  */
  element.prototype.msg_recv=function(jdata,buf){}
  /**
    *Virtual function that receives a response from an associated request
    *@name vmp_component#req_recv
    *@param req request associated
    *@param jdata {Vmp_JData} json data
    *@param payload {Vmp_Buf} buffer data
    *@public
    *@function
  */
  element.prototype.req_recv=function(req,jdata,payload){}
  /**
    *Virtual function that close request
    *@name vmp_component#req_close
    *@param req request associated
    *@public
    *@function
  */
  element.prototype.req_close=function(req){}
  return element;
}
/**Module Component*/ 
class Vmp_Module extends vmp_component(HTMLDivElement)
{
  /**constructor*/
  constructor(){super();this.ismodule_=true;}
  
  /**virtual function*/
  init(id,module=null){this.cmpid_=id;}
}
/**Module tags*/
window.modules_=vmp_wrap_map();
/**
  *Register a module in the project
  *@param nodetype {String} remote node to which the module is associated
  *@param modname {String} module name(the name must have a hyphen in the middle)
  *@param module module class
  *@param type {string}tag type(div,table etc.. derivative element)
*/
function vmp_module_register(nodetype,modname,module)
{
  vmp_wrap_define(modname,module,'div');
  window.modules_.set(nodetype,modname);
}
/**
  *Load a module based on the nodetype
  *@param nodetype {String} remote node to which the module is associated
  *@return module element or undefined in case of failure
*/
function vmp_module_load(nodetype)
{
  let modname=window.modules_.get(nodetype);
  return (modname != undefined)?vmp_wrap_createElement(modname,'div'):undefined;
}
/** 
  *@constant
  *@typedef {Object} Vmp_WebSocketCode Manages the closing codes of a websocket
  *@default
  *@property {number} Normal the connection close
  *@property {number} Shutdown Server is shutting down
  *@property {number} Protocol Some error in the protocol has happened 
  *@property {number} Type The type(text, binary) was not supported
  *@property {number} Abnormal Abnormal local close
  *@property {number} Utf8 The message wasn't in UTF8
  *@property {number} Policy The policy of the server has been broken
  *@property {number} Big The messages received is too big
  *@property {number} Extension Mandatory extension missing
  *@property {number} Unexpected Unexpected happened
*/
const Vmp_WebSocketCode={
  Normal:1000,
  Shutdown:1001,
  Protocol:1002,
  Type:1003,
  Abnormal:1006,
  Utf8:1007,
  Policy:1008,
  Big:1009,
  Extension:1010,
  Unexpected:1011,
  /**
    *@param code {number} input code
    *@return {string} string reason
  */
  reason(code)
  {
    switch(code)
    {
      case this.Normal:return "Connection close";
      case this.Shutdown:return "Server is shutting down ";
      case this.Protocol:return "Some error in the protocol has happened"; 
      case this.Type:return "type was not supported";
      case this.Abnormal:return "Abnormal local close";
      case this.Utf8:return "The message wasn't in UTF8";
      case this.Policy:return "The policy of the server has been broken";
      case this.Big:  return "The messages received is too big";
      case this.Extension:return "Mandatory extension missing";
      case this.Unexpected:return "Unexpected happened";
      default:return "Unknown";
    }
  }
}
/**Utility class for managing json data*/
class Vmp_JData
{
  /**Constructor*/
  constructor()
  {
    /**
     *Object references
     *@type {Object}
     *@public
    */
    this.obj_=null
  }
  /**
    *create a new json data
    *@param jtype {string} json data type
  */
  new_data(jtype){this.obj_={jtype:jtype,jbody:{},}}
  /**
    *set a json given by a json string
    *@param obj {string | Object jdata} text input
  */
  set_data(obj)
  {
    try
    {
      if(vmp_isstring(obj))
        this.obj_=vmp_wrap_json_parse(obj);
      else
        this.obj_=obj;
      if(!(("jtype" in this.obj_) && ("jbody" in this.obj_) && Object.keys(this.obj_) != 2))vmp_except("");
    }
    catch(error){this.obj_=null;vmp_except(error);}
  }
  /**@return json data type*/
  jtype(){return this.obj_.jtype;}
  /**@return json data body*/
  jbody(){return this.obj_.jbody;}
  /**
    *set body object
    *@param obj {Object} body objct
  */
  setjbody(obj){this.obj_.jbody=obj;}
  /**
    *stringify of the jdata object
    *@return {String} json string
  */
  stringify(){return vmp_wrap_json_stringify(this.obj_);}
};

