/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.table lines associated with keys
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 31/08/2024
 */
 /**Json request protocol status code*/
const Vmp_Jrp_StatusCode={
  ok:0, /*!< Used by the jrp protocol for message states 'Success'*/
  err:-1,/*!< Used by the jrp protocol for message states 'Generic Error'*/
  malformed_msg:-2,/*!< Used by the jrp protocol for message states 'Malformed message'*/
  undef_datatype:-3,/*!< Used by the jrp protocol for message states 'Undef datatype in message'*/
  accessdenied:-4,/*!< Used by the jrp protocol for message states 'Access Denied'*/
  duplexsession:-5,/*!< Used by the jrp protocol for message states 'Duplex session found'*/
  protocolbad:-6,/*!< Used by the jrp protocol for message states 'Protocol bad sequence'*/
  closeconnection:-7,/*!< Used by the jrp protocol for message states 'Connection close'*/
  timeout:-8,/*!< Used by the jrp protocol for message states 'Connection timeout'*/
  killed:-9,/*!< Used by the jrp protocol for message states 'Request killed'*/
  input_bad:-10,/*!< Used by the jrp protocol for message states 'Bad input recv'*/
  push_bad:-11,/*!< Used by the jrp protocol for message states 'Bad push recv'*/
  input_notmanaged:-12,/*!< Used by the jrp protocol for message states 'Unmanaged datatype input'*/
  rid_duplex:-13,/*!< Used by the jrp protocol fUsed in session_get to read remote peer permissionsor message states 'Duplex request id'*/
  resource_busy:-14,/*!<User by jrp protocol for message states 'Resource busy'*/
  resource_accessdenied:-15,/*!<User by jrp protocol for message states 'Resource Access Denied'*/
  kill_notcomplete:-16,/*!<User by jrp protocol for message states 'kill incomplete operation'*/
  reason(code)
  {
    switch(code)
    {
      case this.ok:return "Success";
      case this.err:return "Generic Error";
      case this.malformed_msg:return "Malformed message";
      case this.undef_datatype:return "Undef datatyp mypermits_e in message";
      case this.accessdenied:return "Access Denied";
      case this.duplexsession:return "Duplex session found";
      case this.protocolbad:return "Protocol bad sequence";
      case this.closeconnection:return "Connection close";
      case this.timeout:return "Connection timeout";
      case this.killed:return "Request killed";
      case this.input_bad:return "Bad input recv";
      case this.push_bad:return "Bad push recv";
      case this.inpducut_notmanaged:return "Unmanaged datatype input";
      case this.rid_duplex:return "Duplex request id";
      case this.resource_busy:return "Resource Busy";
      case this.resource_accessdenied:return "Resource Access Denied";
      case this.kill_notcomplete:return "Kill incomplete operation";
      default:return "Undef error";
    }
  }
};
/**
 *composes a jrp message in string format
 *@param msg {Object} jrp message
 *@param payload {Vmp_Buf} buffer payload
 *@return buffer {Vmp_Buf} buffer messagetable lines associated with keys
*/
function vmp_jrp_write(msg,payload=undefined)
{
  let buf=new Vmp_Buf();
  let str=vmp_wrap_json_stringify(msg);
  buf.write_size(str.length,4);
  buf.write_str(str);
  if(payload != undefined && payload.size != 0)
    buf.write_buf(payload);
  buf.index();
  return buf;
}
/**
  *parse jrp message
  *@param buf {Vmp_Buf} input message
  *@return {Array} if ok [Vmp_Jrp_StatusCode.status_ok,jrp {Object},payload{Vmp_Buf}],otherwise [Vmp_Jrp_StatusCode.(error code)]
*/
function vmp_jrp_parse(buf)
{
  let ret;
  try
  {
    let size=buf.read_size(4);
    let data=buf.read_str(size);
    let jrp=vmp_wrap_json_parse(data);
    let payload=new Vmp_Buf();
    switch(jrp.msgtype)
    {
      case 'session':
        if(Object.keys(jrp).length != 6)vmp_except("");
        if(!vmp_isint(jrp.permits,0))vmp_except("");
        if(!vmp_isobj(jrp.reqdata))vmp_except("");
        if(!vmp_isstring(jrp.nodetype))vmp_except("");
        if(!vmp_isarray(jrp.broadcast))vmp_except("");
        return[Vmp_Jrp_StatusCode.status_ok,jrp,payload];
      case 'response':
        if(Object.keys(jrp).length != 4)vmp_except("");
        if(!vmp_isint(jrp.rid,min=0))vmp_except("");
        if(!vmp_isint(jrp.payload,0))vmp_except("");
        if(vmp_isarray(jrp.outputs))
        {
          if(jrp.payload != 0)vmp_except("");
          for(obj of jrp.outputs)
            if(!vmp_isobj(obj))vmp_except("");
        }
        else if(!vmp_isobj(jrp.outputs))vmp_except("");
        if(jrp.payload > 0)
        {
            if(data.length != size+jrp.payload)vmp_except("");
              payload.write_arraybuffer(buf.read_arraybuffer(jrp.payload));
            payload.index();
        }
        return[Vmp_Jrp_StatusCode.status_ok,jrp,payload];
        break;
      case 'broadcast':
        if(Object.keys(jrp).length != 3)vmp_except("");
        if(!vmp_isint(jrp.payload,0))vmp_except("");
        if(vmp_isarray(jrp.outputs))
        {
          if(jrp.payload != 0)vmp_except("");
          for(obj of jrp.outputs)
            if(!vmp_isobj(obj))vmp_except("");
        }
        else if(!vmp_isobj(jrp.outputs))vmp_except("");
        if(jrp.payload > 0)
        if(jrp.payload > 0)
        {
            if(data.length != size+jrp.payload)vmp_except("");
              payload.write_arraybuffer(buf.read_arraybuffer(jrp.payload));
            payload.index();
        }
        return[Vmp_Jrp_StatusCode.status_ok,jrp,payload];
      case 'close':
        if(Object.keys(jrp).length != 4)vmp_except("");
        if(!vmp_isint(jrp.rid,0))vmp_except("");
        if(!vmp_isint(jrp.status))vmp_except("");
        if(!vmp_isstring(jrp.msg))vmp_except("");
        return[Vmp_Jrp_StatusCode.status_ok,jrp,payload];
      default:vmp_except("");
    }
  }
  catch(error)
  {
    console.log(`parse error ${error}`);
  }
  return [Vmp_Jrp_StatusCode.malformed_msg];
}
/**
  *Create an abort JRP message
  *@param code {number} exit status code
  *@param msgstr {string} exit message info
*/
function vmp_jrp_abort(code,msgstr="")
{
  if(msgstr === "")
    msgstr=Vmp_Jrp_StatusCode.reason(code);
  let msg={
   msgtype:"abort",
   status:code,
   msg:msgstr,
  }
  return vmp_jrp_write(msg);
}
/**Structure for jrp requests*/
class Vmp_Jrp_Request
{
  /**
    *constructor
    *@param rid {Number} request id
    *@param jdata {Vmp_JData} json data request
    *@param cmp {Component} component associated with request
    *@param manager {Vmp_Jrp_Manager_I} manager associated with request
  */
  constructor(rid,jdata,cmp,manager)
  {
    /**
     *Request id
     *@type {Number}
     *@public
    */
    this.rid_=rid;
    /**
     *json data request
     *@type {Vmp_JData}
     *@public
    */
    this.jdata_=jdata;
    /**
     *request type
     *@type {String}
     *@public
    */
    this.type_=jdata.jtype();
    /**
     *component associated 
     *@type {Component}
     *@public
    */
    this.cmp_=cmp;
    /**
      *manager associated
      *@type {Vmp_Jrp_Manager_I}
      *@public
    **/
    this.manager_=manager;
    /**
      *the request status open(request active),kill(the kill is sending),close(the request is close)
      *@type {string}
      *@public
    **/
    this.status_="open";
    /**
      *the request forward address node,if == '' node not forwarding
      *@type {string}
      *@public
    **/
    this.forward_=this.cmp_.forward();
    /**
      *the request close status
      *@type {number}
      *@public
    **/
    this.close_status_=Vmp_Jrp_StatusCode.ok;
    /**
      *the request close message
      *@type {string}
      *@public
    **/
    this.close_msg_="";
    this.ktimeout_=new Vmp_Timeout();
  }
  /**
    *Returns jreq status
    *@return 1 if request is open,2 if request kill is waiting to close otherwise exit status
  */
  status()
  {
    if(this.status_ == "open")
      return 1;    
    else if(this.status_ == "kill")
      return 2;
    return this.close_status_;
  }
  /**
    *Returns exit messagetable lines associated with keys
    *@return "open" if request is open,2 if request kill is waiting otherwise exit message
  */
  msg()
  {
    if(this.status_ === "close")
        return this.msg_;
    return this.status_;
  }
  /**
    *Send a jrp push message 
    *@throws in case of failure
    *@param outputs {Array(Vmp_JData | Vmp_JData}json data outputs
    *@param payload {Vmp_Buf} json payload data
  */
  push(outputs,payload=undefined)
  {
    if(this.status_ === "open")
    {
      let out=[]
      if(vmp_isarray(outputs))
        outputs.map(item=>out.push(item.obj_));
      else
        out.push(outputs.obj_);
      let psize=(payload == undefined)?0:payload.size();
      if(psize != 0 && out.length != 1)
        vmp_wrap_except("in push msg multi json data with payload present");
      let msg={
        msgtype:"push",
        rid:this.rid_,
        outputs:out,
        payload:psize
      }
      window.main_.send(vmp_jrp_write(msg,payload));
    }
  }
  /**kill timout callback*/
  kill_timeout(){window.main_.abort(Vmp_Jrp_StatusCode.kill_notcomplete);}
  /**Send a jrp kill message*/
  kill()
  {
    if(this.status_ === "open")
    {
      this.status_="kill";
      let msg={
        msgtype:"kill",
        rid:this.rid_,
      }
      window.main_.send(vmp_jrp_write(msg));
      this.manager_.update_request(this);
      this.ktimeout_.start(this.kill_timeout,window.main_.kill_timeout_);
    }
  }
}
/**JRP and session message management class*/
class Vmp_Jrp_Manager_I extends vmp_component(HTMLDivElement)
{
  /**Constructor*/
  constructor()
  {
    super();
    this.table_=vmp_wrap_map();
    this.maxclientreq_=100;
    this.rid_=0;
  }
  /**
    *Call when a session starts
    *@param user {String} user name
    *@param nodetype {String} Remote node type
    *@param permits {Number} User permissions
    *@param bdata {Array} Broadcast type message accepted from Remote Node
    *@param reqdata {Object} Jrp Requests accepted from the Remote Node
  */
  session(user,nodetype,permits,bdata,reqdata)
  {
    this.user_=user;
    this.nodetype_=nodetype;
    this.permits_=permits;
    this.bdata_=bdata;
    this.reqdata_=reqdata;
    this.session_impl();
  }
  /**Call when session ends*/
  session_close()
  {
    this.user_=undefined
    this.nodetype_=undefined
    this.permits_=undefined
    this.reqdata_=undefined
    this.rid_=0;
    this.table_.clear();
    this.session_close_impl();
  }
  /**
   *Send an request
   *@param cmp {Component}component associated
   *@param jdata {Vmp_JData} json data request
   *@param payload {Vmp_Buf} json payload data
   *@return [-1,string error] or [rid(request id)>=0,bufrequest]
  */
  request(cmp,jdata,payload=undefined)
  {
    let req,rid=-1;
    if(this.table_.size === this.maxclientreq_)
    {   
      for(let i=0;i<this.maxclientreq_;i++)
      {
        req=this.table_.get(i)
        if(req.status_==="close"){rid=req.rid_;break;}
      }
      if(rid === -1)
        return [rid,"overflow request"];
    }
    else rid=this.rid_++;
    req=new Vmp_Jrp_Request(rid,jdata,cmp,this);
    this.table_.set(rid,req);
    let psize=(payload === undefined)?0:payload.size();
    let msg={
      msgtype:"request",
      rid:rid,
      input:jdata.obj_,
      forward:req.forward_,
      payload:psize,
    }
    window.main_.send(vmp_jrp_write(msg,payload));
    this.update_request(req);
    return [rid,""];
  }
  /**
    *Receives a jrp response
    *@param rid {Number} request id
    *@param jdata {Vmp_JData} json data
    *@param payload {Vmp_Buf} message payload
  */
  recv_response(rid,jdata,payload)
  {
    let req=this.table_.get(rid);
    if(req != undefined && req.status_==="open")req.cmp_.req_recv(req,jdata,payload);
  }
  /**
    *Close a jrp request
    *@param rid {Number} request id
    *@param status {Number} status close
    *@param msg {String} message close
  */
  recv_close(rid,status,msg="")
  {
    let req=this.table_.get(rid);
    if(req != undefined && !req.isclosed_)
    {
      req.ktimeout_.stop();
      req.status_="close";
      req.close_status_=status;
      req.close_msg_=(msg === "")?Vmp_Jrp_StatusCode.reason(req.status_):msg;
      this.update_request(req);
      req.cmp_.req_close(req);
    }
  }
  /**Virtual function called on session opening*/
  session_impl(){};
  /**Virtual function called on session end*/
  session_close_impl(){};
  /**
    *Virtual function called hen a request has been created or modified, for operations in the derived class
    *@param req {Vmp_Jrp_Request} input request
  */
  update_request(req){}
};

