/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/05/2024
 */

#include "vmp.h"

namespace vampiria { namespace vmp { namespace fs {

void unlink_wrap(vmp::str path)
{
    if(unlink(path.c_str()) == -1)
        vmp::except_errno();
    
}

vmp::str format_dirpath(vmp::str dirpath)
{
    if(dirpath == "")
        return "/";
    if(dirpath[dirpath.size()-1] != '/')
        vmp::unicode::str_cwrite(&dirpath,"/");
    return dirpath;
}

vmp::str union_path(vmp::str dirpath,vmp::str subpath)
{
    vmp::str ret;
    if(subpath.size() != 0 && subpath[0] == '/')
        subpath=vmp::unicode::str_sub(subpath,1,subpath.size()-1);
    vmp::unicode::str_write(&ret,"%s%s",vmp::fs::format_dirpath(dirpath).c_str(),subpath.c_str());
    return ret;
}

void chmod_wrap(vmp::str path,vmp::fs::Mode mode)
{
    if(chmod(path.c_str(),mode) == -1)
        vmp::except_errno();
}

void mkdir_wrap(vmp::str dirpath,vmp::fs::Mode mode)
{
    if(mkdir(dirpath.c_str(),mode) == -1)
       vmp::except_errno();
}

void rmdir_wrap(vmp::str dirpath)
{
    DIR *d = opendir(dirpath.c_str());
    if(d == 0)
        vmp::except_errno(); 
    struct dirent *p;
    while ((p=readdir(d)))
    {
        vmp::str name=p->d_name;
        if(name == "." || name == "..")
            continue;    
        vmp::str bufstr;
        struct stat statbuf;
        vmp::unicode::str_write(&bufstr,"%s%s",vmp::fs::format_dirpath(dirpath).c_str(),name.c_str());
        if(!stat(bufstr.c_str(), &statbuf))
        {
            if(S_ISDIR(statbuf.st_mode))
                vmp::fs::rmdir_wrap(bufstr);
            else
                vmp::fs::unlink_wrap(bufstr);
        }
   }
   closedir(d);
   if(rmdir(dirpath.c_str()) == -1)
       vmp::except_errno();
}

void chdir_wrap(vmp::str path)
{
    if(chdir(path.c_str()) == -1)
        vmp::except_errno();
}

vmp_bool isdir(vmp::str dirpath)
{
    struct stat statbuf;
    if((stat(dirpath.c_str(), &statbuf) == -1) || !S_ISDIR(statbuf.st_mode) ) 
        return false;
    return true;
}

vmp_bool isrdir(vmp::str dirpath)
{
    if(vmp::fs::isdir(dirpath) && (access(dirpath.c_str(),R_OK) == 0))
        return true;
    return false;
}

vmp_bool iswdir(vmp::str dirpath)
{
    if(vmp::fs::isdir(dirpath) && (access(dirpath.c_str(),W_OK) == 0))
        return true;
    return false; 

}

vmp_bool isxdir(vmp::str dirpath)
{
    if(vmp::fs::isdir(dirpath) && (access(dirpath.c_str(),X_OK) == 0))
        return true;
    return false;
}

vmp::str getcwd_wrap()
{
    vmp_char buf[PATH_MAX];
    vmp::bzero_wrap(buf,PATH_MAX);
    if(getcwd(buf,PATH_MAX) == 0)
        vmp::except_errno();
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%s/",buf);
    return ret;
}

vmp::vector<vmp::str> listendir(vmp::str dirpath)
{
    vmp::vector<vmp::str> ret;
    DIR *d = opendir(dirpath.c_str());
    if(d == 0)
        vmp::except_errno();
    struct dirent *p;
    while ((p=readdir(d)))
    {
        vmp::str name=p->d_name;
        if(name == "." || name == "..")
            continue;  
        ret.push_back(name); 
    }
    closedir(d);
    return ret;
}

vmp_bool isfile(vmp::str path)
{
    struct stat statbuf;
    if((stat(path.c_str(), &statbuf) == -1) || !S_ISREG(statbuf.st_mode)) 
       return false;
    return true;
}

vmp_bool isrfile(vmp::str path)
{
    if(vmp::fs::isfile(path) && (access(path.c_str(),R_OK) == 0))
       return true;
    return false;
}

vmp_bool iswfile(vmp::str path)
{
   if(vmp::fs::isfile(path) && (access(path.c_str(),W_OK) == 0))
       return true;
   return false;
}

vmp_bool isxfile(vmp::str path)
{
   if(vmp::fs::isfile(path) && (access(path.c_str(),X_OK) == 0))
       return true;
   return false;
}

vmp::str file_ext(vmp::str path)
{
   vmp_char *c=vmp::strrchr_wrap(path.c_str(),'.');
   return (c == 0)?"":c+1;
}

Mode_t_wrap::Mode_t_wrap()
{
}

Mode_t_wrap::~Mode_t_wrap()
{
}

void Mode_t_wrap::set(mode_t mode)
{
    mode_=mode;
}
        
vmp_bool Mode_t_wrap::isreg()
{
    return S_ISREG(mode_);
}

vmp_bool Mode_t_wrap::isdir()
{
    return S_ISDIR(mode_);
}
        
vmp_bool Mode_t_wrap::ischr()
{
    return S_ISCHR(mode_);
}

vmp_bool Mode_t_wrap::isblk()
{
    return S_ISBLK(mode_);
}

vmp_bool Mode_t_wrap::isfifo()
{
    return S_ISFIFO(mode_);
}
        
vmp_bool Mode_t_wrap::islnk()
{
    return S_ISLNK(mode_);
}

vmp_bool Mode_t_wrap::issock()
{
    return S_ISSOCK(mode_);
}

FileStat::FileStat()
{
    reset();
}

FileStat::FileStat(vmp::str filepath)
{
    set(filepath);
}

FileStat::~FileStat()
{
    reset();
}

void FileStat::reset()
{
    filepath_="";
    vmp::bzero_wrap((void *) &stat_,sizeof(struct stat));
}

void FileStat::set(vmp::str filepath)
{
    reset();
    if(stat(filepath.c_str(),&stat_) == -1)
        vmp::except_errno();
    filepath_=filepath_;
}

vmp::str FileStat::filepath()
{
    return filepath_;
}

void FileStat::dev(vmp::Dev_t_wrap *dev)
{
    vmp::except_check_pointer((void *) dev,"vmp::fs::FileStat::st_dev(dev=null)");
    dev->set(stat_.st_dev);
}

vmp_uint FileStat::ino()
{
    return stat_.st_ino;
}

void FileStat::mode(vmp::fs::Mode_t_wrap *mode)
{
    vmp::except_check_pointer((void *) mode,"vmp::fs::FileStat::st_mode(mode=null)");
    mode->set(stat_.st_mode);
}

vmp_uint FileStat::nlink()
{
    return stat_.st_nlink;
}

vmp_uint FileStat::uid()
{
    return stat_.st_uid;
}

vmp_uint FileStat::gid()
{
    return stat_.st_gid;
}

void FileStat::rdev(vmp::Dev_t_wrap *dev)
{
    vmp::except_check_pointer((void *) dev,"vmp::fs::FileStat::st_rdev(dev=null)");
    dev->set(stat_.st_rdev);
}

vmp_size FileStat::size()
{
    return stat_.st_size;
}
        
vmp_size FileStat::blksize()
{
    return stat_.st_blksize;
}
        
vmp_size FileStat::blocks()
{
    return stat_.st_blocks;
}

vmp_uint FileStat::atime()
{
    return stat_.st_atime;
}
        
vmp_uint FileStat::mtime()
{
    return stat_.st_mtime;
}  
    
vmp_uint FileStat::ctime()
{
    return stat_.st_ctime;
}

}}}

