/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 28/01/2020
 */

#ifndef VAMPIRIA_PYLIB_CLASS_IMPL_H

#define VAMPIRIA_PYLIB_CLASS_IMPL_H 1

namespace vampiria { namespace pylib {

struct instance_cpp
{
    private:
        instance_cpp *next_;
    protected:
        vmp_bool retfree_;
    public:
        instance_cpp();
        virtual ~instance_cpp();
        instance_cpp *next() const;
        virtual void *holds(pylib::type_info) = 0; 
        void install(pylib::Objref *inst,vmp_bool retfree) throw();
        static void* allocate(pylib::Objref *,vmp_size offset,vmp_size size);
        static void deallocate(pylib::Objref* self, void* storage) throw();
};

template <class T>
struct instance_pointer : instance_cpp
{
    typedef T type_;
    
    public:
        instance_pointer(T *p):mp_(std::move(p)){};
        ~instance_pointer() {if(retfree_){delete mp_;}}
    private: // required holder implementation
        void *holds(pylib::type_info dst_t);
        T *mp_;
};

template <class T>
void *instance_pointer<T>::holds(pylib::type_info dst_t)
{
    if(dst_t == pylib::type_id<type_>()) 
        return (void *)mp_;
    return 0;
}

template <class T>
struct instance_value : instance_cpp
{
    typedef T type_;
    
    public:
        instance_value(T v):mv_(std::move(v)){};

    private: // required holder implementation
        void* holds(type_info dst_t);
        T mv_;
   
        
};

template <class T>
void* instance_value<T>::holds(pylib::type_info dst_t)
{
     if(dst_t == pylib::type_id<type_>()) 
        return (void *)&mv_;
     return 0;   
}

template<class DATA=vmp_byte>
struct instance
{
    PyObject_VAR_HEAD
    pylib::Objref *dict_;
    pylib::Objref *weakrefs_; 
    pylib::instance_cpp *objects_;
    
    typedef typename vmp::aligned_storage<vmp::alignment_of<DATA>::value>::type align_t;
    union
    {
         align_t align_;
         vmp_byte bytes_[sizeof(DATA)];
         
    } storage_;
};

pylib::Objref *class_metatype();
pylib::Objref *class_type();
pylib::Objref *class_metatype_eval(vmp::str name,vmp::str prefix);
void *find_instance_impl(pylib::Objref* inst,pylib::type_info type);
pylib::Objref *make_instance(pylib::Objtype *type);

template<class W>
void set_instance(pylib::Objref *obj,W *inst,vmp_bool retfree=true)
{
    typedef pylib::instance_pointer<W> cpp;
    typedef pylib::instance<cpp> instance_t;
    void *memory =cpp::allocate(obj,offsetof(instance_t, storage_),sizeof(cpp));
    try 
    {
        (new (memory) cpp(inst))->install(obj,retfree);
    }
    catch(...) 
    {
        cpp::deallocate(obj,memory);
        throw;
    }    
}

template<class W>
void set_instance(pylib::Objref *obj,W inst,vmp_bool retfree=true)
{
    typedef pylib::instance_value<W> cpp;
    typedef pylib::instance<cpp> instance_t;
    void *memory =cpp::allocate(obj,offsetof(instance_t, storage_),sizeof(cpp));
    try 
    {
        (new (memory) cpp(inst))->install(obj,retfree);
    }
    catch(...) 
    {
        cpp::deallocate(obj,memory);
        throw;
    } 
}

class RegClass
{
     public: // member functions
        pylib::type_info type_;
        pylib::Objtype *p_obj_;
        explicit RegClass(pylib::type_info& t);
        inline vmp_bool operator<(RegClass const& rhs) const;
        inline vmp_bool operator==(RegClass const& rhs) const;
};

inline RegClass::RegClass(pylib::type_info &target_type)
    : type_(target_type)
      ,p_obj_(0)

{}

inline vmp_bool RegClass::operator<(RegClass const& rhs) const
{
    return type_ < rhs.type_;
}

inline vmp_bool RegClass::operator==(RegClass const& rhs) const
{
    return type_ == rhs.type_; 
}

struct Registry
{
    public: 
        static RegClass *lookup(pylib::type_info& t);
        static pylib::Objtype *pytypeobj(pylib::type_info& t); 
};

}}

#endif

