/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 17/09/2024
 */

#ifndef VAMPIRIA_PACKET_WEBSOCKET_EXT_H

#define VAMPIRIA_PACKET_WEBSOCKET_EXT_H 1

namespace vampiria { namespace packet { namespace websocket {

class FramingExtData;

//! Interface for creating websocket extensions
class FramingExt
{
    public:
        //! A constrcutor
        /*!
            @param name extension name
        */
        FramingExt(vmp::str name);
        
        //! A destructor
        virtual ~FramingExt();
        
        vmp::str name_;/*!<extension name*/
        
        //! Function called by the client to have the string to send to the server for configuration
        /*!       
            @return string accepted
        */
        virtual vmp::str message_client()=0;
        
        //!Function called by the client to open the connection when the response from the server arrives
        /*!
            @param params input parameters [name,arg1,arg2...] 
            @param data configuration data
            @return true if configuration is accepted, otherwise false 
        */
        virtual vmp_bool open_client(vmp::vector<vmp::str> params,packet::websocket::FramingExtData *data)=0;
        //! Function called when parameters are available for the pcme i.e. when the connection is opened.
        /*!
             @param params input parameters [name,arg1,arg2...] 
             @param data configuration data
             @return true if configuration is accepted, otherwise false 
       */
        virtual vmp_bool open_server(vmp::vector<vmp::str> params,packet::websocket::FramingExtData *data)=0;
        
        //! Function called when an message complete is received(for every frame).
        /*!
            @param frame input frame
            @param cmp configuration component
        */
        virtual void inframe(packet::websocket::WebSocket_Framing_P *frame,void *cmp)=0;
        
        //! Function called when an message complete is received(for all frames).
        /*!
            @param frames input frames list
            @param data configuration data
        */
        virtual void inframes(vmp::vector<packet::websocket::WebSocket_Framing_P *> frames,packet::websocket::FramingExtData *data)=0;
        
        //! Function called when an message is send(for every frame).
        /*!
            @param frame input frame
            @param data configuration data
        */
        virtual void outframe(packet::websocket::WebSocket_Framing_P *frame,packet::websocket::FramingExtData *data)=0;
        
        //! Function called when an message complete is received(for all frames).
        /*!
            @param frames output frames list
            @param data configuration data
        */
        virtual void outframes(vmp::vector<packet::websocket::WebSocket_Framing_P *> frames,packet::websocket::FramingExtData *data)=0;
        
        
        //! Function called when the session is closed
        /*!
            @param cmp component input
        */
        virtual void close(packet::websocket::FramingExtData *data)=0;
};

//! session data for websocket extensions
class FramingExtData
{
    public:
        //! A constructor
        FramingExtData();
        
        //! A destructor
        ~FramingExtData();
        
        packet::websocket::FramingExt *ext_;/*!<associated extension*/
        vmp::str accepted_;/*!<string accepted to send in session*/
        void *data_;/*!< session data object*/
        
        //! reset object
        void reset();
};

//!helper to manage extensions
class FramingExtManager
{
    private:
        vmp::vector<packet::websocket::FramingExt *> exts_;/*!<extensions list*/
    public:
        //! A constructor
        FramingExtManager();
        
        //! A destructor
        ~FramingExtManager();
        
        //! Add an extension to the list
        /*!
            @param ext extension frame to added
        */
        void add_ext(packet::websocket::FramingExt *ext);
        
        //! Search extension
        /*!
            @param name extension name
            @return the extension associated with the name or 0 in case of failure
        */
        packet::websocket::FramingExt *search(vmp::str name);
        
        //! Function called by the client to have the string to send to the server for configuration
        /*!       
            @return string accepted for all exceptions
        */
        vmp::str message_client();
};

}}}

#endif

