/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 03/09/2024
 */

#include "packet.h"

namespace vampiria { namespace packet { namespace http {

Http_Request_P::Http_Request_P():packet::Packet(packet::http::P_HTTP_REQUEST)
{
    //data_[0]
    insert_data(new packet::DataStrSelect("method",{"GET","POST","OPTIONS","HEAD","PUT","DELETE","TRACE","CONNECT"}));
    
    //data_[1]
    insert_data(new packet::DataStr("uri"));
    
    //data_[2]
    insert_data(new packet::DataStrRegex("version","HTTP\\/\\d+\\.\\d+"));
    
    //data_[3]
    insert_data(new packet::DataStr("headers"));
    
    //data_[4]
    insert_data(new packet::DataBuffer("body"));
}

Http_Request_P::~Http_Request_P()
{
}

void Http_Request_P::read(vmp::Buf *buf)
{
     try
     {
         vmp::str str=buf->read_str(buf->size_reading());
         vmp::Buf body;
         vmp::vector<vmp::str> split=vmp::unicode::str_split(str,"\r\n\r\n");
         vmp_size s=split.size();
         if(s == 0)
             vmp::except_s("");
         else if(s > 1)
         {
             for(vmp_index i=1;i<split.size();i++)
             {
                 body.write_str(split[i].c_str());
                 if(i != s-1)
                     body.write_str("\r\n\r\n");
             }
         }
         vmp::unicode::Strtok strtok(split[0]);
         vmp::str ret=strtok.next(" ");
         if(ret == "")
             vmp::except_s("");
         data_[0]->set("",ret);
         ret=strtok.next(" ");
         if(ret == "")
             vmp::except_s("");
         data_[1]->set("",ret);
         ret=strtok.next("\r\n");
         if(ret == "")
             vmp::except_s("");
         data_[2]->set("",ret);
         vmp::str headers="";
         ret=strtok.next("\r\n");
         while(ret != "")
         {
             vmp::unicode::str_cwrite(&headers,"%s\r\n",ret.c_str());
             ret=strtok.next("\r\n");
         }
         data_[3]->set("",headers);
         body.index();
         packet::DataBuffer *d=(packet::DataBuffer *) data_[4];
         d->read_size_data(&body,body.size());
     }
     catch(vmp::exception &x)
     {
         vmp::except_s("packet::http::Http_Request_P::read() malformed packet");
     }
}

void Http_Request_P::write(vmp::Buf *buf)
{
    data_[0]->write(buf);
    buf->write_str(" ");
    data_[1]->write(buf);
    buf->write_str(" ");
    data_[2]->write(buf);
    buf->write_str("\r\n");
    data_[3]->write(buf);
    buf->write_str("\r\n");
    data_[4]->write(buf);
}

vmp::str Http_Request_P::print()
{
    vmp::str ret="";
    for(vmp_index i=0;i<3;i++)
        vmp::unicode::str_cwrite(&ret,"%s\n",data_[i]->print().c_str());
    packet::http::HeadersHelper hdr;
    vmp::pair<vmp::str,vmp::str> data;
    hdr.read(data_[3]->get(""));
    vmp::PairList<vmp::str,vmp::str> *values=hdr.headers();
    vmp::unicode::str_cwrite(&ret,"headers: [");
    for(vmp_index i=0;i<values->size();i++)
    {
        data=values->get(i);
        if(i == 0)
            vmp::unicode::str_cwrite(&ret,"(%s: %s)",data.first.c_str(),data.second.c_str());
        else
            vmp::unicode::str_cwrite(&ret,",(%s: %s)",data.first.c_str(),data.second.c_str());
    }
    vmp::unicode::str_cwrite(&ret,"]\n");
    vmp::unicode::str_cwrite(&ret,"body:\n[%s]\n",data_[4]->get("").c_str());
    return ret;
}

void Http_Request_P::set_uri(packet::http::UriHelper *uri)
{
    vmp::except_check_pointer((void *) uri,"packet::http::Http_Request_P::set_uri(uri=null)");
    set("uri",uri->print());
}

void Http_Request_P::set_headers(packet::http::HeadersHelper *headers)
{
    vmp::except_check_pointer((void *) headers,"packet::http::Http_Request_P::set_headers(headers=null)");
    set("headers",headers->print());
}

Http_Response_P::Http_Response_P():packet::Packet(packet::http::P_HTTP_RESPONSE)
{
    //data_[0]
    insert_data(new packet::DataStrRegex("version","HTTP\\/\\d+\\.\\d+"));
    
    //data_[1]
    insert_data(new packet::DataDigit("code",100,599));
    
    //data_[2]
    insert_data(new packet::DataStr("reason"));
    
    //data_[3]
    insert_data(new packet::DataStr("headers"));
    
    //data_[4]
    insert_data(new packet::DataBuffer("body"));
    
}

Http_Response_P::~Http_Response_P()
{
}

void Http_Response_P::read(vmp::Buf *buf)
{
     try
     {
         vmp::str str=buf->read_str(buf->size_reading());
         vmp::Buf body;
         vmp::vector<vmp::str> split=vmp::unicode::str_split(str,"\r\n\r\n");
         vmp_size s=split.size();
         if(s == 0)
             vmp::except_s("");
         else if(s > 1)
         {
             for(vmp_index i=1;i<split.size();i++)
             {
                  body.write_str(split[i].c_str());
                  if(i != s-1)
                     body.write_str("\r\n\r\n");
             }
         }
         vmp::unicode::Strtok strtok(split[0]);
         vmp::str ret=strtok.next(" ");
         if(ret == "")
             vmp::except_s("");
         data_[0]->set("",ret);
         ret=strtok.next(" ");
         if(ret == "")
             vmp::except_s("");
         data_[1]->set("",ret);
         ret=strtok.next("\r\n");
         if(ret == "")
             vmp::except_s("");
         data_[2]->set("",ret);
         vmp::str headers="";
         ret=strtok.next("\r\n");
         while(ret != "")
         {
             vmp::unicode::str_cwrite(&headers,"%s\r\n",ret.c_str());
             ret=strtok.next("\r\n");
         }
         data_[3]->set("",headers);
         body.index();
         packet::DataBuffer *d=(packet::DataBuffer *) data_[4];
         d->read_size_data(&body,body.size());
     }
     catch(vmp::exception &x)
     {
         vmp::except_s("packet::http::Http_Response_P::read() malformed packet");
     }
}

void Http_Response_P::write(vmp::Buf *buf)
{
    data_[0]->write(buf);
    buf->write_str(" ");
    data_[1]->write(buf);
    buf->write_str(" ");
    data_[2]->write(buf);
    buf->write_str("\r\n");
    data_[3]->write(buf);
    buf->write_str("\r\n");
    data_[4]->write(buf);
}

vmp::str Http_Response_P::print()
{
    vmp::str ret="";
    for(vmp_index i=0;i<3;i++)
        vmp::unicode::str_cwrite(&ret,"%s\n",data_[i]->print().c_str());
    packet::http::HeadersHelper hdr;
    vmp::pair<vmp::str,vmp::str> data;
    hdr.read(data_[3]->get(""));
    vmp::PairList<vmp::str,vmp::str> *values=hdr.headers();
    vmp::unicode::str_cwrite(&ret,"headers: [");
    for(vmp_index i=0;i<values->size();i++)
    {
        data=values->get(i);
        if(i == 0)
            vmp::unicode::str_cwrite(&ret,"(%s: %s)",data.first.c_str(),data.second.c_str());
        else
            vmp::unicode::str_cwrite(&ret,",(%s: %s)",data.first.c_str(),data.second.c_str());
    }
    vmp::unicode::str_cwrite(&ret,"]\n");
    vmp::unicode::str_cwrite(&ret,"body:\n[%s]\n",data_[4]->get("").c_str());
    return ret;
}

vmp_uint Http_Response_P::get_code()
{
    packet::DataDigit *digit=(packet::DataDigit*) data_[1];
    return (vmp_uint) digit->get_data();
}

void Http_Response_P::set_code(vmp_int code)
{
    packet::DataDigit *d=(packet::DataDigit *)data_[1];
    d->set_data(code);
}
        
void Http_Response_P::set_headers(packet::http::HeadersHelper *headers)
{
    vmp::except_check_pointer((void *) headers,"packet::http::Http_Response_P::set_headers(headers=null)");
    set("headers",headers->print());
}

}}}

