/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/11/2020
 */

#include "packet.h"

namespace vampiria { namespace packet {

DataTypes::DataTypes()
{
}
	
DataTypes::~DataTypes()
{
}

DataBytes::DataBytes(vmp::str field):packet::DataTypes()
{
    field_=field;
}
        
DataBytes::~DataBytes()
{
    field_="";
}

vmp::vector<vmp::str> DataBytes::fields()
{
    vmp::vector<vmp::str> ret;
    ret.push_back(field_);
    return ret;
}

vmp::str DataBytes::print()
{
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%s: %s",field_.c_str(),get(field_).c_str());
    return ret;
}

Bitwise::Bitwise(vmp::str field,vmp_byte init,vmp_byte size)
{
    field_=field;
    if((init > 31) || (size > (32-init)))
        vmp::except_s("internal error (packet::Bitwise())");
    init_=init;
    size_=size;
}

Bitwise::~Bitwise()
{
    field_.clear();
}

vmp::str Bitwise::field()
{
    return field_;
}

vmp_byte Bitwise::init()
{
    return init_;
}

vmp_byte Bitwise::size()
{
    return size_;
}

vmp::str Bitwise::print(vmp_size data)
{
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%s: %s",field_.c_str(),get(data).c_str());
    return ret;
}

DataBits::DataBits(vmp_size datalen):packet::DataTypes()
{
    if(datalen < 1 || datalen > 4)
        vmp::except_s("internal error (packet::DataBits())");
    data_=0;
    datalen_=datalen;
}

DataBits::~DataBits()
{
    vmp::vector_delete_alldata<packet::Bitwise *>(&bits_);
    search_.clear();
}

vmp_size DataBits::get_bits_value(packet::Bitwise *bits)
{
    vmp_byte init=bits->init();
    vmp_byte size=bits->size();
    vmp_size ret=data_;
    ret=ret >> init;
    ret=ret & ((vmp_size)(vmp::math::pow_wrap(2,size)-1));
    return ret;
}

void DataBits::set_bits_value(packet::Bitwise *bits,vmp_size data)
{
    vmp_size mask=0;
    vmp_byte init=bits->init();
    vmp_byte size=bits->size();
    if(data > (vmp::math::pow_wrap(2,size) -1))
        vmp::except_s("Data too large");
    for(vmp_index i=init;i<(init+size);i++)
        mask+=(vmp_size)(vmp::math::pow_wrap(2,i));
    mask = ~mask;
    data_ = data_ & mask;
    data_ = data_ | (data << init);
}

void DataBits::insert_data(packet::Bitwise *bits)
{
    vmp::except_check_pointer((void *) bits,"null pointer input(packet::DataBits())");
    bits_.push_back(bits);
    try
    {
        search_.insert(bits->field(),bits_.size()-1);
    }
    catch(vmp::exception &x)
    {
        vmp::except("fatal error insert data with duplicate labels '%s'(packet::DataBit)",bits->field().c_str());
    }
}

vmp::vector<vmp::str> DataBits::fields()
{
    vmp::vector<vmp::str> ret;
    for(vmp_index i=0;i<bits_.size();i++)
        ret.push_back(bits_[i]->field());
    return ret;    
}

vmp::str DataBits::get(vmp::str field)
{
    vmp_index index;
    if(!search_.search(field,&index))
        vmp::except_s("Field not found");
    vmp_size value=get_bits_value(bits_[index]);
    return bits_[index]->get(value);
}

void DataBits::set(vmp::str field,vmp::str data)
{
    vmp_index index;
    if(!search_.search(field,&index))
        vmp::except_s("Field not found");
    packet::Bitwise *bits=bits_[index];
    vmp_size ret=bits->set(data);
    set_bits_value(bits,ret);    
}

void DataBits::read(vmp::Buf *buf)
{
    data_=buf->read_size(datalen_);
}

void DataBits::write(vmp::Buf *buf)
{
    buf->write_size(data_,datalen_);
}

vmp::str DataBits::print()
{
    vmp::str ret;
    for(vmp_index i=0;i<bits_.size();i++)
    {
        vmp_size value=get_bits_value(bits_[i]);
        if(i==0)
            vmp::unicode::str_write(&ret,"%s",bits_[i]->print(value).c_str());
        else
            vmp::unicode::str_cwrite(&ret,"\n%s",bits_[i]->print(value).c_str());
    }
    return ret; 
}

vmp_size DataBits::get_data(vmp::str field)
{
    vmp_index index;
    if(!search_.search(field,&index))
        vmp::except_s("Field not found");
    return get_bits_value(bits_[index]); 
}

void DataBits::set_data(vmp::str field,vmp_size data)
{
    vmp_index index;
    if(!search_.search(field,&index))
        vmp::except_s("Field not found");
    packet::Bitwise *bits=bits_[index];
    set_bits_value(bits,data); 
}

}}

