/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 05/11/2024
 */

#ifndef VAMPIRIA_NET_SOCKS_H

#define VAMPIRIA_NET_SOCKS_H 1

namespace vampiria { namespace net {

//!Socks user interface
class SocksUI:public event::UI
{
    private:
        vmp::utils::Storage<net::EventSocks> sref_;/*!<Storage event*/
        net::SocksCommon *common_;/*!<Common socks shared object*/
    public:
        //!A Constructor
        /*!
            @param manager input manager
       */
        SocksUI(event::Manager *manager);
        
        //! A Destructor
        ~SocksUI();
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
       
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //!Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //! Set tcp connection event(if value == 0 not change callback)
        /*!
            @param accept function performed when connection is accepted
            @param lclose function performed when close listen socket
            @param close function performed when close server connection
        */
        void set_tcp_event(net::EVTCBACCEPT accept,event::EVTCB lclose,event::EVTCB close);
        
        //! Set timeout value(if value == 0 not change value)
        /*!
            @param ctimeout connection timeout
        */
        void set_timeout(vmp::time::Time ctimeout);
        
        //! Set authentication methods(if value == 0 is disabled method)
        /*!
            @param noauth call callback at connection time for anonymous user verification
            @param userpass call callback at connection time to authenticate users with user and password
        */
        void set_auths(net::SOCKSCBNOAUTH noauth,net::SOCKSCBUSERPASS userpass);

        //! Set the connenction event for command connect(if value == 0 not change callback)
        /*!
            @param cmdconnect function performed when it receives a command request
            @param crecv function performed when a request connect recv package
        */
        void set_connect_event(net::SOCKSCBREQUEST cmdconnect,event::EVTCBBUF crecv);
        
        //! Set the connenction event for command connect(if value == 0 not change callback)
        /*!
            @param cmdbind function performed when it receives a cmd bind request
            @param brecv function performed when a request bind recv package
        */
        void set_bind_event(net::SOCKSCBREQUEST cmdbind,event::EVTCBBUF brecv);
        
        //! Set the connenction eventand data for command udpassociate(if value == 0 not change)
        /*!
            @param cmdudpassociate callback called when it receives a udpassociate request
            @param udprecv function performed when it receives a udpassociate request
            @param udpmaxsize UDP max packet size
        */
        void set_udpassociate_event(net::SOCKSCBREQUEST cmdudpassociate,net::EVTCBRECVFROM udprecv,vmp_size udpmaxsize=net::UDP_MSIZE_DEFAULT);
        
        //! Add socks server listen event
        /*!
            @param local address listener
            @param backlog max connection server
            @return event cell or except in case of failure
        */
        event::Cell *new_server(net::Address *local,vmp_size backlog);
        
        //! Sends a reply error and closes the connection
        /*!
            @param cell cell associated
            @param result result code socks5 error
            @param msg message string(if == 0 use net::msg_result(cd))
        */
        void reply_error(event::Cell *cell,vmp_byte result,vmp::str msg="");
        
        //! Call when a connect request goes well
        /*!
            @param cell cell associated
        */
        void reply_connect(event::Cell *cell);
        
        //! Call when a bind request is listening to
        /*!
            @param cell cell associated
            @param listen listen address bind request
            @return void or except in case of failure
        */
        void reply_bind_listen(event::Cell *cell,net::Address *listen);
        
        //! Call when a bind request receives a connection
        /*!
            @param cell cell associated
            @param connect connect address bind request
            @return void or except in case of failure
        */
        void reply_bind_connect(event::Cell *cell,net::Address *connect);
        
        //! Call when a udp associate request goes well
        /*!
            @param cell cell associated
        */
        void reply_udpassociate(event::Cell *cell);
        
        //! Send packet socks connection for connect and bind connection
        /*!
            @param cell event cell
            @param buf packet buffer
	    @return except error
        */
        void send(event::Cell *cell,vmp::Buf *buf);
        
        //! Send datagram packet,for udpassociate connection
        /*!
            @param cell cell associated
            @param buf data to send
            @param peer remote address to send data
            @return void or except in case of failure
        */
        void sendTo(event::Cell *cell,vmp::Buf *buf,net::Address *peer);
        
        //!Gets socket address local
        /*!
            @param cell cell associated
            @return address local 
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return address local 
        */
        net::Address *peeraddr(event::Cell *cell);
       
        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
        
        //! Returns Connection status
        /*!
            @param cell cell associated
            @return connection status
        */
        vmp_byte cstatus(event::Cell *cell);
        
        //! Returns connection identifier
        /*!
            @param cell cell associated
            @return connection identifier or "" if authentication anonymous
        */
        vmp::str userid(event::Cell *cell);
        
        //! Returns connection permits
        /*!
            @param cell cell associated
            @return connection permits 
        */
        vmp_uint permits(event::Cell *cell);
        
        //! Returns the command received from the socket session
        /*!
            @param cell cell associated
            @return command socks or 0x00 in case of failure
        */
        vmp_byte socks_command(event::Cell *cell);
        
        //! Returns the command received from the socket session in string mode
        /*!
            @param cell cell associated
            @return string command socks
        */
        vmp::str socks_command_str(event::Cell *cell);
        
        //! Returns the address in request
        /*!
            @param cell cell associated
            @sa address_
            @return address in request
        */
        net::Address *socks_address(event::Cell *cell);
        
        //! Returns the bind address of a UDP connection(request udp associate)
        /*!
            @param cell cell associated
            @return bind address UDP connection or except in case of failure
        */
        net::Address *socks_udpbind(event::Cell *cell);
};

}}

#endif


