/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 09/01/2024
 */

#ifndef VAMPIRIA_NET_ADDRESS_H

#define VAMPIRIA_NET_ADDRESS_H 1

namespace vampiria { namespace net {

typedef struct sockaddr Sockaddr;/*!< struct sockaddr type(see linux doc)*/

//!Check if the input string is an ipv4 address
/*!
    @param ip input string
    @return true if input string is ipv4 address,otherwise false
*/
vmp_bool is_ipv4_raw(vmp::str ip);

//! It takes an ipv4 network in subnet mask format and returns the list of IPs associated with the network
/*!
    @param subnet subnet ipv4
    @param mask network mask(0-32)
    @return ipv4 address list
*/
vmp::vector<vmp::str> ipv4_expand(vmp::str subnet,vmp_uint mask);

//!Check if the input string is an ipv6 address
/*!
    @param ip input string
    @return true if input string is ipv6 address,otherwise false
*/
vmp_bool is_ipv6_raw(vmp::str ip);

//!Check if the input string is an ip address
/*!
    @param ip input string
    @return true if input string is ip address,otherwise false
*/
vmp_bool is_ip_raw(vmp::str ip);

//! Manages ip addresses and dns resolve
class Address
{
    private:
        struct addrinfo *addrinfo_;/*!< Address in network for system call network*/ 
        vmp::str  host_; /*!< address in host format (ex www.ragnu.it)*/
        vmp::str  ip_;/*! Ip address in string format*/
        vmp::str  service_; /*!< Service port*/
        vmp_bool onlyhost_; /*!<If set to true no ip is set and not no dns is used*/
        vmp::str dnstype_;/*!<dns method used for host address resolution:<BR> 
                             system use getaddrinfo(default)<BR>
                             tcp4  use tcp dns A<BR>
                             tcp6  use tcp dns AAAA<BR>
                             tcp46 use tcp dns A first,if not resolve use tcp dns AAAA<BR>
                             tcp64 use tcp dns AAAA first if not resolve use tcp dns A)*/
        vmp::str dnsip_;/*!< ip of the dns server to use(don't use with system dnstype)*/
        vmp::str dnsport_;/*!<port of the dns server to use(don't use with system dnstype)*/
        vmp::time::Time dnstout_;/*!<socket timeout for dns connection (don't use with system dnstype)*/
        
        //! Identify an Internet host and and service and set addrinfo structure
        void init();

        //! Setting ip in string format (ip_) and service(service_)
        void init_s();
    public:
        //! A constructor dnstype=system
        Address();
        
        //! A constructor
        /*!
            @param dnstype (see dnstype_ private variable)
            @param dnsip   (see dnsip_ private variable)
            @param dnsport (see dnsport_ private variable)
            @param dnstout (see dnstout_ private variable)
        */
        Address(vmp::str dnstype,vmp::str dnsip,vmp::str dnsport="53",vmp::time::Time dnstout=5.0);
        
        //! A destructor
        ~Address();

        //! Reset
        void reset();
        
        //!Setting host and service port
        /*!
            @param host address in host format (ex www.ragnu.it)
            @param service service port("0" service not setting)
            @param onlyhost If set to true no ip is set and not no dns is used
        */ 
        void set(vmp::str host,vmp::str service="0",vmp_bool onlyhost=false);
        
        //!Setting address. Input address in struct sockaddr type(see linux doc)
        /*!
            @param addr input address
            @param addrsize address size
        */
        void set(net::Sockaddr *addr,vmp_size addrsize);
        
        //!Setting ip address and service port
        /*!
            @param ip ip address in format string
            @param service service port
        */
        void set_ip_raw(vmp::str ip,vmp::str service="0");
        
        //!Copy the input values into the input
        /*!
            @param address address to copy
        */
        void copy(Address *address);

        //!Returns address in struct sockaddr mode(for use in linux net mode ex bind(),connect() etc)
        /*!
            @return address in struct sockaddr mode
        */
        net::Sockaddr *addr();

        //!Returns size address in struct sockaddr mode(for use in linux net mode ex bind(),connect() etc)
        /*!
            @return size address in struct sockaddr mode
        */
        vmp_size addrsize();

        //! Returns address host
        /*!
            @sa host_
            @return address host
        */    
        vmp::str host();
        
        //! Returns address ip
        /*!
            @sa ip_
            @return address ip or execpt in case the address is setting in onlyhost mode
        */ 
        vmp::str ip();

        //! Returns service
        /*!
            @sa service_
            @return service
        */ 
        vmp::str service();

        //!Returns the byte sequence of the ip
        /*!
             @return byte sequence(except PF_UNSPEC)
        */
        vmp::vector<vmp_byte> bytes_ip();
        
        //!Returns iptype
        /*!
            @return PF_UNSPEC(unspecificated address),AF_INET(ipv4),AF_INET(ipv6)
        */
        vmp_int iptype();
        
        //!Check if ip is an ipv4 address
        /*!
            @sa ip_
            @return true ok,otherwise false
        */
        vmp_bool is_ipv4();

        //!Check if ip is an ipv6 address
        /*!
            @sa ip_
            @return true ok,otherwise false
        */
        vmp_bool is_ipv6();
        
        //! Check if address is setting in only host mode
        /*!
            @sa onlyhost_
            @return true if address is setting in only host mode,otherwise false
        */
        vmp_bool is_onlyhost();
};

//!Writes the ip address to the input buffer. If address type is not settting except error
/*!
    @param address input address
    @param buf output buf
*/
void ip_to_buf(net::Address *address,vmp::Buf *buf);

//!Reads an ipv4 address from the buffer
/*!
    @param address address output
    @param buf read buffer
*/
void ipv4_from_buf(net::Address *address,vmp::Buf *buf);

//!Reads an ipv6 address from the buffer
/*!
    @param address address output
    @param buf read buffer
*/
void ipv6_from_buf(net::Address *address,vmp::Buf *buf);

//! Manages mac ethernet address
class MacAddress
{
    private:
        vmp::str saddr_;/*!<mac address in string format(00:01:02:03:04:05)*/
        vmp::vector<vmp_byte> baddr_;/*!<mac address as a sequence of bytes in a vector(6 bytes)*/
    public:
        //! A constructor
        MacAddress();
        
        //! A destructor
        ~MacAddress();

        //!reset
        void reset();
        
        //! Set mac address
        /*!
            @param addr mac address in string format(00:01:02:03:04:05)
        */
        void set(vmp::str addr);
        
        //! Set mac address
        /*!
            @param addr mac as a sequence of bytes in a vector(6 bytes)
        */
        void set(vmp::vector<vmp_byte> baddr);

        //! Returns mac address in string format(00:01:02:03:04:05)
        /*!
            @sa saddr_
            @return mac address in string format
        */ 
        vmp::str str_addr();

        //! Returns mac address as a sequence of bytes in a vector(6 bytes)
        /*!
            @sa saddr_
            @return mac address as a sequence of bytes in a vector(6 bytes)
        */ 
        vmp::vector<vmp_byte> byte_addr();
};

//!Check if the input string is an mac ethernet address
/*!
    @param mac input string
    @return true is mac ethernet address,otherwise false
*/
vmp_bool is_macaddress_raw(vmp::str mac);

//!Writes mac ethernet address to the input buffer
/*!
    @param mac  input mac ethernet address
    @param buf output buf
*/
void macaddress_to_buf(net::MacAddress *mac,vmp::Buf *buf);

//!Reads an mac ethernet address address from the buffer
/*!
    @param mac mac ethernet address output
    @param buf read buffer
*/
void macaddress_from_buf(net::MacAddress *mac,vmp::Buf *buf);

}}

#endif

