/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/10/2024
*/

#ifndef VAMPIRIA_JRP_LIB_H

#define VAMPIRIA_JRP_LIB_H 1

namespace vampiria { namespace jrp {

//! Internal usage input data accepted
class JrpReqData
{
    public:
        //! A constructor
        JrpReqData();
        
        //! A Destructor
        ~JrpReqData();
        
        vmp::str input_;/*!<Input jdata request type*/
        vmp::vector<vmp::str> push_;/*!<Push jdata type*/
        vmp::vector<vmp::str> response_;/*!<Response jdata type*/
        vmp::str permits_s_;/*!<Permits in string format(see vmp::unicode::str_toindex_list)*/
        vmp::vector<vmp_index> permits_;/*!<permission list index*/
        
        //! Returns input jdata request type
        /*!
            @ref input_
            @return input jdata request type
        */
        vmp::str input();
        
        //! Returns push jdata type
        /*!
            @ref push_
            @return push jdata type
        */
        vmp::vector<vmp::str> push();
        
        //! Returns response jdata type
        /*!
            @ref response_
            @return response jdata type
        */
        vmp::vector<vmp::str> response();
        
        //! Set Permission data
        /*!
            @ref permits_
            @ref permits_s_
            @param permits_s permission setting
            @return void or except in case of failure
        */
        void set_permits(vmp::str permits_s);
        
        //! Return permits in string format
        /*!
            @ref permits_s_
            @return permits in string format
        */
        vmp::str permits();
};

class JrpReqSession
{
    public:
        vmp::Table<vmp::str,jrp::JrpReqData *> reqdata_;/*!< Information table of the types of jdata accepted by requests,exchanged when the session starts*/
        vmp::Table<vmp::str,void *> bdata_;/*!<Information table of type of of jdata accepted by broadcast message,exchanged when the session starts*/
        vmp::str nodetype_;/*!<Indicates the type of jrp node associated with the session */
        vmp_uint permits_;/*!<Used in session_get() to read the permissions assigned to the local peer from the remote peer*/
        vmp_bool forward_;/*!<Indicates whether the peer associated with the session can forward requests*/
        
        //! A constructor
        JrpReqSession();
        
        //! A destructor
        ~JrpReqSession();
        
        //! Reset data 
        void reset();
        
        //!Check if you have permissions to run the event associated with the input
        /*!
            @param input jdata input type
            @param permits permits to be verified
            @return true if matching input permission otherwise return false
        */
        vmp_bool match_reqdata(vmp::str input,vmp_uint permits);
        
        //!Check if the type is accepted by broadcast messages
        /*!
            @param input jdata input type
            @param permits permits to be verified
            @return true if matching input permission otherwise return false
        */
        vmp_bool match_bdata(vmp::str input);
        
        //! Create the reqdata object in json format
        /*!
            @param out the output json object where to write the data
            @param permits permissions assigned to the peer(if permits == 0 all data output)
            @return void or except in case of failure
        */
        void json_reqdata(json::JsonObj *out,vmp_uint permits=0);
        
        //! Create jrp session message
        /*!
            @param buf output buffer
            @param permits permissions assigned to the peer(if permits == 0 all data output)
            @return void or except in case of failure
        */
        void session(vmp::Buf *buf,vmp_uint permits=0);
        
        //! Read jrp session message and set session structure
        /*!
            @param root json object session message
            @return void or except in case of failure
        */
        void session_get(json::JsonObj *root);
        
        //! Add a request to table of the types of jdata accepted by requests
        /*!
            @param input jdata input type
            @param push list of push data accepted from the request
            @param response list of response outputs generated by the request
            @param permits permission string (see vmp::unicode::str_toindex_list)
            @ref reqdata_
            @return void or except in case of failure
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits="*");
        
        //! Add a broadcast data to the accepted list
        /*!
            @param input jdata input type
        */
        void add_broadcastdata(vmp::str input);
        
        //! Search for data with input input
        /*!
            @param input input jdata type
            @return json request data or 0 in case of failure
        */
        jrp::JrpReqData *search_reqdata(vmp::str input);
        
        //! Returns all input
        /*!
            @return input list 
        */
        vmp::vector<vmp::str> all_input();
        
        //! Returns the list of inputs that have a push of the topic 
        /*!
            @param push push request data associated with input
            @return input list 
        */
        vmp::vector<vmp::str> search_push(vmp::str push);
        
        //! Returns the list of inputs that have a push and output of the topic 
        /*!
            
            @param response response data associated with input
            @return input list
        */
        vmp::vector<vmp::str> search_response(vmp::str response);
        
        //! Returns the type of jrp node associated with the session */
        /*!
            @ref nodetype_
            @return nodetype
        */
        vmp::str nodetype();
        
        //! Returns remote peer permits
        /*!
            @ref permits_
            @return remote peer permissions
        */
        vmp_uint permits();
        
        //! Returns if the peer associated with the session can forward requests*/
        /*!
            @ref forward_
            @return true if the session can forward requests otherwise false
        */
        vmp_bool forward();
};

//! Message parsing structure jrp
class MsgParse
{
    public:
        //! A constructor
        MsgParse();

        //! A destructor
        ~MsgParse();
        
        json::Json json_;/*!<output jrp message(json structure)*/
        vmp::str msgtype_;/*!<message type*/
        vmp_int status_;/*!<exit status*/
        vmp::str msg_;/*!<exit message*/
        vmp_uint rid_;/*!<request id*/
        json::JsonObj input_;/*!<input data object*/
        vmp::str forward_;/*!<forward address*/
        vmp::Buf payload_;/*!<binary data*/
        json::JsonObj outputs_;/*!< Outputs data object*/
        
        //! Reset data
        void reset();
};

//! Read a json message and check if it is well formed
/*!
    @param buf input buffer
    @param msg message output
    @param session for session message.(if session == 0 no data read)
    @return jrp::status_ok or in case of failure jrp::status_malformed_msg
*/
vmp_int parse(vmp::Buf *buf,jrp::MsgParse *msg,jrp::JrpReqSession *session=0);

//! Internal usage
void jrp_write_str(vmp::Buf *buf,vmp::str jrp,vmp::Buf *payload=0);

//! Internal usage
void jrp_write(vmp::Buf *buf,json::Json *json,vmp::Buf *payload=0);

//! Create jrp abort message
/*!
    @param buf output buffer
    @param status abort status
    @param msg abort message
    @return void or except in case of failure
*/
void abort(vmp::Buf *buf,vmp_int status,vmp::str msg="");

//! Create jrp request message
/*!
     @param buf output buffer
     @param rid request id
     @param input jdata request input
     @param payload binary data(if == 0 no binary data send)
     @param forward for a request that needs to be redirected to another peer, give us the address of the peer being forwarded(if value == "" local peer)
     @return void or except in case of failure
*/
void request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input,vmp::Buf *payload,vmp::str forward="");

//! Create jrp push message(Data sent by the peer who sent the request)
/*!
    @param buf output buffer
    @param rid request id(equal to the rid of the request to which the answer is associated)
    @param outputs jdata outputs push(array or single)
    @param payload binary data(if == 0 no binary data send)
    @return void or except in case of failure
*/
void push(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs,vmp::Buf *payload);

//! Create jrp response message(Data sent by the peer who recv the request)
/*!
    @param buf output buffer
    @param rid request id(equal to the rid of the request to which the answer is associated)
    @param outputs jdata outputs response(array or single)
    @param payload binary data(if == 0 no binary data send)
    @return void or except in case of failure
*/
void response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs,vmp::Buf *payload);

//! Create jrp close message
/*!
     @param buf output buffer
     @param rid request id(equal to the rid of the request to which the close is associated)
     @param status exit staus
     @param msg exit string message
     @return void or except in case of failure
*/
void close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg="");

//! Create jrp kill message
/*!
    @param buf output buffer
    @param rid request id (equal to the rid of the request to which the kill is associated)
    @return void or except in case of failure
*/
void kill(vmp::Buf *buf,vmp_int rid);

//! Create jrp broadcast message
/*!
     @param buf output buffer
     @param input jdata request input
     @param payload binary data(if == 0 no binary data send)
     @return void or except in case of failure
*/
void broadcast(vmp::Buf *buf,json::JsonObj *input,vmp::Buf *payload);

}}

#endif

