/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 07/10/2024
*/

#ifndef VAMPIRIA_JRP_JRPREQ_H

#define VAMPIRIA_JRP_JRPREQ_H 1

namespace vampiria { namespace jrp {

class JrpCommon;

//! Class used fo json request management
class JrpReq
{
    private:
        vmp::str type_;/*!<jrp type of structure("" not assigned,"request" send request,"managed" managed request)*/
        jrp::JrpCommon *common_;/*!<Object that created the request*/
        event::Cell *cell_;/*!<Event Cell associated*/
        event::Cell *kill_;/*!<Kill event timer*/
        vmp_index rid_;/*!<jrp request id*/
        vmp_index count_;/*!<References count*/
        vmp::str key_;/*!<jrp request key(used to insert it in the request tables)*/
        json::Json jdata_;/*!<jdata of the request */
        vmp_bool isclosed_;/*!<Connection is closed? */
        vmp_int status_;/*!<Closing status of the request*/
        vmp::str msg_;/*!<Closing message of the request*/
        vmp::str forward_;/*!<if the request is to be forwarded address of the peer ' for local requests the value is == ""*/
        vmp_bool responseforward_;/*!If enabled the request is handled as forwarded. Used by client requests to forward data*/
        
        //! Internal usage
        event::Manager *get_manager();
        
        //! Reset Object
        void reset();
    public:
        //! Internal usage (see json::jrp::JrpCommon_I)
        void setevent(jrp::JrpCommon *common,event::Cell *cell,vmp_int rid,json::JsonObj *jdata,vmp::str type,vmp::str forward);
        
        //! Internal usage, close and free kill timer
        void kill_end();
        
        //! A Constructor
        JrpReq();
        
        //! A Destructor
        ~JrpReq();
        
        //!Returns the object that created the request
        /*!
            @return the object that created the request
        */
        jrp::JrpCommon *common();
        
        //! Returns jrp type of structure
        /*!
            @ref type_
            @return "" not assigned,"request" send request,"managed" managed request
        */
        vmp::str type();
        
        //! Returns forward peer address
        /*!
            @ref forward_
            @return forward peer address or "" if request is local
        */
        vmp::str forward();
        
        //! Returns if response forward is active
        /*!
            @ref responseforward_
            @return true is active otherwise false
        */
        vmp_bool responseforward();
        
        //! Active response forward,the data is sent to the callback response forwarding as it was received(The jdata list is passed in full)
        /*!
            @ref responseforward_
        */
        void active_responseforward();
        
        //! Returns cell event associated
        /*!
            @ref cell_
            @return cell event associated
        */
        event::Cell *cell();
        
        //! Returns request id
        /*!
            @ref rid_
            @return request id
        */
        vmp_index rid();
        
        //!Returns the jdata type assigned to the request
        /*!
            @ref jdata
            @return jdata type
        */
        vmp::str jdata_type();
        
        //! Returns Object keys
        /*!
            @ref key_
            @return Object key
        */
        vmp::str key();

        //!Returns the pointer of json jdata assigned to the request
        /*!
            @ref jdata
            @return pointer of json structute
        */
        json::Json *jdata_json();
        
        //!Returns the jdata root (jsonobj) assigned to the request
        /*!
            @ref jdata
            @return json root jdata
        */
        json::JsonObj *jdata_root();

        //!Returns jreq status
        /*!
            @param status_
            @return 1 if request is open otherwise exit status
        */
        vmp_int status();
        
        //!Returns exit message
        /*!
            @ref msg_
            @return "open" if request is open otherwise exit message
        */
        vmp::str msg();
        
        //!Allocate a reference
        /*!
            @ref count_
            @return new references count or except in case of failure
        */
        vmp_index alloc_internal();
        
        //!Allocate a reference(monitor mode)
        /*!
            @ref count_
            @return new references count
        */
        vmp_index alloc();
        
        //!Release a references
        /*!
            @ref count_
            @return new references count or except in case of failure(0 if connection is free)
        */
        vmp_index release_internal();

        //!Release a references(monitor and a management error occurs mode)
        /*!
            @ref count_
            @return new references count(0 the jreq is free)
        */
        vmp_index release();
        
        //! Send a jrp push message(type="request" monitor mode)
        /*!
            @param outputs json data outputs
            @param payload payload buffer
            @return void or except in case of failure
        */
        void push(json::JsonObj *outputs,vmp::Buf *payload=0);
        
        //! Send a jrp response message(type="managed" monitor mode)
        /*!
            @param outputs json data outputs
            @param payload payload buffer
            @return void or except in case of failure
        */
        void response(json::JsonObj *outputs,vmp::Buf *payload=0);

        //! Send a jrp kill message(type="request" monitor mode)
        /*!
            
            @return void or except in case of failure
        */
        void kill();

        //! Send a jrp close message(type="managed" monitor mode)
        /*!
            @param buf buffer output
            @return void or except in case of failure
        */
        void close(vmp_int status,vmp::str msg="");
        
        //! Creates an oper message from the request(see json::japi_oper())
        /*!
            @param oper output message
            @param bin  binary data
            @return void or except in case of failure
        */
        void request_oper(json::JsonObj *oper,vmp::Buf *bin=0);
        
        //! Creates an oper message from the push(see json::japi_oper_push())
        /*!
            @param oper output message
            @param jdata push data
            @param bin  binary data
            @return void or except in case of failure
        */
        void push_oper(json::JsonObj *oper,json::JsonObj *jdata,vmp::Buf *bin=0);

        //! Creates an kill oper message from the request(see json::japi_oper_kill())
        /*!
            @param oper output message
            @return void or except in case of failure
        */
        void kill_oper(json::JsonObj *oper);
        
        //! Send an response or a closing of the oper massage in input(see json::japi_oper_data() and json::japi_oper_close())
        /*!and a management error occurs
            @param oper json data type json::japi("operdata") or json::japi("operclose")
            @return void or except in case of failure
        */
        void send_from_oper(json::JsonObj *oper);
        
        //! Used to set the closing values received from the peer(type="request") or local closure
        /*!
            @param status closing status of the request
            @param msg closing message of the request 
        */
        void recv_close(vmp_int status,vmp::str msg="");
};

}}

#endif

