/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 15/10/2024
*/

#include "jrp.h"

namespace vampiria { namespace jrp {

vmp::str msg_status(vmp_int status)
{
    switch(status)
    { 
        case jrp::status_ok:
            return "Success";
        case jrp::status_err:
            return "Generic Error";
        case jrp::status_malformed_msg:
            return "Malformed message";
        case jrp::status_undef_datatype:
            return "Undef datatyp mypermits_e in message";
        case jrp::status_accessdenied:
            return "Access Denied";
        case jrp::status_duplexsession:
            return "Duplex session found";
        case jrp::status_protocolbad:
            return "Protocol bad sequence";
        case jrp::status_closeconnection:
            return "Connection close";
        case jrp::status_timeout:
            return "Connection timeout";
        case jrp::status_killed:
            return "Request killed";
        case jrp::status_input_bad:
            return "Bad input recv";
        case jrp::status_push_bad:
            return "Bad push recv";
        case jrp::status_input_notmanaged:
            return "Unmanaged datatype input";
        case jrp::status_rid_duplex:
            return "Duplex request id";
        case jrp::status_resource_busy:
            return "Resource Busy";
        case jrp::status_resource_accessdenied:
            return "Resource Access Denied";
        case jrp::status_kill_notcomplete:
            return "kill incomplete operation";
        case jrp::status_forward_denied:
            return "Forward Denied";
        case jrp::status_request_overflow:
            return "Overflow request";
        default:
            return "Undef error";
    }
    return "";
}

JrpUI::JrpUI(event::Manager *manager,crypto::Ctx_Peer *ctx,jrp::JrpCommon *jrpcommon,vmp::utils::Logger *logger,vmp::time::Time ctimeout):event::UI(manager)
{
    vmp::except_check_pointer((void *)ctx,"jrp::JrpUI(ctx=null)");
    vmp::except_check_pointer((void *)jrpcommon,"jrp::JrpUI(jrpcommon=null)");
    crypto::init();
    logger_=logger;
    jrpcommon_=jrpcommon;
    p2pcommon_=new crypto::P2pCommon(manager);
    p2pcommon_->set_p2p_init(ctx,ctimeout);
    p2pcommon_->set_p2p_client_event(jrp::jrpui_tcpconnectcb,jrp::jrpui_csessioncb,jrp::jrpui_cclosecb);
    set_p2p_client_event(0,0,0);
    p2pcommon_->set_p2p_server_event(jrp::jrpui_tcpacceptcb,jrp::jrpui_ssessioncb,jrp::jrpui_lclosecb,jrp::jrpui_sclosecb);
    set_p2p_server_event(0,0,0,0);
}

JrpUI::~JrpUI()
{
    if(p2pcommon_ != 0)
        delete p2pcommon_;
    p2pcommon_=0;
    crypto::end();
}

void JrpUI::set_jrp_event(event::Cell *cell)
{
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    jrp::EventJrp *jrp=jrpcommon_->new_event(jrp::jrp_recv_peer,jrp::jrp_close_peer);
    p2p->evt_ssl_sub_active((crypto::EventSslSub *)jrp);  
}

vmp::str JrpUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp_int type=p2p->evt_ssl_type();
    vmp::str strtype=p2p->evt_ssl_strtype();
    net::Address *local=p2p->evt_connection_localaddr(),*peer;
    if(type == crypto::CONN_SSLLISTEN)
        vmp::unicode::str_write(&ret,"Jrp(%s:[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if((type == crypto::CONN_SSLSERVER) || (type == crypto::CONN_SSLCLIENT))
    {
        peer=p2p->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"Jrp(%s:[%s:%s]<->[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                            peer->host().c_str(),peer->service().c_str());
    }
    else if(type == crypto::CONN_SSLPROXY)
    {
        peer=p2p->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"Jrp(%s:[%s:%s]<->[proxy]<->[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                           peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}

void JrpUI::close_event(event::Cell *cell)
{
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    p2p->evt_p2p_close();
}

void JrpUI::free_ref(event::Cell *cell)
{
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    p2p->evt_p2p_free();
    jrp::EventJrp *jrp=(jrp::EventJrp *)p2p->evt_ssl_sub_reset();
    if(jrp != 0)
        jrpcommon_->free_event(jrp);
    p2pref_.free(p2p);
}

event::Event *JrpUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) p2pref_.get();
}

void JrpUI::set_p2p_client_event(event::EVTCB tcpconnect,event::EVTCB csession,event::EVTCB cclose)
{
    if(tcpconnect == 0)
        tcpconnect_=event::empty_ev;
    else
        tcpconnect_=tcpconnect;
    if(csession == 0)
        sessionclient_=event::empty_ev;
    else
        sessionclient_=csession;
    if(cclose == 0)
        closeclient_=event::empty_ev;
    else
        closeclient_=cclose;
}

void JrpUI::set_p2p_server_event(net::EVTCBACCEPT tcpaccept,event::EVTCB ssession,event::EVTCB lclose,event::EVTCB sclose)
{
    if(tcpaccept == 0)
        tcpaccept_=net::empty_accept_ev;
    else
        tcpaccept_=tcpaccept;
    if(ssession == 0)
        sessionserver_=event::empty_ev;
    else
        sessionserver_=ssession;
    if(lclose == 0)
        closelisten_=event::empty_ev;
    else
        closelisten_=lclose;
    if(sclose == 0)
        closeserver_=event::empty_ev;
    else
        closeserver_=sclose;
    
}

void JrpUI::set_key_update(vmp_uint minutes)
{
    p2pcommon_->set_ssl_keyupdate(0,minutes);
}

event::Cell *JrpUI::new_client(net::Address *peer,net::ProxyChain *proxy)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    crypto::EventP2p *p2p=p2pref_.get();
    try
    {
        cell=p2p->evt_p2p_client(this,peer,p2pcommon_,proxy);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"JrpUI::new_client(peer='%s:%s')",peer->host().c_str(),peer->service().c_str());
    }
    catch(vmp::exception &x)
    {
        p2pref_.free(p2p);
        vmp::unicode::str_write(&error,"JrpUI::new_client(peer='%s:%s') error '%s'",peer->host().c_str(),peer->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

event::Cell *JrpUI::new_listen(net::Address *local,vmp_uint backlog)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    crypto::EventP2p *p2p=p2pref_.get();
    try
    {
        cell=p2p->evt_p2p_listen(this,local,p2pcommon_,backlog);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"JrpUI::new_listen(local='%s:%s')",local->host().c_str(),local->service().c_str());
    }
    catch(vmp::exception &x)
    {
        p2pref_.free(p2p);
        vmp::unicode::str_write(&error,"JrpUI::new_listen(local='%s:%s') error '%s'",local->host().c_str(),local->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

net::Address *JrpUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    net::Address *ret=p2p->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *JrpUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    net::Address *ret=p2p->evt_connection_peeraddr();
    manager->unlock();
    return ret;
}

net::ProxyChain *JrpUI::proxychain(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    net::ProxyChain *ret=p2p->evt_connection_proxychain();
    manager->unlock();
    return ret;
}
        
vmp_int JrpUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp_int ret=p2p->evt_ssl_type();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp::str ret=p2p->evt_ssl_strtype();
    manager->unlock();
    return ret;
}

jrp::JrpCommon *JrpUI::jrpcommon()
{
    return jrpcommon_;
}

crypto::Ctx_Peer *JrpUI::ctx()
{
    return p2pcommon_->ctx_;
}

vmp::utils::Logger *JrpUI::logger()
{
    if(logger_ == 0)
        vmp::except_s("jrp::JrpUI::logger() no logger is associated");
    return logger_;
}

vmp_uint JrpUI::permits(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    vmp_uint ret=jrp->peersession_.permits();
    manager->unlock();
    return ret;
}

vmp_uint JrpUI::permits_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    manager_->lock();
    if(!p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("jrp::JrpUI::permits_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    vmp_uint ret=jrp->peersession_.permits();
    manager_->unlock();
    return ret;
}

void JrpUI::get_peer_x509(event::Cell *cell,crypto::X509_Wrap *cout)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    p2p->evt_ssl_peer_x509(cout);
    manager->unlock();
}

vmp::str JrpUI::peer_fingerprint(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp::str ret=p2p->peer_.fingerprint();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::peer_subject(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp::str ret=p2p->peer_.subject();
    manager->unlock();
    return ret;
}

vmp_uint JrpUI::peer_permits(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp_uint ret=p2p->peer_.permits();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::peer_nodetype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    vmp::str ret=jrp->peersession_.nodetype();
    manager->unlock();
    return ret;
}

vmp_bool JrpUI::peer_forward(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    vmp_bool ret=jrp->peersession_.forward();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::peer_subject_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    if(!p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("jrp::JrpUI::subject_peer_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp::str ret=p2p->peer_.subject();
    manager_->unlock();
    return ret;
}

vmp_uint JrpUI::peer_permits_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    manager_->lock();
    if(!p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("jrp::JrpUI::peer_permits_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    crypto::EventP2p *p2p=cell->event<crypto::EventP2p>();
    vmp_uint ret=p2p->peer_.permits();
    manager_->unlock();
    return ret;
}

vmp::str JrpUI::peer_nodetype_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    manager_->lock();
    if(!p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("jrp::JrpUI::peer_nodetype_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    vmp::str ret=jrp->peersession_.nodetype();
    manager_->unlock();
    return ret;
}

vmp_bool JrpUI::peer_forward_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    manager_->lock();
    if(!p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("jrp::JrpUI::peer_nodetype_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    vmp_uint ret=jrp->peersession_.forward();
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_peer(vmp::str input)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<event::Cell *> peer;
    crypto::EventP2p *p2p;
    jrp::EventJrp *jrp;
    manager_->lock();
    if(input == "")
        ret=p2pcommon_->p2ptable_.all_keys();
    else
    {
        peer=p2pcommon_->p2ptable_.all_data();
        for(vmp_index i=0;i<peer.size();i++)
        {
            p2p=peer[i]->event<crypto::EventP2p>();
            jrp=(jrp::EventJrp *)p2p->sub_;
            if(jrp->peersession_.search_reqdata(input) != 0)
                ret.push_back(p2p->peer_.fingerprint());
        }
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_peer_p(vmp::str push)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<event::Cell *> peer;
    crypto::EventP2p *p2p;
    jrp::EventJrp *jrp;
    manager_->lock();
    peer=p2pcommon_->p2ptable_.all_data();
    for(vmp_index i=0;i<peer.size();i++)
    {
        p2p=peer[i]->event<crypto::EventP2p>();
        jrp=(jrp::EventJrp *)p2p->sub_;
        if(jrp->peersession_.search_push(push).size() != 0)
            ret.push_back(p2p->peer_.fingerprint());
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_peer_o(vmp::str response)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<event::Cell *> peer;
    crypto::EventP2p *p2p;
    jrp::EventJrp *jrp;
    manager_->lock();
    peer=p2pcommon_->p2ptable_.all_data();
    for(vmp_index i=0;i<peer.size();i++)
    {
        p2p=peer[i]->event<crypto::EventP2p>();
        jrp=(jrp::EventJrp *)p2p->sub_;
        if(jrp->peersession_.search_response(response).size() != 0)
            ret.push_back(p2p->peer_.fingerprint());
    }
    manager_->unlock();
    return ret;   
}

vmp::vector<vmp::str> JrpUI::all_input(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    ret=jrp->peersession_.all_input();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::all_input_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
        jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
        ret=jrp->peersession_.all_input();
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_input_push(event::Cell *cell,vmp::str push)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    ret=jrp->peersession_.search_push(push);
    manager->unlock();
    return ret;
}
        
vmp::vector<vmp::str> JrpUI::search_input_push_f(vmp::str fingerprint,vmp::str push)
{
    event::Cell *cell=0;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
        jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
        if(push == "")
            ret=jrp->peersession_.all_input();
        else
            ret=jrp->peersession_.search_push(push);
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_input_response(event::Cell *cell,vmp::str response)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    ret=jrp->peersession_.search_response(response);
    manager->unlock();
    return ret;
}
 
vmp::vector<vmp::str> JrpUI::search_input_response_f(vmp::str fingerprint,vmp::str response)
{
    event::Cell *cell=0;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
       if(response == "")
           ret=jrp->peersession_.all_input();
       else
           ret=jrp->peersession_.search_response(response);
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_push(event::Cell *cell,vmp::str input)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    jrp::JrpReqData *req=jrp->peersession_.search_reqdata(input);
    ret=req->push();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_push_f(vmp::str fingerprint,vmp::str input)
{
    event::Cell *cell=0;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
       jrp::JrpReqData *req=jrp->peersession_.search_reqdata(input);
       ret=req->push();
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_response(event::Cell *cell,vmp::str input)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    jrp::JrpReqData *req=jrp->peersession_.search_reqdata(input);
    ret=req->response();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_response_f(vmp::str fingerprint,vmp::str input)
{
    event::Cell *cell=0;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
       jrp::JrpReqData *req=jrp->peersession_.search_reqdata(input);
       ret=req->response();
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::broadcastdata_peer(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
    ret=jrp->peersession_.bdata_.all_keys();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::broadcastdata_peer_f(vmp::str fingerprint)
{
    event::Cell *cell=0;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventP2p>()->sub_;
       ret=jrp->peersession_.bdata_.all_keys();
    }
    manager_->unlock();
    return ret;
}
 
jrp::JrpReq *JrpUI::request(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload,vmp::str forward)
{
    event::Manager *manager=cell->get_manager();
    jrp::JrpReq *ret;
    manager->lock();
    try
    {
        ret=jrpcommon_->new_request(cell,jdata,payload,forward);
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::str tmp=x.what();
        if(tmp == "Overflow request")
            vmp::except(x.what());
        vmp::except("jrp::JrpUI::request() %s",x.what());
    }
    manager->unlock();
    return ret;
}

jrp::JrpReq *JrpUI::request_f(vmp::str fingerprint,json::JsonObj *jdata,vmp::Buf *payload,vmp::str forward)
{
    jrp::JrpReq *ret;
    event::Cell *cell;
    manager_->lock();
    if(!p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("JrpUI::request_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    try
    {
        ret=jrpcommon_->new_request(cell,jdata,payload,forward);
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::str tmp=x.what();
        if(tmp == "Overflow request")
            vmp::except(x.what());
        vmp::except("openssl::jrp::pkg::JrpUI::request_f() %s",x.what());
    }
    manager_->unlock();
    return ret;
}

void JrpUI::broadcast(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        jrpcommon_->broadcast(cell,jdata,payload);
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::except("jrp::JrpUI::request() %s",x.what());
    }
    manager->unlock();
}

WssJrpUI::WssJrpUI(event::Manager *manager,crypto::Ctx_Peer_Web *ctx,vmp::str webroot,jrp::JrpCommon *jrpcommon,vmp::utils::Logger *logger,vmp::time::Time ctimeout):event::UI(manager)
{
    vmp::except_check_pointer((void *)ctx,"jrp::WssJrpUI(ctx=null)");
    vmp::except_check_pointer((void *)jrpcommon,"WssJrpUI(jrpcommon=null)");
    crypto::init();
    logger_=logger;
    jrpcommon_=jrpcommon;
    
    wsscommon_=new crypto::WssCommon(manager);
    wsscommon_->set_wss_init(ctx,webroot,ctimeout);
    wsscommon_->set_wss_event(jrp::wssjrpui_tcpacceptcb,wssjrpui_sessioncb,jrp::wssjrpui_lclosecb,jrp::wssjrpui_sclosecb);
}

WssJrpUI::~WssJrpUI()
{
    if(wsscommon_ != 0)
        delete wsscommon_;
    wsscommon_=0;
    crypto::end();
}

void WssJrpUI::set_jrp_event(event::Cell *cell)
{
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    jrp::EventJrp *jrp=jrpcommon_->new_event(jrp::jrp_recv_server,jrp::jrp_close_server);
    wss->evt_ssl_sub_active((crypto::EventSslSub *)jrp);  
}
     
vmp::str WssJrpUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    vmp_int type=wss->evt_ssl_type();
    vmp::str strtype=wss->evt_ssl_strtype();
    net::Address *local=wss->evt_connection_localaddr(),*peer;
    if(type == crypto::CONN_SSLLISTEN)
        vmp::unicode::str_write(&ret,"WssJrp(%s:[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if((type == crypto::CONN_SSLSERVER))
    {
        peer=wss->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"WssJrp(%s:[%s:%s]<->[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}

void WssJrpUI::close_event(event::Cell *cell)
{
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    wss->evt_wss_close();
}

void WssJrpUI::free_ref(event::Cell *cell)
{
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    wss->evt_wss_free();
    jrp::EventJrp *jrp=(jrp::EventJrp *)wss->evt_ssl_sub_reset();
    if(jrp != 0)
        jrpcommon_->free_event(jrp);
    wssref_.free(wss);
}
        
event::Event *WssJrpUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) wssref_.get();
}

void WssJrpUI::set_http(vmp::str servername)
{
    wsscommon_->set_wss_http(servername,0,0,0);
}

void WssJrpUI::set_wss_event(net::EVTCBACCEPT tcpaccept,event::EVTCB session,event::EVTCB lclose,event::EVTCB sclose)
{
    if(tcpaccept == 0)
        tcpaccept_=net::empty_accept_ev;
    else
        tcpaccept_=tcpaccept;
    if(session == 0)
        session_=event::empty_ev;
    else
        session_=session;
    if(lclose == 0)
        closelisten_=event::empty_ev;
    else
        closelisten_=lclose;
    if(sclose == 0)
        closeserver_=event::empty_ev;
    else
        closeserver_=sclose;
}

event::Cell *WssJrpUI::new_listen(net::Address *local,vmp_uint backlog,vmp::vector<vmp::str> hosts,vmp::vector<vmp::str> origins,vmp_bool localhost)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    crypto::EventWss *wss=wssref_.get();
    try
    {
        cell=wss->evt_wss_listen(this,local,wsscommon_,backlog,hosts,origins,localhost);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"WssJrpUI::new_listen(local='%s:%s')",local->host().c_str(),local->service().c_str());
    }
    catch(vmp::exception &x)
    {
        wssref_.free(wss);
        vmp::unicode::str_write(&error,"WssJrpUI::new_listen(local='%s:%s') error '%s'",local->host().c_str(),local->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

net::Address *WssJrpUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    net::Address *ret=wss->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *WssJrpUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    net::Address *ret=wss->evt_connection_peeraddr();
    manager->unlock();
    return ret;
}

vmp_int WssJrpUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    vmp_int ret=wss->evt_ssl_type();
    manager->unlock();
    return ret;
}

vmp::str WssJrpUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    vmp::str ret=wss->evt_ssl_strtype();
    manager->unlock();
    return ret;
}

jrp::JrpCommon *WssJrpUI::jrpcommon()
{
    return jrpcommon_;
}

crypto::Ctx_Peer_Web *WssJrpUI::ctx()
{
    return wsscommon_->ctx_;     
}

vmp::utils::Logger *WssJrpUI::logger()
{
    if(logger_ == 0)
        vmp::except_s("jrp::WssJrpUI::::logger() no logger is associated");
    return logger_;
}

vmp::str WssJrpUI::peer_user(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    vmp::str ret=wss->sub_->sessionid_;
    manager->unlock();
    return ret;
}

vmp_uint WssJrpUI::peer_permits(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    vmp_uint ret=wss->sub_->sessionpermits_;
    manager->unlock();
    return ret;
}

void WssJrpUI::broadcast(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        jrpcommon_->broadcast(cell,jdata,payload);
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::except("jrp::WssJrpUI::request() %s",x.what());
    }
    manager->unlock();
}

}}

