/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/10/2024
*/

#ifndef VAMPIRIA_JRP_EVENTJRP_H

#define VAMPIRIA_JRP_EVENTJRP_H 1

namespace vampiria { namespace jrp {

//! Generic jrp event type callback(kill,close)
/*!
    @param jreq jreq structure associated
*/
typedef void (*JREQCB)(jrp::JrpReq *jreq);

//! Jrp event type request callback(kill,close)
/*!
    @param jreq jreq structure associated
    @param payload payload data
*/
typedef void (*JREQREQCB)(jrp::JrpReq *jreq,vmp::Buf *payload);

//! Error jrp event type callback(request err)
/*!
    @param cell event cell
    @param jreq jreq structure associated
*/
typedef void (*JREQERRCB)(jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg);

//! jrp event data type callback(response and push)
/*!
    @param jreq jreq structure associated
    @param jdata jdata outputs response
    @param payload payload data
*/
typedef void (*JREQDATACB)(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload);

//! jrp broadcast callback
/*!
    @param cell from which the message comes
    @param jdata input jdata
    @param payload payload data
*/
typedef void (*JBROADCASTCB)(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload);

//!empty callback JREQCB
void empty_jreqcb(jrp::JrpReq *jreq);

//!empty callback JREQCB
void empty_jreqreqcb(jrp::JrpReq *jreq,vmp::Buf *payload);

//!empty callback JREQERRCB
void empty_jreqerrcb(jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg);

//!empty callback JREQDATACB
void empty_jreqdatacb(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload);

//!empty callback JREQDATACB
void empty_jbroadcastcb(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload);

class JrpCommon;

//! A json request protocol event(ssl subevent)
class EventJrp:public crypto::EventSslSub
{
    private:
        //! Reset object
        void evtsub_jrp_reset();
    public:
        //! A constructor
        EventJrp();
        
        //! A destructor
        ~EventJrp();
        
        jrp::JrpCommon *common_;/*< jrp common object common to all connections*/
        jrp::JrpReqSession peersession_;/*!<peer session data*/
        
        vmp_index refrid_;/*!<Internal usage references local id*/
        vmp::Table<vmp_index,jrp::JrpReq *> srvreq_;/*!<Request from peer*/
        vmp::Table<vmp_index,jrp::JrpReq *> cltreq_;/*!<Request to peer*/
        
        //! Send a session close jrp
        /*!
            @param code jrp code message
            @param msg jrp close message
        */
        void evtsub_jrp_close(vmp_int code,vmp::str msg="");
        
        //! virtual function
        vmp::pair<vmp::str,vmp::str> evtsub_session_client(vmp_uint permits);
        
        //! virtual function
        void evtsub_session_client_buf(vmp::pair<vmp::str,vmp::str> data,vmp::Buf *out);
        
        //! virtual function
        vmp::pair<vmp::str,vmp::str> evtsub_session_server(vmp_uint permits,vmp::str subprotocols,vmp::str session);
        
        //! virtual function
        void evtsub_session_server_buf(vmp::pair<vmp::str,vmp::str> data,vmp::Buf *out);
        
        //! virtual function
        void evtsub_session_client_confirm(vmp::str subprotocols,vmp::str session);
        
        //! virtual function
        void evtsub_ssl_close_impl();
        
        //! virtual function
        void evtsub_ssl_free_impl();
};

//! Used for objects shared to jrp sessions
class JrpCommon
{
    private:
        vmp::utils::Storage<jrp::EventJrp> jevt_;/*!<Storage event jrp*/
        vmp::utils::Storage<jrp::JrpReq> jreq_;/*!<Storage request event*/
    public:
        JrpCommon();
        ~JrpCommon();
        
        //!Calles from destructor,internal usgae
        void destroy();
        
        jrp::JrpReqSession session_;/*!<Jrprequests processed by the peer*/
        vmp_size maxclientreq_;/*!<Maximum number of requests a peer can send(default 100)*/
        vmp_size maxssrvpeer_;/*!<maximum number of requests handled per peer (default 10)*/
        
        vmp::time::Time killwait_;/*!<Wait time for closing a request after sending a kill(default 60s)*/
        
        jrp::JREQREQCB requestcb_;/*!<Callback call when a request message has been received*/
        jrp::JREQREQCB requestforwardcb_;/*!<Callback call when a request message must be forwarded*/
        jrp::JREQERRCB requesterrcb_;/*!<Callback call when a request message has been received and a management error occurs(ex.Resource Access Denied) */
        jrp::JREQDATACB responsecb_;/*!<Callback call when a response message has been received(The list of jdata is divided into individual jdata)*/
        jrp::JREQDATACB responseforwardcb_;/*!<Callback call when a response message must be forwarded(The jdata list is passed in full)*/
        jrp::JREQDATACB pushcb_;/*!<Callback call when a push message has been received(The list of jdata is divided into individual jdata)*/
        jrp::JREQDATACB pushforwardcb_;/*!<Callback call when a push message has been must be forwarded(The jdata list is passed in full)*/
        jrp::JREQCB killcb_;/*!<Callback call when a kill message has been received*/
        jrp::JREQCB closecb_;/*!<Callback call when a close message has been received*/
        
        jrp::JBROADCASTCB broadcastcb_;/*!<Callback call when a broadcast message has been received*/
        
        //! Set limits configurations
        /*!
            @param maxssrvpeer maximum number of requests a peer can make(if value = 0 not setting nothing)
            @param maxclientreq maximum number of requests I can request(if value = 0 not setting nothing)
            @param killwait Wait time for closing a request after sending a kill(if value = 0 not setting nothing)
        */
        void set_limits(vmp_size maxssrvpeer,vmp_size maxclientreq,vmp::time::Time killwait);
        
        //! Set callback function. if value == 0 do not update anything.
        /*!
            @param requestcb callback call when a request message has been received
            @param requesterrcb callback call when a request message has been received and a management error occurs(ex.Resource Access Denied)
            @param killcb callback call when a kill message has been received
            @param pushcb Callback call when a push message has been received
            @param responsecb Callback call when a response message has been received
            @param close callback call when a close message has been received
        */
        void jrp_callback(jrp::JREQREQCB requestcb,jrp::JREQERRCB requesterrcb,jrp::JREQCB killcb,jrp::JREQDATACB pushcb,jrp::JREQDATACB responsecb,jrp::JREQCB closecb);
        
        //! Set callback function for forwarding operation.if value == 0 setting empty callback Do not update anything.
        /*!
            @param requestcb callback call when a request forward message has been received   
            @param pushcb Callback call when a push message has been received
            @param responsecb Callback call when a kill message has been received
        */
        void jrp_forward_callback(jrp::JREQREQCB requestcb,jrp::JREQDATACB pushcb,jrp::JREQDATACB responsecb);
        
        //! Set callback function for forwarding operation.if value == 0 setting empty callback Do not update anything.
        /*!
            @param broadcastcb callback call when a broadcast message has been received  
        */
        void jrp_broadcast_callback(jrp::JBROADCASTCB broadcastcb);
        
        //!Initializes the local session
        /*!
            @param nodetype type of node
            @param forward the session can be forwarded
            @ref reqdata_
        */
        void init_session(vmp::str nodetype,vmp_bool forward=false);
        
        //! Add a request to table of the types of jdata accepted by requests to able of sessions
        /*!
            @param input jdata input type
            @param push list of push data accepted from the request
            @param response list of response outputs generated by the request
            @param permits permission string (see vmp::unicode::str_toindex_list)
            @ref reqdata_
            @return void or except in case of failure
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits="*");
        
        //! Add a broadcast data to the accepted list
        /*!
            @param input jdata input type
        */
        void add_broadcastdata(vmp::str input);
        
        //! Reset session data
        /*!
            @ref reqdata_
        */
        void reset_session();
        
        //! Return session nodetype
        /*!
            @ref reqdata_
            @return session nodetype
        */
        vmp::str nodetype();
        
        //! Return if the session can be forwarded
        /*!
            @ref reqdata_
            @return true if the session can be forwarded,otherwise false
        */
        vmp_bool forward();
         
         //! Create a jrp event for a session
        /*!
            @param recv peer data reception(if == 0 not setting value)
            @param close close callbaack(if == 0 not setting value)
            @return void or except in case of failure
        */
        jrp::EventJrp *new_event(event::EVTCBBUF recv,event::EVTCB close);
        
        //! free json event
        /*!
            @param jrp json event to free
        */
        void free_event(jrp::EventJrp *jrp);
        
        //! Create a jrp request and set data for its management(set type_="request").Request id must be assigned before calling this function
        /*!
            @param cell event that manages the requests
            @param jdata json data request object
            @param payload payload data
            @param forward forward address("" not forwarding request)
            @return new request or except in case of failure("Overflow request" in case the maximum number of requests has been reached)
        */
        jrp::JrpReq *new_request(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload=0,vmp::str forward="");
        
        //! Receives a request and sets the values for its management(set type="managed").
        /*!
            @param msg message parse request
            @param cell event that manages the requests
            @return new request or except in case of failure
        */
        jrp::JrpReq *recv_request(jrp::MsgParse *msg,event::Cell *cell);
        
        //! free json request
        /*!
            @param jreq json request to free
        */
        void free_request(jrp::JrpReq *jreq);
        
        //! Send a jrp broadcast message
        /*!
            @param cell event that manages the requests
            @param jdata json data request object
            @param payload payload data
            @return void or except in case of failure
        */
        void broadcast(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload);
        
        //!Returns maximum number of requests a peer can send*/
        /*!
            @ref maxclientreq_
            @return maximum number of requests a peer can send
        */
        vmp_size maxclientreq();
        
        //!Returns maximum number of requests handled per peer
        /*!
            @ref maxssrvpeer_
            @return maximum number of requests handled per peer
        */
        vmp_size maxssrvpeer();
        
        //!Returns wait time for closing a request after sending a kill
        /*!
            @ref killwait_
            @return wait time for closing a request after sending a kill
        */
        vmp::time::Time killwait();
};

//! Returns the type of session used by the cell(monitor mode)(see crypto::EventSsl)
/*!
    @param cell event cell
    @return type of session used
*/
vmp::str sessiontype(event::Cell *cell);
        
//! Returns the id of session used by the cell(monitor mode)(see crypto::EventSsl)
/*!
    @param cell event cell
    @return session id
*/
vmp::str sessionid(event::Cell *cell);
        
//! Returns the permits of session used by the cell(monitor mode)(see crypto::EventSsl)
/*!
    @param cell event cell
    @return session permits
*/
vmp_uint sessionpermits(event::Cell *cell);

}}

#endif

