/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 15/10/2024
*/

#include "vmp.h"
#include "net.h"
#include "event.h"
#include "json.h"
#include "crypto.h"

#include "jrp/lib.h"
#include "jrp/jrpreq.h"
#include "jrp/eventjrp.h"
#include "jrp/cb.h"

#ifndef VAMPIRIA_JRP_H

#define VAMPIRIA_JRP_H 1

namespace vampiria { namespace jrp {

const vmp::str protoname_="jrp.ragnu.it";/*!<subprotocols jrp official name*/

const vmp_int status_ok=0; /*!< Used by the jrp protocol for message states 'Success'*/
const vmp_int status_err=-1;/*!< Used by the jrp protocol for message states 'Generic Error'*/
const vmp_int status_malformed_msg=-2;/*!< Used by the jrp protocol for message states 'Malformed message'*/
const vmp_int status_undef_datatype=-3;/*!< Used by the jrp protocol for message states 'Undef datatype in message'*/
const vmp_int status_accessdenied=-4;/*!< Used by the jrp protocol for message states 'Access Denied'*/
const vmp_int status_duplexsession=-5;/*!< Used by the jrp protocol for message states 'Duplex session found'*/
const vmp_int status_protocolbad=-6;/*!< Used by the jrp protocol for message states 'Protocol bad sequence'*/
const vmp_int status_closeconnection=-7;/*!< Used by the jrp protocol for message states 'Connection close'*/
const vmp_int status_timeout=-8;/*!< Used by the jrp protocol for message states 'Connection timeout'*/
const vmp_int status_killed=-9;/*!< Used by the jrp protocol for message states 'Request killed'*/
const vmp_int status_input_bad=-10;/*!< Used by the jrp protocol for message states 'Bad input recv'*/
const vmp_int status_push_bad=-11;/*!< Used by the jrp protocol for message states 'Bad push recv'*/
const vmp_int status_input_notmanaged=-12;/*!< Used by the jrp protocol for message states 'Unmanaged datatype input'*/
const vmp_int status_rid_duplex=-13;/*!< Used by the jrp protocol fUsed in session_get to read remote peer permissionsor message states 'Duplex request id'*/
const vmp_int status_resource_busy=-14;/*!<User by jrp protocol for message states 'Resource busy'*/
const vmp_int status_resource_accessdenied=-15;/*!<User by jrp protocol for message states 'Resource Access Denied'*/
const vmp_int status_kill_notcomplete=-16;/*!<User by jrp protocol for message states 'kill incomplete operation'*/
const vmp_int status_forward_denied=-17;/*!<User by jrp protocol for message states 'Forward Denied'*/
const vmp_int status_request_overflow=-18;/*!<User by jrp protocol for message states 'Overflow request'*/

const vmp_index min_permits=1;/*!<Minimum limit of JRP permits*/
const vmp_index max_permits=vmp::UINT16MAX;/*!<Maximum limit of JRP permissions*/

//! Takes in input a Jrp int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str msg_status(vmp_int status);

//!Json request protocol user interface
class JrpUI:public event::UI
{
    private:
        crypto::P2pCommon *p2pcommon_;/*!<p2p common object*/
        vmp::utils::Storage<crypto::EventP2p> p2pref_;/*!<Storage p2p event*/
        
        jrp::JrpCommon *jrpcommon_;/*!<Object shared by all events*/
        vmp::utils::Logger *logger_;/*!<Logger object,if equal 0 not logger associated*/
    public:
        //! A Constructor
        /*!
            @param manager input manager
            @param ctx context peer identity
            @param jrpcommon jrp common object(Used for jrp protocol configurations)
            @param logger logger object,if equal 0 not logger associated
        */
        JrpUI(event::Manager *manager,crypto::Ctx_Peer *ctx,jrp::JrpCommon *jrpcommon,vmp::utils::Logger *logger=0,vmp::time::Time ctimeout=0);

        //!A Destructor
        ~JrpUI();
        
        //! Internal usage(no monitor mode)
        void set_jrp_event(event::Cell *cell);

        event::EVTCB tcpconnect_;/*!<Called when a connection tcp connect is complete(client connection)*/
        event::EVTCB sessionclient_;/*!<Callback call when a valid connection has estabilized(client)*/
        net::EVTCBACCEPT tcpaccept_;/*!<Function performed when connection tcp is accepted from listen connections*/
        event::EVTCB sessionserver_;/*!<Callback call when a valid connection has estabilized(server)*/
        event::EVTCB closeclient_;/*!<Close peer client connection*/
        event::EVTCB closelisten_;/*!<Close server listen event*/
        event::EVTCB closeserver_;/*!<Close peer server connection*/
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //! Set p2p event client(if input value = 0 empty callback setting)
        /*!
            @param tcpconnect called when a connection tcp connect is complete
            @param csession called when session init in client connection 
            @param cclose called when the client is close
        */
        void set_p2p_client_event(event::EVTCB tcpconnect,event::EVTCB csession,event::EVTCB cclose);
        
        //! Set p2p event server(if input value = 0 empty callback setting)
        /*!
            @param tcpaccept called when tcp server accept connection
            @param ssession called when session init in server connection
            @param lclose called when the server listen close connection
            @param sclose called when the server server close connection
        */
        void set_p2p_server_event(net::EVTCBACCEPT tcpaccept,event::EVTCB ssession,event::EVTCB lclose,event::EVTCB sclose);
        
        //! Set how often the server should send the key update in minutes
	/*!
	    @param minutes minutes for update, if == 0 no setting(default)
	*/
	void set_key_update(vmp_uint minutes);
	
        //! Create a client connection(monitor mode).Timeout is managed within the event
        /*!
            @param peer peer address
            @param proxy proxy information,if use proxy
            @return event cell or except in case the failure
        */
        event::Cell *new_client(net::Address *peer,net::ProxyChain *proxy=0);
        
        //! Listen server jrp(monitor manager).Timeout is managed within the event
        /*!
            @param local listen address
            @param backlog number of connections accepted
            @return event cell or except in case the failure
        */
        event::Cell *new_listen(net::Address *local,vmp_uint backlog);
        
        //!Gets socket address local
        /*!
            @param cell cell associate
            @return local addres
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return peer address 
        */
        net::Address *peeraddr(event::Cell *cell);

        //! Gets peer proxy info(for client connection)
        /*!
            @param cell cell associated
            @return proxy info or except in case of failure
        */
        net::ProxyChain *proxychain(event::Cell *cell);
        
        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventCoonection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
        
        //! Returns jrp common object shared
        /*!
            @ref jrpcommon_
            @return jrp common object shared
        */
        jrp::JrpCommon *jrpcommon();
        
        //! Returns associated context
        /*!
            @return context associated
        */
        crypto::Ctx_Peer *ctx();
        
        //! Returns associated logger
        /*!
            @ref logger_
            @return associated logger or except in case no logger is associated
        */
        vmp::utils::Logger *logger();

        //! Permission assigned by remote peer to local peer(monitor mode)
        /*!
            @param cell cell associated
            @return permits assigned
        */
        vmp_uint permits(event::Cell *cell);
        
        //! Permission assigned by remote peer to local peer(monitor mode)
        /*!
            @param fingerprint input fingerprint
            @return permits assigned
        */
        vmp_uint permits_f(vmp::str fingerprint);
        
        //!Gets the peer's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_peer_x509(event::Cell *cell,crypto::X509_Wrap *cout);
        
        //! Returns the fingerprint of the peer associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return fingerprint peer
        */
        vmp::str peer_fingerprint(event::Cell *cell);

        //! Returns the subject of the peer associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return subject peer
        */
        vmp::str peer_subject(event::Cell *cell);

        //! Returns the permits of the peer associated with the input cell(monitor mode)
        /*!
            @param cell input cell
            @return permits peer
        */
        vmp_uint peer_permits(event::Cell *cell);
        
        //!Returns the nodetype of the peer associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return nodetype peer
        */
        vmp::str peer_nodetype(event::Cell *cell);
        
        //! Returns true if peer accepts forward packets
        /*!
            @param cell event cell
            @return nodetype peer
        */
        vmp_bool peer_forward(event::Cell *cell);
        
        //! Returns the subject of the peer associated with the input fingerprint(monitor mode) 
        /*!
            @param fingerprint input fingerprint
            @return subject peer or except in case the failure
        */
        vmp::str peer_subject_f(vmp::str fingerprint);
        
        //! Returns the permits of the peer associated with the input fingerprint(monitor manager)
        /*!
            @param fingerprint input fingerprint
            @return permits peer or except in case the failure
        */
        vmp_uint peer_permits_f(vmp::str fingerprint);
        
        //!Returns the nodetype of the peer associated with the input fingerprint(monitor mode)
        /*!
            @param fingerprint input fingerprint
            @return nodetype peer
        */
        vmp::str peer_nodetype_f(vmp::str fingerprint);
        
        //! Returns true if peer accepts forward packets
        /*!
            @param fingerprint input fingerprint
            @return nodetype peer
        */
        vmp_bool peer_forward_f(vmp::str fingerprint);
        
        //! Returns the fingerprints of the peer associated with the input fingerprint(monitor mode)       
        /*!
            @param input input request data required(input ="" return all peer fingerprint)
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer(vmp::str input="");
        
        //! Returns the fingerprints of the peer which have at least one push data of the assigned type (monitor mode)       
        /*!
            @param push push data required
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer_p(vmp::str push);
        
        //! Returns the fingerprints of the peer which have at least one response of the assigned type (monitor mode)       
        /*!
            @param response push data required
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer_o(vmp::str response);
        
        //! Return all peer inputs
        /*!
            @param cell event cell associated with the peer
            @return input list
        */
        vmp::vector<vmp::str> all_input(event::Cell *cell);
        
        //! Return all peer inputs
        /*!
            @param fingerprint fingerprint event associated with the peer
            @return input list
        */
        vmp::vector<vmp::str> all_input_f(vmp::str fingerprint);
        
        //!Return all peer inputs that have a given push
        /*!
            @param cell event cell associated with the peer
            @param push push data
            @return input list
        */
        vmp::vector<vmp::str> search_input_push(event::Cell *cell,vmp::str push);
        
        //!Return all peer inputs that have a given push
        /*!
            @param fingerprint fingerprint event associated with the peer
            @param push push data
            @return input list
        */
        vmp::vector<vmp::str> search_input_push_f(vmp::str fingerprint,vmp::str push);
        
        //!Return all peer inputs that have a given response
        /*!
            @param cell event cell associated with the peer
            @param response response data
            @return input list
        */
        vmp::vector<vmp::str> search_input_response(event::Cell *cell,vmp::str response);
        
        //!Return all peer inputs that have a given response
        /*!
            @param fingerprint fingerprint event associated with the peer
            @param response response data
            @return input list
        */
        vmp::vector<vmp::str> search_input_response_f(vmp::str fingerprint,vmp::str response);
        
        //! Returns the list of push data that have an input of the topic(monitor mode)
        /*!
            @param cell event cell associated with the peer
            @param input request data input
            @return output list stefano orsi
        */  
        vmp::vector<vmp::str> search_push(event::Cell *cell,vmp::str input);
        
        //! Returns the list of outputs that have an input of the topic(monitor manager)
        /*!
            @param fingerprint fingerprint associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_push_f(vmp::str fingerprint,vmp::str input);
        
        //! Returns the list of response data that have an input of the topic(monitor mode)
        /*!
            @param cell event cell associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_response(event::Cell *cell,vmp::str input);
        
        //! Returns the list of response that have an input of the topic(monitor mode)
        /*!
            @param fingerprint fingerprint associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_response_f(vmp::str fingerprint,vmp::str input);
        
        //! Returns the list of broadcast messages that the peer manages
        /*!
            @param cell event cell associated with the peer
            @return output list 
        */ 
        vmp::vector<vmp::str> broadcastdata_peer(event::Cell *cell);
        
        //! Returns the list of broadcast messages that the peer manages
        /*!
            @param fingerprint fingerprint associated with the peer
            @return output list 
        */ 
        vmp::vector<vmp::str> broadcastdata_peer_f(vmp::str fingerprint);
        
        //!Send a request jrp a peer associated with the cell(monitor mode)
        /*!
            @param cell input cell
            @param jdata jdata inputstefano orsi
            @param payload payload data
            @param forward forward peer(if == "" request not forwarding)
            @return the structure associated with the request or except in case of failure("Overflow request" if the number of requests exceeds the maximum number)
        */
        jrp::JrpReq *request(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload=0,vmp::str forward="");

        //!Send a request jrp a peer associated with the fingerprint(monitor mode)
        /*!
            @param fingerprint input cell
            @param jdata jdata input
            @param payload payload data
            @param forward forward peer(if == "" request not forwarding)
            @return the structure associated with the request or except in case of failure("Overflow request" if the number of requests exceeds the maximum number)
        */
        jrp::JrpReq *request_f(vmp::str fingerprint,json::JsonObj *jdata,vmp::Buf *payload=0,vmp::str forward="");
        
        //! Send a jrp broadcast message(monitor mode)
        /*!
            @param cell event that manages the requests
            @param jdata json data request object
            @param payload payload data
            @return void or except in case of failure
        */
        void broadcast(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload=0);
};

//! A Secure WebSocket server User Interface,that handles requests jrp(json request protocol), only in server mode. Used for accessing jrp nodes via web browser with its user interface.
class WssJrpUI:public event::UI
{
    private:
        crypto::WssCommon *wsscommon_;/*!<p2p common object*/
        vmp::utils::Storage<crypto::EventWss> wssref_;/*!<Storage p2p event*/
        
        jrp::JrpCommon *jrpcommon_;/*!<Object shared by all events*/
        vmp::utils::Logger *logger_;/*!<Logger object,if equal 0 not logger associated*/
        
    public:
        //! A Constructor
        /*!
            @param manager input manager
            @param ctx context web socket
            @param webroot webroot root directory of the web program
            @param jrpcommon jrp common object(Used for jrp protocol configurations)
            @param logger logger object,if equal 0 not logger associated
            @param ctimeout connection timeout(if == 0 not setting)
        */
        WssJrpUI(event::Manager *manager,crypto::Ctx_Peer_Web *ctx,vmp::str webroot,jrp::JrpCommon *jrpcommon,vmp::utils::Logger *logger=0,vmp::time::Time ctimeout=0);

        //!A Destructor
        ~WssJrpUI();
        
        net::EVTCBACCEPT tcpaccept_;/*!<Function performed when connection tcp is accepted from listen connections*/
        event::EVTCB session_;/*!<Callback call when a valid connection has estabilized*/
        event::EVTCB closelisten_;/*!<Close listen event*/
        event::EVTCB closeserver_;/*!<Close server connection*/
        
        //! Internal usage(no monitor mode)
        void set_jrp_event(event::Cell *cell);
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //! Set parameters for sessions
        /*!
            @param servername server name(if value == "" not setting nothing)
        */
        void set_http(vmp::str servername);
        
        //! Set callback function. if value == 0 setting empty callback.
        /*!
            @param tcpaccept function performed when connection tcp is accepted from listen connections
            @param session call when a jrp session has been created(Server)
            @param lclose function performed at the end listen event(Listen connection)
            @param sclose function performed at the end of the event(Server)
        */
        void set_wss_event(net::EVTCBACCEPT tcpaccept,event::EVTCB session,event::EVTCB lclose,event::EVTCB sclose);
        
        //! Listen server jrp(monitor manager).Timeout is managed within the event
        /*!
            @param local listen address
            @param backlog the maximum length to which the  queue  of pending  connections
            @param hosts hosts accepted list
            @param origins Alternative sources (e.g. 'localhost') the port set is local
            @param localhost If true the connections to this interface are local,otherwise are remote
            @return event cell or except in case the failure
        */
        event::Cell *new_listen(net::Address *local,vmp_uint backlog,vmp::vector<vmp::str> hosts,vmp::vector<vmp::str> origins,vmp_bool localhost=false);
        
        //!Gets socket address local
        /*!
            @param cell cell associate
            @return local addres
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return peer address 
        */
        net::Address *peeraddr(event::Cell *cell);
        
        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventCoonection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
        
        //! Returns jrp common object shared
        /*!
            @ref jrpcommon_
            @return jrp common object shared
        */
        jrp::JrpCommon *jrpcommon();
        
        //! Returns associated context
        /*!
           @return context associated
        */
        crypto::Ctx_Peer_Web *ctx();
        
        //! Returns associated logger
        /*!
            @sa logger_
            @return associated logger or except in case no logger is associated
        */
        vmp::utils::Logger *logger();
        
        //! Returns the user associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return user associated
        */
        vmp::str peer_user(event::Cell *cell);

        //! Returns the permits of the user associated with the input cell(monitor mode)
        /*!
            @param cell input cell
            @return permits user
        */
        vmp_uint peer_permits(event::Cell *cell);
        
        //! Send a jrp broadcast message(monitor mode)
        /*!
            @param cell event that manages the requests
            @param jdata json data request object
            @param payload payload data
            @return void or except in case of failure
        */
        void broadcast(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload=0);
};

}}

#endif

#include "jrp/misc.h"

