/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 07/01/2024
 */

#ifndef VAMPIRIA_EVENT_TIMERUI_H

#define VAMPIRIA_EVENT_TIMERUI_H 1

namespace vampiria { namespace event {

//!Event used from Timerui
class TimerRef:public event::EventTimer
{
    public:
       //!A Constructor
       TimerRef();

       //!A Destructor
       ~TimerRef();

       vmp::str id_;/*!<id event*/
};

//!Timer User interface
class TimerUI:public event::UI
{
    private:
        vmp::utils::Storage<TimerRef> tref_;/*!<Storage event*/
        event::EVTCB event_;/*!Callback event*/
        event::EVTCB cevent_;/*!Callback Close Event*/
    public:
        //!A Constructor
        /*!
            @param manager input manager
        */
        TimerUI(event::Manager *manager);
        
        //!A Destructor
        ~TimerUI();

        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);

        //!Set Timer event
        /*!
            @param event callback timer active
            @param cevent callback close event 
        */
        void set_event(event::EVTCB event,event::EVTCB cevent);

        //!Create new timer event
        /*!
            @param id id event
            @param timeval time interval to active event(> 0.0 active timer, = 0.0 create timer but not active, < 0.0 except error)
            @return cell associated with event or except in case of failure
        */
        event::Cell *new_timer(vmp::str id,vmp::time::Time timeval=0.0);

        //!Active timer event.If already active set the new time
        /*!
            @param cell cell associated
            @param timeval time interval to active event
            @return void or except in case of failure
        */
        void active_timer(event::Cell *cell,vmp::time::Time timeval);

        //!Deactive timer event.To reactivate it, use active_timer
        /*!
            @param cell cell associated
        */
        void deactive_timer(event::Cell *cell);

        //! Check to see if the timer is active
        /*!
            @param cell cell associated
            @return true timer is active otherwise false
        */
        vmp_bool isactive_timer(event::Cell *cell);

        //!Returns time interval
        /*!
            @param cell cell associated
            @return time interval
        */
        vmp::time::Time get_timeval(event::Cell *cell);

        //!Returns id event
        /*!
            @param cell cell associated
            @return id event
        */
        vmp::str id(event::Cell *cell);
};

//!Event used from Croniterui
class CroniterRef:public event::EventTimer
{
    public:
       //!A Constructor
       CroniterRef();

       //!A Destructor
       ~CroniterRef();

       vmp::str evtname_;/*!<Event name*/
       vmp::str crontab_;/*!<Crontab event*/
       vmp::time::Localtime next_;/*!<Execution time of the next event*/
       event::EVTCB event_;/*!<Cron event execution callback*/
       vmp_bool badnext_;/*!<Internal usage set to true if a closure occurs because there is no next event*/
       
       //! Internal usage
       vmp_uint evt_croniter_next();
};

//!Croniter User interface
class CroniterUI:public event::UI
{
    private:
        vmp::utils::Storage<CroniterRef> cref_;/*!<Storage event*/
    public:
        //!A Constructor
        /*!
            @param manager input manager
        */
        CroniterUI(event::Manager *manager);
        
        //!A Destructor
        ~CroniterUI();

        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Create a cron event
        /*!
            @param evtname event name
            @param crontab cron tab string
            @param event event executed every time of cron
            @param clevent close event executed every time of cron
            @param active activate event?
            @return cell associated with event or except in case of failure
        */
        event::Cell *new_croniter(vmp::str evtname,vmp::str crontab,event::EVTCB event,event::EVTCB clevent,vmp_bool active=true);

        //!Active croniter event.If already active set the new time
        /*!
            @param cell cell associated
            @return void or except in case of failure
        */
        void active_croniter(event::Cell *cell);

        //!Deactive croniter event.To reactivate it, use active_croniter
        /*!
            @param cell cell associated
        */
        void deactive_croniter(event::Cell *cell);

        //! Check to see if the croniter is active
        /*!
            @param cell cell associated
            @return true croniter is active otherwise false
        */
        vmp_bool isactive_croniter(event::Cell *cell);
        
        //! Returns the name of the event
        /*!
            @param cell cell associated
            @return event name
        */
        vmp::str evtname(event::Cell *cell);
        
        //! Returns the crontab string
        /*!
            @param cell cell associated
            @return cronatab name
        */
        vmp::str crontab(event::Cell *cell);
        
        //! Returns the execution time of the next event
        /*!
            @param cell cell associated
            @return the execution time of the next event or except in case the event is inactive or not next cron event
        */
        vmp_uint nextcron(event::Cell *cell);
        
        //! Returns true if a closure occurs because there is no next event
        /*!
            @param cell cell associated
            @return true if a closure occurs because there is no next event,otherwise false
        */
        vmp_bool badnext(event::Cell *cell);
};

}}

#endif

