/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 07/01/2024
 */

#include "event.h"

namespace vampiria { namespace event {

TimerRef::TimerRef():event::EventTimer()
{
}
       
TimerRef::~TimerRef()
{
}

TimerUI::TimerUI(event::Manager *manager):UI(manager)
{
    set_event(0,0);
}

TimerUI::~TimerUI()
{
}

vmp::str TimerUI::identity(event::Cell *cell)
{
    vmp::str ret;
    TimerRef *tevent=cell->event<TimerRef>();
    vmp::unicode::str_write(&ret,"Timer event [id=%s][timeval=%f]",tevent->id_.c_str(),tevent->timeval_);
    return ret;
}

void TimerUI::close_event(event::Cell *cell)
{
    TimerRef *tevent=cell->event<TimerRef>();
    tevent->evt_timer_close();
}
        
void TimerUI::free_ref(event::Cell *cell)
{
    TimerRef *tevent=cell->event<TimerRef>();
    tevent->id_=""; 
    tevent->evt_timer_free();
    tref_.free(tevent);
}

void TimerUI::set_event(event::EVTCB event,event::EVTCB cevent)
{
    event_=event;
    cevent_=cevent;
}

event::Cell *TimerUI::new_timer(vmp::str id,vmp::time::Time timeval)
{
    event::Cell *cell;
    manager_->lock();
    TimerRef *tevent=tref_.get();
    try
    {
        cell=tevent->evt_timer_new(this,timeval,event_,cevent_);
        tevent->id_=id;
    }
    catch(vmp::exception &x)
    {
        tref_.free(tevent);
        manager_->unlock();
        vmp::except_s(x.what()); 
    }
    manager_->unlock();
    return cell;
}

void TimerUI::active_timer(event::Cell *cell,vmp::time::Time timeval)
{
    event::Manager *manager=cell->get_manager();
    TimerRef *tevent=cell->event<TimerRef>();
    manager->lock();
    try
    {
        tevent->evt_timer_active(timeval);    
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what()); 
    }
    manager->unlock();
}

void TimerUI::deactive_timer(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    TimerRef *tevent=cell->event<TimerRef>();
    manager->lock();
    tevent->evt_timer_deactive();    
    manager->unlock();
}

vmp_bool TimerUI::isactive_timer(event::Cell *cell)
{
    vmp_bool ret;
    event::Manager *manager=cell->get_manager();
    TimerRef *tevent=cell->event<TimerRef>();
    manager->lock();
    ret=tevent->evt_timer_isactive();    
    manager->unlock();
    return ret;
}

vmp::time::Time TimerUI::get_timeval(event::Cell *cell)
{
    TimerRef *tevent=cell->event<TimerRef>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::time::Time timeval=tevent->timeval_;
    manager->unlock();
    return timeval;
}

vmp::str TimerUI::id(event::Cell *cell)
{
    TimerRef *tevent=cell->event<TimerRef>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::str id=tevent->id_;
    manager->unlock();
    return id;
}

CroniterRef::CroniterRef()
{
    evtname_="";
    crontab_="";
    event_=0;
    badnext_=false;
}

CroniterRef::~CroniterRef()
{
}

vmp_uint CroniterRef::evt_croniter_next()
{
    return next_.next_cronizer(crontab_)-vmp::time::time_wrap();
}

void croniter_evt(event::Cell *cell)
{
    CroniterRef *cevent=cell->event<CroniterRef>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        cevent->evt_timer_active(cevent->evt_croniter_next());
    }
    catch(vmp::exception &x)
    {
        vmp::str error;
        vmp::unicode::str_write(&error,"event::CroniterUI::active_croniter(evtname=%s) 'Bad next croniter'",cevent->evtname_.c_str());
        manager->cell_close(cell,event::ERROR,error);
        cevent->badnext_=true;
    }
    manager->unlock();
    cevent->event_(cell);
}

CroniterUI::CroniterUI(event::Manager *manager):UI(manager)
{
}

CroniterUI::~CroniterUI()
{
}

vmp::str CroniterUI::identity(event::Cell *cell)
{
    vmp::str ret;
    CroniterRef *cevent=cell->event<CroniterRef>();
    vmp::unicode::str_write(&ret,"Croniter event [name=%s]",cevent->evtname_.c_str());
    return ret;
}

void CroniterUI::close_event(event::Cell *cell)
{
    CroniterRef *cevent=cell->event<CroniterRef>();
    cevent->evt_timer_close();
}

void CroniterUI::free_ref(event::Cell *cell)
{
    CroniterRef *cevent=cell->event<CroniterRef>();
    cevent->evtname_="";
    cevent->crontab_="";
    cevent->event_=0;
    cevent->badnext_=false;
    cevent->evt_timer_free();
    cref_.free(cevent);
}

event::Cell *CroniterUI::new_croniter(vmp::str evtname,vmp::str crontab,event::EVTCB event,event::EVTCB clevent,vmp_bool active)
{
    event::Cell *cell;
    CroniterRef *cevent=cref_.get();
    manager_->lock();
    try
    {
        cevent->evtname_=evtname;
        cevent->crontab_=crontab;
        cevent->next_.set();
        cevent->evt_croniter_next();
        cell=cevent->evt_timer_new(this,0,croniter_evt,clevent);
        if(event != 0)
            cevent->event_=event;
        else
            cevent->event_=event::empty_ev;
    }
    catch(vmp::exception &x)
    {
        cevent->evtname_="";
        cevent->crontab_="";
        cref_.free(cevent);
        manager_->unlock();
        vmp::except_s(x.what()); 
    }
    manager_->unlock();
    if(active)
        active_croniter(cell);
    return cell;
}

void CroniterUI::active_croniter(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    try
    {
        cevent->next_.set();
        cevent->evt_timer_active(cevent->evt_croniter_next());
    }
    catch(vmp::exception &x)
    {
        vmp::str error;
        vmp::unicode::str_write(&error,"event::CroniterUI::active_croniter(evtname=%s) 'Bad next croniter'",cevent->evtname_.c_str());
        manager->cell_close(cell,event::ERROR,error);
        cevent->badnext_=true;
    }
    manager->unlock();
}

void CroniterUI::deactive_croniter(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    cevent->evt_timer_deactive();
    manager->unlock();
}

vmp_bool CroniterUI::isactive_croniter(event::Cell *cell)
{
    vmp_bool ret;
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    ret=cevent->evt_timer_isactive();
    manager->unlock();
    return ret;
}

vmp::str CroniterUI::evtname(event::Cell *cell)
{
    vmp::str ret;
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    ret=cevent->evtname_;
    manager->unlock();
    return ret;
}

vmp::str CroniterUI::crontab(event::Cell *cell)
{
    vmp::str ret;
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    ret=cevent->crontab_;
    manager->unlock();
    return ret;
}

vmp_uint CroniterUI::nextcron(event::Cell *cell)
{
    vmp::str error="";
    vmp_uint ret;
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    if(cevent->badnext_)
        vmp::unicode::str_write(&error,"event::CroniterUI::active_croniter(evtname=%s) 'Bad next croniter'",cevent->evtname_.c_str());
    else if(!cevent->evt_timer_isactive())
        vmp::unicode::str_write(&error,"event::CroniterUI::active_croniter(evtname=%s) 'inactive event'",cevent->evtname_.c_str());
    else
        ret=cevent->next_.time();
    manager->unlock();
    if(error!="")
        vmp::except_s(error);
    return ret;
}

vmp_bool CroniterUI::badnext(event::Cell *cell)
{
    vmp_bool ret;
    event::Manager *manager=cell->get_manager();
    CroniterRef *cevent=cell->event<CroniterRef>();
    manager->lock();
    ret=cevent->badnext_;
    manager->unlock();
    return ret;
}

}}

