/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 15/10/2024
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

vmp::str p2p_msg_status(vmp_int status)
{
    switch(status)
    {
        case crypto::p2p_ok:
            return "Success";
        case crypto::p2p_err:
            return "Generic Error";
        case crypto::p2p_accessdenied:
            return "Access Denied";
        case crypto::p2p_duplexsession:
            return "Duplex session found";
        case crypto::p2p_protocolbad:
            return "Protocol bad";
        case crypto::p2p_extbad:
            return "Extensions framing error";
        case crypto::p2p_sessionbad:
            return "peer session bad";
        default:
            return "Undef error";
    }
    return "";
}

void p2p_session(vmp::Buf *buf,vmp::str subprotocols,vmp::str session,vmp::str extensions)
{
    json::Json json;
    json::JsonObj *root=json.root(),tmp;
    root->add_object_str("p2ptype","session",&tmp);
    root->add_object_str("subprotocols",subprotocols,&tmp);
    root->add_object_str("session",session,&tmp);
    root->add_object_str("extensions",extensions,&tmp);
    buf->write_str(json.json_str());
    buf->index();
}

void p2p_session_error(vmp::Buf *buf,vmp_int retcode,vmp::str retmsg)
{
    json::Json json;
    json::JsonObj *root=json.root(),tmp;
    root->add_object_str("p2ptype","error",&tmp);
    root->add_object_number("retcode",retcode,&tmp);
    root->add_object_str("retmsg",retmsg,&tmp);
    buf->write_str(json.json_str());
    buf->index();
}

P2pCommon::P2pCommon(event::Manager *manager):crypto::SslCommon(manager)
{
    ctx_=0;
    csession_=event::empty_ev;
    ssession_=event::empty_ev;
    set_ssl_client_event(0,crypto::p2psession,crypto::p2psession_recv,0);
    set_ssl_server_event(0,crypto::p2psession,crypto::p2psession_recv,0,0);
    set_ssl_framing_cb(0,crypto::ssl_subprotocols_recv);
}

P2pCommon::~P2pCommon()
{
    vmp::vector<event::Cell *> data=p2ptable_.all_data();
    for(vmp_index i=0;i<data.size();i++)
        data[i]->release();
    p2ptable_.clear();
}

void P2pCommon::set_p2p_init(crypto::Ctx_Peer *ctx,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)ctx,"P2pCommon::set_p2p_init(ctx=null)");
    ctx_=ctx;
    set_ssl_init(ctx->get(),ctimeout);
}

void P2pCommon::set_p2p_client_event(event::EVTCB tcpconnect,event::EVTCB csession,event::EVTCB cclose)
{
    set_ssl_client_event(tcpconnect,0,0,cclose);
    if(csession != 0)
        csession_=csession;
}

void P2pCommon::set_p2p_server_event(net::EVTCBACCEPT tcpaccept,event::EVTCB ssession,event::EVTCB lclose,event::EVTCB sclose)
{
    set_ssl_server_event(tcpaccept,0,0,lclose,sclose);
    if(ssession != 0)
        ssession_=ssession;    
}

EventP2p::EventP2p():crypto::EventSsl()
{
    evt_p2p_reset();
}

EventP2p::~EventP2p()
{
}

void EventP2p::evt_p2p_reset()
{
    peer_.reset();
    isopensession_=false;
}

event::Cell *EventP2p::evt_p2p_client(event::UI *ui,net::Address *peer,crypto::P2pCommon *common,net::ProxyChain *proxy)
{
    return evt_ssl_client(ui,peer,(crypto::SslCommon *)common,proxy);
}

event::Cell *EventP2p::evt_p2p_listen(event::UI *ui,net::Address *local,crypto::P2pCommon *common,vmp_uint backlog)
{
    return evt_ssl_listen(ui,local,(crypto::SslCommon *)common,backlog);
}

void EventP2p::evt_p2p_close()
{
    if(isopensession_)
    {
        try
        {
            crypto::P2pCommon *common=(crypto::P2pCommon *)common_;
            event::Cell *tmp;
            common->p2ptable_.cancel(peer_.fingerprint(),&tmp);
            event::Manager *manager=tmp->get_manager();
            manager->cell_release(tmp);
        }
        catch(vmp::exception &x)
        {
        }
        isopensession_=false;
    }
    evt_ssl_close();
}

void EventP2p::evt_p2p_free()
{
    evt_p2p_reset();
    evt_ssl_free();
}

}}

