#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 06/11/2024

import vmpsys    as sys
import vmpconfig as config
import vmpcheck  as vcheck

##Optional included path
included_=[]

##c++ library installed
clibs_={}

##Check and install dependencies(internal lib)<BR>
#
#@param depends dependencies list<BR>
#@param setup   generic class setup value
def dependencies(depends,setup):
  for d in depends:
    if d in clibs_:
       if not d in setup.deprun:
         clibs_[d].install(setup)
    else:
       sys.msg_failure("Internal Lib \'{0}\'".format(d),'not found')

##Check external command header and libs<BR>
#
#@param cmd external command list<BR>
#@param hdr header list<BR>
#@param libs library list
def check(cmd,hdr,libs):
  for c in cmd:
    sys.search_cmd(c.split())
  for h in hdr:
    vcheck.header(h)
  for l in libs:
    vcheck.lib(l[0],l[1])

##Library installer c++ Object 
class clib:
  def __init__(self):
    ##Library name
    self.name=''
    ##Command external ['ls -l',...]
    self.cmd=[]
    ##Header included ['stdio.h'...]
    self.hdr=[]
    ##External library [('extlibname','min version'),...]
    self.libs=[]
    ##Files to compile [('directory files',['file1',...]),...]
    self.files=[]
    ##Compiler Options ['`pkg-config --cflags python3`',...]
    self.coptions=[]
    ##Linker options ['`pkg-config --libs python3`',...]
    self.loptions=[]
    ##Internal library dependencies ['vmp',...]
    self.dependencies=[]
    ##Is it an external package to the framework?
    self.pkg=False
    ##Destination path lib (default framework/lib).
    self.libpath=sys.path(config.fwlib_)
    ##Config file create?
    self.config=False

  def rootpath(self):
    if self.pkg:
      return sys.path(config.gpkg_)
    else:
      return sys.path(config.fwsrc_)
 
  ##Install library<BR>
  #
  #@param setup generic class setup value
  def install(self,setup):
    path="{0}{1}/".format(self.rootpath(),self.name)
    sobj="{0}{1}.so".format(self.libpath,self.name)
    setup.deprun[self.name]=self
    dependencies(self.dependencies,setup)
    setup.coptions=setup.coptions+self.coptions
    if not sys.isfile(sobj):
      check(self.cmd,self.hdr,self.libs)
      if self.config:
        sys.config_create("{0}config.h".format(sys.path(config.fwsrc_)),config.configlist_)
      try:
        objfiles=sys.compiler(path,config.cpp_(included_,setup.coptions),self.files) + setup.sobject
        sys.linker(path,config.cppso_(),objfiles,self.loptions,sobj)
      except Exception as e:
        sys.msg_failure("Build Lib \'{0}\'".format(self.name),e)
    setup.sobject.append(sobj)
    sys.msg_ok("Build Lib \'{0}\'".format(self.name))
    
  #Uninstall library 
  def clear(self):
    if self.config:
      sys.config_remove("{0}config.h".format(sys.path(config.fwsrc_)))
    sobj="{0}{1}.so".format(config.config_path(self.libpath),self.name)
    if not sys.isfile(sobj):
      return
    sys.rm_file(sobj)
    sys.rm_cobj("{0}{1}".format(self.rootpath(),self.name),self.files)
    
##Remove all cpp library
def clear():
   for name in clibs_:
     clibs_[name].clear()

vmplib=clib()
vmplib.name='vmp'
vmplib.hdr=['assert.h','dirent.h','errno.h','fcntl.h','ctype.h','pthread.h','pty.h','signal.h','sys/time.h','sys/stat.h','sys/types.h','sys/wait.h','sys/ioctl.h','sys/sysmacros.h','net/if.h','string.h','strings.h','stdarg.h','stdio.h','stdlib.h','time.h','unistd.h']
vmplib.files=[('./',['args','basetypes','buf','env','error','io','math','signal','thread','time','unicode','utils']),('lib/',['dev','fd','fs','lib','proc','types'])]
vmplib.loptions=['-pthread','-lutil','-lc']
vmplib.config=True
clibs_[vmplib.name]=vmplib

pylib=clib()
pylib.name='pylib'
pylib.files=[('./',['args','lib','module','stub']),('lib/',['arg_from_python','class_impl','converter','def','pybase'])]
pylib.libs=[('python3','3.11')]
pylib.coptions=['`pkg-config --cflags python3`']
pylib.loptions=['`pkg-config --libs python3`']
pylib.dependencies=['vmp']
clibs_[pylib.name]=pylib

processlib=clib()
processlib.name='process'
processlib.files=[('./',['box','exec','stdw'])]
processlib.dependencies=['vmp']
clibs_[processlib.name]=processlib

eventlib=clib()
eventlib.name='event'
eventlib.dependencies=['vmp']
eventlib.files=[('./',['cell','manager','ui']),('lib/',['eventinput','eventsimplex','eventtimer','eventthread','lib']),('ui/',['breakloopui','inputui','simplexui','timerui','threadui'])]
clibs_[eventlib.name]=eventlib

netlib=clib()
netlib.name='net'
netlib.hdr=['sys/socket.h','netdb.h']
netlib.files=[('./',['address','dns','net','netdevice','socket','tcp','udp','socks']),('connection/',['cb','connection','udpcb','tcpcb','proxy','proxycb','eventsocks','sockscb'])]
netlib.dependencies=['vmp','event']
clibs_[netlib.name]=netlib

packetlib=clib()
packetlib.name='packet'
packetlib.files=[('./',['arp','ethernet','ip','packet','raw','tl']),('datatypes/',['address','bitwise','buffer','datatypes','number','size','str','types']),('http/',['http','protocol']),('websocket/',['ext','helper','websocket'])]
packetlib.dependencies=['vmp','net']
clibs_[packetlib.name]=packetlib

xmllib=clib()
xmllib.name='xml'
xmllib.files=[('./',['parser','xml'])]
xmllib.dependecies=['vmp']
xmllib.libs=[('libxml++-2.6','2.40.1')]
xmllib.coptions=['`pkg-config libxml++-2.6 --cflags --libs`']
xmllib.loptions=['`pkg-config libxml++-2.6 --cflags --libs`']
clibs_[xmllib.name]=xmllib

moduleslib=clib()
moduleslib.name='modules'
moduleslib.files=[('./',['modules','modinput','modparser'])]
moduleslib.dependencies=['vmp','process','xml','net']
clibs_[moduleslib.name]=moduleslib

jsonlib=clib()
jsonlib.name='json'
jsonlib.files=[('./',['json','japi','jdata','jtable','jlist'])]
jsonlib.dependencies=['vmp','event','net','packet']
jsonlib.libs=[('json-c','0.16')]
jsonlib.loptions=['`pkg-config json-c --cflags --libs`']
clibs_[jsonlib.name]=jsonlib

rawnetlib=clib()
rawnetlib.name='rawnet'
rawnetlib.dependencies=['vmp','packet','event']
rawnetlib.files=[('./',['arpui','lib','rawnet','rawevent','rawui'])]
rawnetlib.libs=[('libpcap','1.10.3')]
rawnetlib.coptions=['`pkg-config --cflags libpcap`']
rawnetlib.loptions=['`pkg-config --libs  libpcap`']
clibs_[rawnetlib.name]=rawnetlib

dblib=clib()
dblib.name='db'
dblib.dependencies=['vmp']
dblib.files=[('./',['db'])]
dblib.libs=[('sqlite3','3.40.1')]
dblib.loptions=['`pkg-config sqlite3 --cflags --libs`']
clibs_[dblib.name]=dblib

cryptolib=clib()
cryptolib.name='crypto'
cryptolib.dependencies=['vmp','net','event','json','packet','db']
cryptolib.cmd=['openssl version','chmod --version','cp --version']
cryptolib.files=[('./',['crypto','eventssl','eventp2p','eventwss','lib','hash','pkey','ssl','x509']),('ctx/',['ctx','base','https','peer']),('cb/',['sslcb','framingcb','p2pcb','wsscb'])]
cryptolib.libs=[('openssl','3.0.9')]
cryptolib.loptions=['`pkg-config --libs  openssl`']
clibs_[cryptolib.name]=cryptolib

jrplib=clib()
jrplib.name='jrp'
jrplib.dependencies=['vmp','net','event','json','crypto']
jrplib.files=[('./',['eventjrp','jrp','jrpreq','lib',]),('cb/',['cb','jrpui','wssui']),('misc/',['status'])]
clibs_[jrplib.name]=jrplib

