/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/04/2024
 */

#include "vmp.h"
#include "net.h"
#include "event.h"
#include "json.h"
#include "db0/db0.h"
#include "openssl4/openssl4.h"

#include "web/ctx.h"
#include "eventwss.h"

#ifndef VAMPIRIA_WEB_PKG_H

#define VAMPIRIA_WEB_PKG_H 1

namespace vampiria { namespace web { namespace pkg {

//! A Secure WebSocket server User Interface,that handles requests jrp(json request protocol), only in server mode. Used for accessing jrp nodes via web browser with its user interface.
class WssAJrpUI:public event::UI
{
    //private:
    
    public:
        //! A Constructor
        /*!
            @param manager input manager
            @param ctx context web socket
            @param localhost If true, check connections for local hosts, if false, all hosts are considered remote
            @param logger logger object,if equal 0 not logger associated
        */
        WssAJrpUI(event::Manager *manager,web::pkg::Ctx_Peer_Web *ctx,vmp_bool localhost,vmp::utils::Logger *logger=0);

         //!A Destructor
        ~WssAJrpUI();
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //! Listen server jrp(monitor manager).Timeout is managed within the event
        /*!
            @param local listen address
            @return event cell or except in case the failure
        */
        event::Cell *new_listen(net::Address *local);
        
        //! Renegotiate the session key with the peer
	/*!
	    @param cell event cell
	    @return void or except in case of failure
	*/
	void key_update(event::Cell *cell);
	
	//!Gets socket address local
        /*!
            @param cell cell associate
            @return local addres
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return peer address 
        */
        net::Address *peeraddr(event::Cell *cell);
        
        //! Return the jrp session values that local peer sends to the remote
        /*!
            @return jrp session object
        */
        json::jrp::JrpReqSession *session_localpeer();
        
        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventCoonection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
        
        //! Returns associated context
        /*!
            @sa ctx_
            @return context associated
        */
        web::pkg::Ctx_Peer_Web *ctx();
        
        //! Returns associated logger
        /*!
            @sa logger_
            @return associated logger or except in case no logger is associated
        */
        vmp::utils::Logger *logger();
        
        //! Returns the user associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return user associated
        */
        vmp::str peer_user(event::Cell *cell);

        //! Returns the permits of the user associated with the input cell(monitor mode)
        /*!
            @param cell input cell
            @return permits user
        */
        vmp_uint peer_permits(event::Cell *cell);
        
        //! Adds a request data
        /*!
            @param input input request data
            @param push data push associated with the input
            @param response data response associated with the input
            @param permits permits string(see json::jrp::JrpCommon_I)
            @return void or except in case of failure
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits="*");

        //! Add default status value
        /*!
            @param permits permits string(see json::jrp::JrpCommon_I)
            @return void or except in case of failure
        */
        void add_reqstatus(vmp::str permits="*");
};

}}}

#endif

