/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 21/03/2024
 */
 
#include "openssl_.h"

VAMPIRIA_PYLIB_INIT(vmp_openssl4_);

class PySslUI:public openssl::pkg::SslUI
{
   public:
        PySslUI(event::Manager *manager);
        PySslUI(event::Manager *manager,vmp::time::Time ctimeout);
        ~PySslUI();

        pylib::PyCall pytcpconnect_;
        pylib::PyCall pycsslconnect_;
        pylib::PyCall pycrecv_;
        pylib::PyCall pycclose_;

        pylib::PyCall pyaccept_;
        pylib::PyCall pyssslconnect_;
        pylib::PyCall pylclose_;
        pylib::PyCall pysrecv_;
        pylib::PyCall pysclose_;
};

PySslUI::PySslUI(event::Manager *manager):openssl::pkg::SslUI(manager)
{
}

PySslUI::PySslUI(event::Manager *manager,vmp::time::Time ctimeout):openssl::pkg::SslUI(manager,ctimeout)
{
}

PySslUI::~PySslUI()
{
}

struct stub_sslui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PySslUI *>(cell->ui<PySslUI>(),false);
        }
        return 0;
    }
};

struct SslUI_Constructor:public pylib::Constructor<PySslUI>
{
    PySslUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 1)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             return new PySslUI(manager);
         }
         else if(args.size() == 2)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             vmp::time::Time ctimeout=args.get<vmp::time::Time>(1);
             return new PySslUI(manager,ctimeout);
         }
         return 0;
    }
};

void ssl_pytcpconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pytcpconnect_.call(cargs);
}

void ssl_pycsslconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pycsslconnect_.call(cargs);
}

void ssl_pycrecv(event::Cell *cell,vmp::Buf *buf)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    sslui->pycrecv_.call(cargs);
}

void ssl_pycclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pycclose_.call(cargs);
}

struct stub_SslUI_set_event_client:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 4)
        {
            event::EVTCB      tcpconnect;
            event::EVTCB      sslconnect;
            event::EVTCBBUF   crecv;
            event::EVTCB      cclose;
            
            args.get_callable(0,instance->pytcpconnect_);
            args.get_callable(1,instance->pycsslconnect_);
            args.get_callable(2,instance->pycrecv_);
            args.get_callable(3,instance->pycclose_);
            
            if(pylib::is_none(instance->pytcpconnect_.ref()))
                tcpconnect=0;
            else
                tcpconnect=ssl_pytcpconnect;
            if(pylib::is_none(instance->pycsslconnect_.ref()))
                sslconnect=0;
            else 
                sslconnect=ssl_pycsslconnect;
            if(pylib::is_none(instance->pycrecv_.ref()))
                crecv=0;
            else 
                crecv=ssl_pycrecv;
            if(pylib::is_none(instance->pycclose_.ref()))
                cclose=0;
            else 
                cclose=ssl_pycclose;            
            instance->set_event_client(tcpconnect,sslconnect,crecv,cclose);
            return pylib::retnone();                      
        }
        return 0;
    }
};

void ssl_pyaccept(event::Cell *listen,event::Cell *server)
{
    PySslUI *sslui=listen->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(listen,false);
    cargs.add<event::Cell *>(server,false);
    sslui->pyaccept_.call(cargs);
}

void ssl_pyssslconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pyssslconnect_.call(cargs);
}

void ssl_pylclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pylclose_.call(cargs);
}

void ssl_pysrecv(event::Cell *cell,vmp::Buf *buf)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    sslui->pysrecv_.call(cargs);
}

void ssl_pysclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pysclose_.call(cargs);
}

struct stub_SslUI_set_event_server:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 5)
        {
            net::EVTCBACCEPT  accept;
            event::EVTCB      sslconnect;
            event::EVTCB      lclose; 
            event::EVTCBBUF   srecv;
            event::EVTCB      sclose;
            
            args.get_callable(0,instance->pyaccept_);
            args.get_callable(1,instance->pyssslconnect_);
            args.get_callable(2,instance->pylclose_);
            args.get_callable(3,instance->pysrecv_);
            args.get_callable(4,instance->pysclose_);
            
            if(pylib::is_none(instance->pyaccept_.ref()))
                accept=0;
            else 
                accept=ssl_pyaccept;
            if(pylib::is_none(instance->pyssslconnect_.ref()))
                sslconnect=0;
            else 
                sslconnect=ssl_pyssslconnect;
            if(pylib::is_none(instance->pylclose_.ref()))
                lclose=0;
            else 
                lclose=ssl_pylclose;
            if(pylib::is_none(instance->pysrecv_.ref()))
                srecv=0;
            else 
                srecv=ssl_pysrecv;
            if(pylib::is_none(instance->pysclose_.ref()))
                sclose=0;
            else 
                sclose=ssl_pysclose;
            instance->set_event_server(accept,sslconnect,lclose,srecv,sclose);
            return pylib::retnone();                      
        }
        return 0;
    }
};

struct stub_SslUI_new_client:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 2)
        {
            net::Address *peer=args.get<net::Address *>(0);
            openssl::pkg::Ctx *ctx=args.get<openssl::pkg::Ctx *>(1);
            return pylib::Converter<event::Cell *>(instance->new_client(peer,ctx),false);
        }
        else if(args.size() == 3)
        {
            net::Address *peer=args.get<net::Address *>(0);
            openssl::pkg::Ctx *ctx=args.get<openssl::pkg::Ctx *>(1);
            net::ProxyInfo *proxy=0;
            if(!args.is_none(2))
                proxy=args.get<net::ProxyInfo *>(2);
            return pylib::Converter<event::Cell *>(instance->new_client(peer,ctx,proxy),false);
        }
        return 0;
    }
};

class PyJrpUI:public openssl::pkg::JrpUI
{
   public:
        PyJrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx,vmp::utils::Logger *logger=0);
        ~PyJrpUI();

        pylib::PyCall pytcpconnect_;
        pylib::PyCall pytcpaccept_;
        pylib::PyCall pycsession_;
        pylib::PyCall pyssession_;
        pylib::PyCall pycclose_;
        pylib::PyCall pylclose_;
        pylib::PyCall pysclose_;
        
        pylib::PyCall pyrequest_;
        pylib::PyCall pyrequesterr_;
        pylib::PyCall pykill_;
        pylib::PyCall pypush_;
        pylib::PyCall pyresponse_;
        pylib::PyCall pyclose_;
};

PyJrpUI::PyJrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx,vmp::utils::Logger *logger):openssl::pkg::JrpUI(manager,ctx,logger)
{
}

PyJrpUI::~PyJrpUI()
{
}

struct stub_jrpui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PyJrpUI *>(cell->ui<PyJrpUI>(),false);
        }
        return 0;
    };
};

struct JrpUI_Constructor:public pylib::Constructor<PyJrpUI>
{
    PyJrpUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 2)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             openssl::pkg::Ctx_Peer *ctx=args.get<openssl::pkg::Ctx_Peer *>(1);
             return new PyJrpUI(manager,ctx);
         }
         else if(args.size() == 3)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             openssl::pkg::Ctx_Peer *ctx=args.get<openssl::pkg::Ctx_Peer *>(1);
             vmp::utils::Logger *logger=0;
             if(!args.is_none(2))
                logger=args.get<vmp::utils::Logger *>(2);
             return new PyJrpUI(manager,ctx,logger);
         }
         return 0;
    }
};

void jrp_pytcpconnect(event::Cell *cell)
{
    PyJrpUI *jrpui=cell->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    jrpui->pytcpconnect_.call(cargs);
}

void jrp_pytcpaccept(event::Cell *listen,event::Cell *server)
{
    PyJrpUI *jrpui=listen->ui<PyJrpUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(listen,false);
    cargs.add<event::Cell *>(server,false);
    jrpui->pytcpaccept_.call(cargs);    
}

void jrp_pycsession(event::Cell *cell)
{
    PyJrpUI *jrpui=cell->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    jrpui->pycsession_.call(cargs);
}

void jrp_pyssession(event::Cell *cell)
{
    PyJrpUI *jrpui=cell->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    jrpui->pyssession_.call(cargs);
}

void jrp_pycclose(event::Cell *cell)
{
    PyJrpUI *jrpui=cell->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    jrpui->pycclose_.call(cargs);
}

void jrp_pylclose(event::Cell *cell)
{
    PyJrpUI *jrpui=cell->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    jrpui->pylclose_.call(cargs);
}

void jrp_pysclose(event::Cell *cell)
{
    PyJrpUI *jrpui=cell->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    jrpui->pysclose_.call(cargs);
}

struct stub_JrpUI_set_connect_event:public pylib::Member_def<PyJrpUI>
{
    pylib::Objref *def(PyJrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 7)
        {
            event::EVTCB tcpconnect;
            net::EVTCBACCEPT tcpaccept;
            event::EVTCB csession;
            event::EVTCB ssession;
            event::EVTCB cclose;
            event::EVTCB lclose;
            event::EVTCB sclose;
            
            args.get_callable(0,instance->pytcpconnect_);
            args.get_callable(1,instance->pytcpaccept_);
            args.get_callable(2,instance->pycsession_);
            args.get_callable(3,instance->pyssession_);
            args.get_callable(4,instance->pycclose_);
            args.get_callable(5,instance->pylclose_);
            args.get_callable(6,instance->pysclose_);
            
            if(pylib::is_none(instance->pytcpconnect_.ref()))
                tcpconnect=0;
            else 
                tcpconnect=jrp_pytcpconnect;
            if(pylib::is_none(instance->pytcpaccept_.ref()))
                tcpaccept=0;
            else 
                tcpaccept=jrp_pytcpaccept;
            if(pylib::is_none(instance->pycsession_.ref()))
                csession=0;
            else 
                csession=jrp_pycsession;
            if(pylib::is_none(instance->pyssession_.ref()))
                ssession=0;
            else 
                ssession=jrp_pyssession;
            if(pylib::is_none(instance->pycclose_.ref()))
                cclose=0;
            else 
                cclose=jrp_pycclose;
            if(pylib::is_none(instance->pylclose_.ref()))
                lclose=0;
            else 
                lclose=jrp_pylclose;
            if(pylib::is_none(instance->pysclose_.ref()))
                sclose=0;
            else 
                sclose=jrp_pysclose;
            
            instance->set_connect_event(tcpconnect,tcpaccept,csession,ssession,cclose,lclose,sclose);
            return pylib::retnone();
        }
        return 0;
    }
};

void jrp_pyrequest(json::jrp::JrpReq *jreq)
{
    PyJrpUI *jrpui=jreq->cell()->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<json::jrp::JrpReq *>(jreq,false);
    jrpui->pyrequest_.call(cargs);    
}

void jrp_pyrequesterr(json::jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg)
{
    PyJrpUI *jrpui=jreq->cell()->ui<PyJrpUI>();
    pylib::CArgs cargs(3);
    cargs.add<json::jrp::JrpReq *>(jreq,false);
    cargs.add<vmp_int>(errcode);
    cargs.add<vmp::str>(msg);
    jrpui->pyrequesterr_.call(cargs);  
}

void jrp_pykill(json::jrp::JrpReq *jreq)
{
    PyJrpUI *jrpui=jreq->cell()->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<json::jrp::JrpReq *>(jreq,false);
    jrpui->pykill_.call(cargs);    
}

void jrp_pypush(json::jrp::JrpReq *jreq,json::JsonObj *jdata)
{
    PyJrpUI *jrpui=jreq->cell()->ui<PyJrpUI>();
    pylib::CArgs cargs(2);
    cargs.add<json::jrp::JrpReq *>(jreq,false);
    cargs.add<json::JsonObj *>(jdata,false);
    jrpui->pypush_.call(cargs);  
}

void jrp_pyresponse(json::jrp::JrpReq *jreq,json::JsonObj *jdata)
{
    PyJrpUI *jrpui=jreq->cell()->ui<PyJrpUI>();
    pylib::CArgs cargs(2);
    cargs.add<json::jrp::JrpReq *>(jreq,false);
    cargs.add<json::JsonObj *>(jdata,false);
    jrpui->pyresponse_.call(cargs);  
}

void jrp_pyclose(json::jrp::JrpReq *jreq)
{
    PyJrpUI *jrpui=jreq->cell()->ui<PyJrpUI>();
    pylib::CArgs cargs(1);
    cargs.add<json::jrp::JrpReq *>(jreq,false);
    jrpui->pyclose_.call(cargs);    
}

struct stub_JrpUI_set_request_event:public pylib::Member_def<PyJrpUI>
{
    pylib::Objref *def(PyJrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 6)
        {
           json::jrp::JREQCB request;
           json::jrp::JREQERRCB requesterr;
           json::jrp::JREQCB kill;
           json::jrp::JREQDATACB push;
           json::jrp::JREQDATACB response;
           json::jrp::JREQCB close;
           
           args.get_callable(0,instance->pyrequest_);
           args.get_callable(1,instance->pyrequesterr_);
           args.get_callable(2,instance->pykill_);
           args.get_callable(3,instance->pypush_);
           args.get_callable(4,instance->pyresponse_);
           args.get_callable(5,instance->pyclose_);
           
           if(pylib::is_none(instance->pyrequest_.ref()))
                request=0;
           else 
                request=jrp_pyrequest;
           if(pylib::is_none(instance->pyrequesterr_.ref()))
                requesterr=0;
           else 
                requesterr=jrp_pyrequesterr;
           if(pylib::is_none(instance->pykill_.ref()))
                kill=0;
           else 
                kill=jrp_pykill;
           if(pylib::is_none(instance->pypush_.ref()))
                push=0;
           else 
                push=jrp_pypush;
           if(pylib::is_none(instance->pyresponse_.ref()))
                response=0;
           else 
                response=jrp_pyresponse;
           if(pylib::is_none(instance->pyclose_.ref()))
                close=0;
           else 
                close=jrp_pyclose;
           instance->set_request_event(request,requesterr,kill,push,response,close);
           return pylib::retnone();
        }
        return 0;
    }
};

struct stub_JrpUI_new_client:public pylib::Member_def<PyJrpUI>
{
    pylib::Objref *def(PyJrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            net::Address *peer=args.get<net::Address *>(0);
            return pylib::Converter<event::Cell *>(instance->new_client(peer),false);
        }
        else if(args.size() == 2)
        {
            net::Address *peer=args.get<net::Address *>(0);
            net::ProxyInfo *proxy=0;
            if(!args.is_none(1))
                proxy=args.get<net::ProxyInfo *>(1);
            return pylib::Converter<event::Cell *>(instance->new_client(peer,proxy),false);
        }
        return 0;
    }
};

struct stub_JrpUI_add_reqdata:public pylib::Member_def<PyJrpUI>
{
    pylib::Objref *def(PyJrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 3)
        {
            vmp::str input=args.get<vmp::str>(0);
            vmp::vector<vmp::str> push=args.get<vmp::vector<vmp::str> >(1);
            vmp::vector<vmp::str> response=args.get<vmp::vector<vmp::str> >(2);
            instance->add_reqdata(input,push,response);
            return pylib::retnone();
        }
        else if(args.size() == 4)
        {
            vmp::str input=args.get<vmp::str>(0);
            vmp::vector<vmp::str> push=args.get<vmp::vector<vmp::str> >(1);
            vmp::vector<vmp::str> response=args.get<vmp::vector<vmp::str> >(2);
            vmp::str permits=args.get<vmp::str>(3);
            instance->add_reqdata(input,push,response,permits);
            return pylib::retnone();
        }
        return 0;
    }
};

struct stub_JrpUI_add_reqstatus:public pylib::Member_def<PyJrpUI>
{
    pylib::Objref *def(PyJrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 0)
        {
            instance->add_reqstatus();
            return pylib::retnone();    
        }    
        else if(args.size() == 1)
        {
            vmp::str permits=args.get<vmp::str>(0);
            instance->add_reqstatus(permits);
            return pylib::retnone();
        }
        return 0;
    }
};

struct stub_JrpUI_search_peer:public pylib::Member_def<PyJrpUI>
{
    pylib::Objref *def(PyJrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 0)
            return pylib::Converter<vmp::vector<vmp::str> >(instance->search_peer());
        else if(args.size() == 1)
        {
            vmp::str input=args.get<vmp::str>(0);
            return pylib::Converter<vmp::vector<vmp::str> >(instance->search_peer(input));
        }
        return 0;
   }
};

void openssl_init(pylib::Module module)
{
    //openssl0/openssl0.hset_connect_event
    
    module.def_("init",openssl::pkg::init);
    module.def_("end",openssl::pkg::end);

    //Wrapper cell->ui()
    module.def_with_stub("sslui",stub_sslui());
    
    pylib::Class<PySslUI> sslui=module.class_with_constructor<PySslUI>("SslUI",SslUI_Constructor());
    sslui.def_with_stub("set_event_client",stub_SslUI_set_event_client());
    sslui.def_with_stub("set_event_server",stub_SslUI_set_event_server());
    sslui.def_with_stub("new_client",stub_SslUI_new_client());
    sslui.def_("new_listen",&PySslUI::new_listen,false);
    sslui.def_("send",&PySslUI::send);
    sslui.def_("key_update",&PySslUI::key_update);
    sslui.def_("localaddr",&PySslUI::localaddr,false);
    sslui.def_("peeraddr",&PySslUI::peeraddr,false);
    sslui.def_("proxyinfo",&PySslUI::proxyinfo,false);
    sslui.def_("connection_type",&PySslUI::connection_type);
    sslui.def_("connection_strtype",&PySslUI::connection_strtype);
    sslui.def_("get_local_x509",&PySslUI::get_local_x509);
    sslui.def_("get_peer_x509",&PySslUI::get_peer_x509);
    
    //Wrapper cell->ui()
    module.def_with_stub("jrpui",stub_jrpui());
    
    pylib::Class<PyJrpUI> jrpui=module.class_with_constructor<PyJrpUI>("JrpUI",JrpUI_Constructor());
    jrpui.def_("set_timers",&PyJrpUI::set_timers);
    jrpui.def_("set_limits",&PyJrpUI::set_limits);
    jrpui.def_with_stub("set_connect_event",stub_JrpUI_set_connect_event());
    jrpui.def_with_stub("set_request_event",stub_JrpUI_set_request_event());
    jrpui.def_with_stub("new_client",stub_JrpUI_new_client());
    jrpui.def_("new_listen",&PyJrpUI::new_listen,false);
    jrpui.def_("key_update",&PyJrpUI::JrpUI::key_update);
    jrpui.def_("localaddr",&PyJrpUI::localaddr,false);
    jrpui.def_("peeraddr",&PyJrpUI::peeraddr,false);
    jrpui.def_("proxyinfo",&PyJrpUI::proxyinfo,false);
    jrpui.def_("session_localpeer",&PyJrpUI::session_localpeer,false);
    jrpui.def_("connection_type",&PyJrpUI::connection_type);
    jrpui.def_("connection_strtype",&PyJrpUI::connection_strtype);
    jrpui.def_("ctx",&PyJrpUI::ctx,false);
    jrpui.def_("logger",&PyJrpUI::logger,false);
    jrpui.def_("permits",&PyJrpUI::permits);
    jrpui.def_("get_peer_x509",&PyJrpUI::get_peer_x509);
    jrpui.def_("peer_fingerprint",&PyJrpUI::peer_fingerprint);
    jrpui.def_("peer_subject",&PyJrpUI::peer_subject);
    jrpui.def_("peer_permits",&PyJrpUI::peer_permits);
    jrpui.def_("peer_subject_f",&PyJrpUI::peer_subject_f);
    jrpui.def_("peer_permits_f",&PyJrpUI::peer_permits_f);
    jrpui.def_with_stub("add_reqdata",stub_JrpUI_add_reqdata());
    jrpui.def_with_stub("add_reqstatus",stub_JrpUI_add_reqstatus());
    jrpui.def_with_stub("search_peer",stub_JrpUI_search_peer());
    jrpui.def_("search_peer_p",&PyJrpUI::search_peer_p);
    jrpui.def_("search_peer_o",&PyJrpUI::search_peer_o);
    jrpui.def_("all_input",&PyJrpUI::all_input);
    jrpui.def_("all_input_f",&PyJrpUI::all_input_f);
    jrpui.def_("search_input_push",&PyJrpUI::search_input_push);
    jrpui.def_("search_input_push_f",&PyJrpUI::search_input_push_f);
    jrpui.def_("search_input_response",&PyJrpUI::search_input_response);
    jrpui.def_("search_input_response_f",&PyJrpUI::search_input_response_f);
    jrpui.def_("search_push",&PyJrpUI::search_push);
    jrpui.def_("search_push_f",&PyJrpUI::search_push_f);
    jrpui.def_("search_response",&PyJrpUI::search_response);
    jrpui.def_("search_response_f",&PyJrpUI::search_response_f);
    jrpui.def_("request",&PyJrpUI::request,false);
    jrpui.def_("request_f",&PyJrpUI::request_f,false);
}

void vampiria_pylib_init_module(pylib::Module init_module)
{
    openssl_init(init_module);
    openssl_hash(init_module);    
    openssl_pkey(init_module);
    openssl_x509(init_module);
    openssl_ssl(init_module);
    openssl_ctx(init_module);
}

 
 
