#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 04/04/2024

import vmp
import vmp_net as net
import vmp_plugin.base as base
import vmp_openssl4_ as openssl
import vmp_json as json

ptype_="openssl.clientjrp"

def session(cell):
  plugin=cell.getvar("plugin")
  plugin.peerstatus="connect"
  
def close(cell):
  plugin=cell.remvar("plugin")
  plugin.peer.release()
  plugin.peer=None
  plugin.peerstatus="close"
  plugin.exitlevel=cell.closelevel()
  plugin.exitcode=cell.errcode()
  if cell.ret() == event.CELLRET_ERROR:
    plugin.exitmsg=cell.str_error()
  elif cell.ret() == event.CELLRET_TIMEOUT:
    plugin.exitmsg="timeout"

def response_d(jreq,data):
  cell=jreq.cell()
  plugin=cell.getvar("plugin")
  rid=jreq.rid()
  if rid in plugin.debug:
    j=json.Json()
    j.json_new_obj(data)
    vmp.output(j.json_str())
  plugin.response(jreq,data)
  
def close_d(jreq):
  cell=jreq.cell()
  plugin=cell.getvar("plugin")
  rid=jreq.rid()
  request=plugin.requests[rid]
  request.jreq=None
  request.status=jreq.status()
  request.msg=jreq.msg()
  jreq.release()
  if rid in plugin.debug:
    vmp.output("Close request '{0}' ({1},{2})".format(rid,request.status,request.msg))
    plugin.debug.pop(rid)
    if len(plugin.debug) == 0:
      plugin.prompt()
      
def helper(plugin,args):
  cmds=dict(sorted(plugin.cmds.items()))
  if(len(args) == 0):
    vmp.output("Command list(Use '@' in front of command to run debug mode,and '@' for return in normal mode):")
    for cmd in cmds:
      vmp.output("{0:<20} - {1}".format(cmd,cmds[cmd].desc))
  else:
    cmd=args[0]
    if cmd in cmds:
      vmp.output("Command '{0}' - {1}".format(cmd,cmds[cmd].desc))
      for i in range(0,len(cmds[cmd].args)):
        vmp.output("  arg{0:<2} - {1}".format(i,cmds[cmd].args[i]))
      vmp.output("min arguments number: {0}".format(cmds[cmd].minargs))
    else:
      vmp.except_s("not found helper cmd {0}".format(args[0]))
  return []

def local(plugin,args):
  vmp.output("Subject:        [{0}]".format(plugin.subject))
  vmp.output("Finegrprint:    [{0}]".format(plugin.ctx.fingerprint()))
  vmp.output("Root Directory: [{0}]".format(plugin.rootpath))
  return []
 
def connect(plugin,args):
  if plugin.peerstatus == "close":
    try:
      vmp.output("Send Connect")
      plugin.exitlevel=""
      plugin.exitcode=None
      plugin.exitmsg=""
      addr=net.Address()
      if plugin.proxyinfo == None:
        addr.set(args[0],"{0}".format(args[1]))
      else:
        addr.set(args[0],"{0}".format(args[1]),True)
      plugin.peer=plugin.jrpui.new_client(addr,plugin.proxyinfo)
      plugin.peer.setvar("plugin",plugin)
      plugin.peer.alloc()
      plugin.requests={}
      plugin.debug={}    
      plugin.vars={}
    except Exception as e:
      vmp.error(str(e))
  else:
    vmp.error("Client already connected to a peer")
  return []

def connectstatus(plugin,args):
  if plugin.peerstatus == "close":
    if plugin.exitcode == None:
      vmp.output("Disconnect")
    elif plugin.exitmsg == "timeout":
      vmp.output("Close Timeout")
    elif plugin.exitmsg == "":
      vmp.output("Close ok (exitlevel={0})".format(plugin.exitlevel))
    else:
      vmp.output("Close (exitlevel={0},errocde={1},errmsg={2})".format(plugin.exitlevel,plugin.exitcode,plugin.exitmsg))
  elif plugin.peerstatus == "connecting":
    vmp.output("Connecting")
  else:
    vmp.output("Connect      peer [{0}]".format(plugin.peer.identity()));
    vmp.output("Subject      peer [{0}]".format(plugin.jrpui.peer_subject(plugin.peer)))
    vmp.output("Fingerprint  peer  [{0}]".format(plugin.jrpui.peer_fingerprint(plugin.peer)))
    vmp.output("LocalPermits peer [{0}]".format(plugin.jrpui.permits(plugin.peer)))
    vmp.output("Session Data peer:")
    vinput=plugin.jrpui.all_input(plugin.peer)
    for v in vinput:
      vmp.output("  (input=[{0}],push={1},response={2})".format(v,plugin.jrpui.search_push(plugin.peer,v),plugin.jrpui.search_response(plugin.peer,v)))
  return []
 
def disconnect(plugin,args):
  if plugin.peer != None:
    plugin.peer.close("top")
    vmp.output("Send Disconnect")
  else:
    vmp.error("Peer already disconnect")
  return []

def requests(plugin,args):
  requests=dict(sorted(plugin.requests.items()))
  vmp.output("|  rid  |  type      | status | exit message")
  for req in requests:
    data=requests[req]
    if data.status == 'open':
      vmp.output("|  {0:<5}|  {1:<5}   | open   | {2}".format(req,data.type,''))
    else:
      vmp.output("|  {0:<5}|  {1:<5}     | {2:<7}   | {3}".format(req,data.type,data.status,data.msg))
  return []

def kill(plugin,args):
  if args[0] in plugin.requests:
    data=plugin.requests[args[0]]
    if data.status == 'open':
      data.jreq.kill()
      vmp.output("Send Kill {0}".format(args[0]))
      return []
  vmp.error("Request {0} not active".format(args[0]))
  return []
  
def quit(plugin,args):
  plugin.manager.stop()
  return []
  
def varars(plugin,args):
  if len(args) == 0:
    vmp.output("Vars list:")
    for v in plugin.vars:
      vmp.output("{0:<20} - {1}".format(v,plugin.vars[v].desc))   
  elif args[0] in plugin.vars:
    vmp.output("{0}".format(plugin.vars[args[0]].value))
  else:
    vmp.error("Var '{0}' not found",args[0])
  return []

def debug(plugin,args):
  try:
    ret=vmp.unicode.str_toindex_list(args[0],0,vmp.INTMAX)
    if len(ret) == 0:
      vmp.except_s("Syntax error {0}".format(args[0]))
    plugin.active_debug(ret)
    if len(plugin.debug) == 0:
      vmp.except_s("Not Active Request Input")
  except Exception as e:
    plugin.debug={}
    vmp.except_s(str(e))
  return []
  
#Internal usage
class Requests:
  def __init__(self,jreq):
    self.jreq=jreq
    self.type=jreq.type()
    self.status="open"
    self.msg=''
     
class Var:
  def __init__(self,value,desc):
    self.value=value
    self.desc=desc
    
##Internal usage
class Cmd:
  def __init__(self,desc,minargs,args,cbfunc,prompt):
    self.desc=desc
    self.minargs=minargs
    self.args=args
    self.cbfunc=cbfunc
    self.prompt=prompt

  def check(self,args):
    retargs=[]
    l=len(args)
    if (l < self.minargs) or (l > len(self.args)):
      vmp.except_s("bad arguments number")
    i=0
    for arg in args:
      t=self.args[i]
      try:
        if t[1] == 'generic':
          retargs.append(args[i])
        elif t[1] == 'integer':
          minvalue=vmp.INTMIN
          maxvalue=vmp.INTMAX
          if(len(t) >= 3) and t[2] != None:
            minvalue=t[2]
          if(len(t) >= 4) and t[3] != None:
            maxvalue=t[3]
          v=vmp.unicode.str_todigit_range(args[i],minvalue,maxvalue)
          retargs.append(v)
        elif t[1] == 'real':
          minvalue=vmp.INTMIN
          maxvalue=vmp.INTMAX
          if(len(t) >= 3) and t[2] != None:
            minvalue=t[2]
          if(len(t) >= 4) and t[3] != None:
            maxvalue=t[3]
          v=vmp.unicode.str_toreal_range(args[i],minvalue,maxvalue)
          retargs.append(v)
        elif t[1] == 'hex':
          if not str_istype(args[i],"xdigit"):
            vmp.except_s("{0} not exadecimal type".format(args[i]))
          retargs.append(args[i])
        elif t[1] == 'ip':
          if not net.is_ip_raw(args[i]):
            vmp.except_s("{0} not ip type".format(args[i]))
          retargs.append(args[i])
        elif t[1] == 'ip4':
          if not net.is_ipv4_raw(args[i]):
            vmp.except_s("{0} not ipv4 type".format(args[i]))
        elif t[1] == 'ip6':
          if not net.is_ipv6_raw(args[i]):
            vmp.except_s("{0} not ipv6 type".format(args[i]))
          retargs.append(args[i])
        elif t[1] == 'mac':
          if not net.is_macaddress_raw(args[i]):
            vmp.except_s("{0} not mac type".format(args[i]))
          retargs.append(args[i])
        else:
          vmp.except_s("fatal error bad description tuple")
      except Exception as e:
        vmp.except_s("argn={0} {1}".format(i,str(e)))
      i=i+1
    return retargs
    
  def exec(self,plugin,args):
    retargs=self.check(args)
    return self.cbfunc(plugin,retargs)
  
##Plugin interface used for input and output operations.
class Plugin(base.Plugin):
  def __init__(self,data,options):
    base.Plugin.__init__(self,ptype_,data,5,options)
    self.logger=vmp.utils.Logger()
    self.manager=data[0]
    self.rootpath=data[1]
    self.subject=data[2]
    self.prompt_active=data[3]
    self.debug=data[4]
    
    self.cmds={}
    
    self.add_command("helper","Helper command",0,[('command name','generic')],helper)
    self.add_command("local","Information about the identity of the local peer",0,[],local)
    self.add_command("connect","Connect to jrp peer",2,[('host address peer','generic'),('host service','integer',1,65535)],connect)
    self.add_command("connectstatus","Print connection client status",0,[],connectstatus)
    self.add_command("disconnect","Disconnect peer",0,[],disconnect)
    self.add_command("requests","List of requests sent",0,[],requests)
    self.add_command("kill","Send a kill message to a request ",1,[("request number",'integer',0,None)],kill)
    self.add_command("quit","Exit console",0,[],quit,False)
    self.add_command("debug","Set the console to debug mode and print the responses associated with input requests. With '@' return in normal mode",1,[('request list','generic')],debug,False)
    self.add_command("vars","Print variable list if argument is not specified, otherwise print variable value with argument name",0,[('variable name','generic')],varars)
    
    self.ctx=openssl.Ctx_Peer(vmp.fs.union_path(self.rootpath,"x509"),1,self.subject)
    
    if self.debug:
      self.logger.set(vmp.fs.union_path(self.rootpath,'client.log'),vmp.utils.LOG_DEBUG,True)
    else:
      self.logger.set(vmp.fs.union_path(self.rootpath,'client.log'),vmp.utils.LOG_INFO,True)
    
    self.jrpui=openssl.JrpUI(data[0],self.ctx,self.logger)
    self.jrpui.set_connect_event(None,None,session,None,close,None,None);
    self.jrpui.set_request_event(None,None,None,None,response_d,close_d)
    
    self.proxyinfo=None
    self.peer=None
    self.peerstatus="close"
    self.exitlevel=""
    self.exitcode=None
    self.exitmsg=""
    
    self.requests={}
    self.debug={}
    self.vars={}
   
  ## Active debug mode
  #
  #@param reqlist request id list ex. [1,3,5]
  def active_debug(self,reqlist):
    self.debug={}
    for value in reqlist:
      if (value in self.requests) and (self.requests[value].status == "open"):
        self.debug[value]=''
  
  def prompt(self):
    if self.prompt_active:
      vmp.output_raw("ragnu>");
    
  ## Set Variable value
  #
  #@param name variable name
  #@param value variable value
  #@param desc description field
  def setvar(self,name,value,desc):
    if name in self.vars:
      self.vars[name].value=value
      self.vars[name].desc=desc
    else:
      self.vars[name]=Var(value,desc)
  
  ## Delete Variable value
  #
  #@param name variable name
  def delvar(self,name):
    if name in self.vars:
      self.vars.pop(name)
  
  ## Send a request jrp a peer connected
  #
  #@param jobj json object to send
  #@return request id or or except in case of failure
  def send_request(self,jobj):
    if self.peer != None:
      jreq=self.jrpui.request(self.peer,jobj)
      rid=jreq.rid()
      self.requests[rid]=Requests(jreq)
      jreq.alloc()
      vmp.output("Send Request {0}".format(rid))
      return rid
    vmp.except_s("Peer not connected")
  
  ## Push a value to request jrp
  #
  #@param rid request id
  #@param jobj json object to push
  #@return void or except in case of failure
  def push_request(self,rid,jobj):
     if self.peer == None:
       vmp.except_s("Peer not connected")
     if (rid in self.requests) and (self.requests[rid].status == "open"):
       self.requests[rid].push(jobj)
       vmp.output("Send Push to request {0}".format(rid))
     else:
       vmp.except_s("Reaquest {0} not open".format(rid))
  
  ## Add command to plugin
  #
  #@param name command name
  #@param helper helper string
  #@param minargs minimum number of arguments
  #@param args list of tuple Tuple [(argument decription,argtype,...var arguments),...]
  #argtype :
  #'generic' - generic arguments
  #'integer' - integer value var arguments=(minvalue='',maxvalue='') ''=not assigned
  #'real'    - integer value var arguments=(minvalue='',maxvalue='') ''=not assigned
  #'hex'     - hexadecimal value
  #'ip'      - ip value
  #'ip4'     - ipv4 value
  #'ip6'     - ipv6 value
  #'mac'     - mac address value 
  #@param callback function of type [list request id open request]  cbfunc(plugin,args,debug),return true if debug mode
  #@prompt if true activates prompt after execution
  def add_command(self,name,helper,minargs,args,cbfunc,prompt=True):
    self.cmds[name]=Cmd(helper,minargs,args,cbfunc,prompt)
  
  ## Run a command
  #
  #@param cmdline command line
  def command(self,cmdline):
    if self.debug != {}:
      if cmdline == '@':
        self.debug={}
      else:
        return
    else:
      if cmdline != '\n':
        try:
          split=vmp.unicode.shlex_split(cmdline)
          if len(split) > 0:
            cmd=split[0]
            args=split[1:]
            if cmd != '':
              if cmd[0] == '@':
                debug=True
                cmd=cmd[1:]
              else:
                debug=False
            if cmd in self.cmds:
              try:
                ret=self.cmds[cmd].exec(self,args)
                if len(ret) != 0:
                  if debug:
                    self.active_debug(ret)
                    return
                if not self.cmds[cmd].prompt:
                  return
              except Exception as e:
                vmp.error("Command '{0}'".format(str(e)))
            else:
              vmp.except_s("Command '{0}' not found".format(cmd)) 
        except Exception as e:
          vmp.error(str(e))
          vmp.time.vsleep(0.2)
    self.prompt()
      
  ##Virtual function for the management of incoming responses
  #
  #@param jreq json request associated
  #@param jdata jsonObject data recv
  def response(self,jreq,djata):
    pass
       
##Plugin interface topic
class PluginTopic(base.PluginTopic):
  def __init__(self,modname,plugname):
    base.PluginTopic.__init__(self,ptype_,modname,plugname)
    self.set_desc("Plugin Interface used for creating jrp clients")
    self.add_data("An event manager used for JRP events")
    self.add_data("The root directory of the files used by the plugin")
    self.add_data("The subject for the peer client certificate")
    self.add_function("prompt","Print the prompt on the screen")
    self.add_function("setvar","Set a variable",[('name','variable name'),('name','variable value'),('desc','description field')])
    self.add_function("delvar","Delete Variable value",[('name','variable name')])
    self.add_function("send","Send a request jrp a peer connected",[('jobj','json object to send')],'request id or or except in case of failure')
    self.add_function("push_request","Push a value to request jrp",[('rid','request id'),('jobj','json object to push')],'void or except in case of failure')
    self.add_function("add_command","Add command to plugin",[('name','command name'),('helper','helper string'),('minargs','minimum number of arguments'),('args','arguments list description (see doxygen doc in plugin type)'),('callback','callback function to run command (see doxygen doc in plugin type)'),('prompt','if true active prompt')])
    self.add_function("command","Send the plugin a command to run",[('cmdline','command string')])
    self.add_function("response","Virtual function for the management of incoming responses",[('jreq','json request associated'),('jdata','json object data recv')])
    
    
