/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 21/03/2024
 */

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/md5.h>
#include <openssl/sha.h>
#include <openssl/x509v3.h>
#include <openssl/bn.h>
#include <openssl/asn1.h>
#include <openssl/evp.h>
#include <openssl/x509.h>
#include <openssl/x509_vfy.h>
#include <openssl/pem.h>
#include <openssl/bio.h>


#include "vmp.h"
#include "net.h"
#include "event.h"
#include "json.h"
#include "db0/db0.h"

#include "openssl/lib.h"
#include "openssl/hash.h"
#include "openssl/pkey.h"
#include "openssl/x509.h"
#include "openssl/ctx.h"
#include "openssl/ssl.h"
#include "openssl/cb.h"
#include "openssl/eventssl.h"
#include "openssl/eventp2p.h"
#include "openssl/eventjrp.h"

#ifndef VAMPIRIA_OPENSSL_PKG_H

#define VAMPIRIA_OPENSSL_PKG_H 1

namespace vampiria { namespace openssl { namespace pkg {

//Initialized ssl lib
void init();

//Free structure ssl lib
void end();

//! Ssl user interface
class SslUI:public event::UI
{
    private:
        vmp::utils::Storage<openssl::pkg::EventSsl> sref_;/*!<Storage event*/

        event::EVTCB tcpconnect_;/*!<Called by the client after the tcp connect is done*/
        event::EVTCB sslconnect_;/*!<Called by the client after the SSL handshake is done*/
        event::EVTCBBUF crecv_;/*!< Client receive callback*/
        event::EVTCB    cclose_;/*!<Client close callback*/

        net::EVTCBACCEPT tcpaccept_;/*!<Called by the client after the tcp connect is done*/
        event::EVTCB sslaccept_;/*!<Callback call after server ssl handshake is done*/
        event::EVTCB    lclose_;/*!<Close listen event*/
        event::EVTCBBUF srecv_;/*!<Receive server callback*/
        event::EVTCB    sclose_;/*!<Close connection server*/
        
        vmp::time::Time ctimeout_;/*!<Connection timeout*/
    public:
        //!A Constructor
        /*!
            @param manager input manager
            @param ctimeout Connection timeout
        */
        SslUI(event::Manager *manager,vmp::time::Time ctimeout=3.0);
        
        //!A Destructorne
        ~SslUI();

        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);

        //!Virtual class implemented
        void close_event(event::Cell *cell);

        //!Virtual class implemented
        void free_ref(event::Cell *cell);

        //!Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);

        //!Set client event
        /*!
            @param tcpconnect called by the client after the tcp connect is done
            @param sslconnect callback call after client ssl handshake is done
            @param crecv receive client callback
            @param cclose close client callback
        */
        void set_event_client(event::EVTCB tcpconnect,event::EVTCB sslconnect,event::EVTCBBUF crecv,event::EVTCB cclose);
        
        //! Set server event
        /*!
            @param tcpaccept called by the client after the tcp connect is done
            @param sslsconnect callback call after server ssl handshake is done
            @param lclonese close listen event
            @param srecv receive server callback
            @param sclose close connection server
        */
        void set_event_server(net::EVTCBACCEPT tcpaccept,event::EVTCB sslaccept,event::EVTCB lclose,event::EVTCBBUF srecv,event::EVTCB sclose);
        
        //! Add ssl client event. Callback events assigned with set_event_client
        /*!
            @param peer remote server address
            @param ctx ssl context
            @param proxy proxy information,if use proxy
            @return event cell or except in case of failure
        */
        event::Cell *new_client(net::Address *peer,openssl::pkg::Ctx *ctx,net::ProxyInfo *proxy=0);

        //! Add ssl server listen event. Callback events assigned with set_event_server
        /*!
            @param local address listener
            @param backlog max connection server
            @param ctx ssl context
            @return event cell or except in case of failure
        */
        event::Cell *new_listen(net::Address *local,vmp_size backlog,openssl::pkg::Ctx *ctx);

        //! Send packet
        /*!
            @param cell event cell
            @param buf packet buffer
	    @return void or except in case of failure
        */
        void send(event::Cell *cell,vmp::Buf *buf);
	
	//! Renegotiate the session key with the peer
	/*!
	    @param cell event cell
	    @return void or except in case of failure
	*/
	void key_update(event::Cell *cell);
	
	//!Gets socket address local
        /*!
            @param cell cell associate
            @return local addres
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return peer address 
        */
        net::Address *peeraddr(event::Cell *cell);

        //! Gets peer proxy info(for client connection)
        /*!
            @param cell cell associated
            @return proxy info or except in case of failure
        */
        net::ProxyInfo *proxyinfo(event::Cell *cell);

        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventCoonection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
        
        //!Gets the local's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_local_x509(event::Cell *cell,openssl::pkg::X509_Wrap *cout);
        
        //!Gets the peer's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_peer_x509(event::Cell *cell,openssl::pkg::X509_Wrap *cout);
};

//!Json request protocol user interface
class JrpUI:public event::UI
{
    private:
        openssl::pkg::JrpCommon *common_;/*!<Object shared by all events*/
        vmp::utils::Logger *logger_;/*!<Logger object,if equal 0 not logger associated*/
        
        vmp::utils::Storage<openssl::pkg::EventJrp> jref_;/*!<Storage event*/
    public:
        //! A Constructor
        /*!
            @param manager input manager
            @param ctx context peer identity
            @param logger logger object,if equal 0 not logger associated
        */
        JrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx,vmp::utils::Logger *logger=0);

        //!A Destructor
        ~JrpUI();

        event::EVTCB tcpconnect_;/*!<Called when a connection tcp connect is complete(client connection)*/
        event::EVTCB sessionclient_;/*!<Callback call when a valid connection has estabilized(client)*/
        
        net::EVTCBACCEPT tcpaccept_;/*!<Function performed when connection tcp is accepted from listen connections*/
        event::EVTCB sessionserver_;/*!<Callback call when a valid connection has estabilized(server)*/
        event::EVTCB closeclient_;/*!<Close peer client connection*/
        event::EVTCB closelisten_;/*!<Close server listen event*/
        event::EVTCB closeserver_;/*!<Close peer server connection*/
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //! Set timer configurations for SSL peer connections
        /*!
            @param ctimeout ssl Connection handshake timeout(if value <= 0 not setting nothing)
            @param stimeout jrp session init timeout(if value <= 0 not setting nothing)
            @param timeout  jrp session timeout. if value <= 0 not setting nothing)
        */
        void set_timers(vmp::time::Time ctimeout,vmp::time::Time stimeout,vmp::time::Time timeout);
        
        //! Set limits configurations for SSL peer connections
        /*!
            @param backlog  max server connection pending(if value = 0 not setting nothing)
            @param maxssrvpeer maximum number of requests a peer can make(if value = 0 not setting nothing)
            @param maxclientreq maximum number of requests I can request(if value = 0 not setting nothing)
        */
        void set_limits(vmp_size backlog,vmp_size maxssrvpeer,vmp_size maxclientreq);
        
        //! Set callback function. if value == 0 setting empty callback.
        /*!
            @param tcpconnect called when a connection tcp connect is complete(client connection)
            @param tcpaccept function performed when connection tcp is accepted from listen connections
            @param csession call when a jrp session has been created(Client)
            @param ssession call when a jrp session has been created(Server)
            @param lclose function performed at the end listen event(Listen connection)
            @param cclose function performed at the end of the event(Client)
            @param sclose function performed at the end of the event(Client)
        */
        void set_connect_event(event::EVTCB tcpconnect,net::EVTCBACCEPT tcpaccept,event::EVTCB csession,event::EVTCB ssession,event::EVTCB cclose,event::EVTCB lclose,event::EVTCB sclose);
        
        //! Set callback for request event(if event input value = 0 setting empty callback)
        /*!
            @param request    Callback call when a request message has been received
            @param requesterr Callback call when a request message has been received and a management error occurs(ex.Resource Access Denied)
            @param kill       Callback call when a kill message has been received
            @param push       Callback call when a response message has been received
            @param response   Callback call when a response message has been received
            @param close      Callback call when a close message has been received
        */
        void set_request_event(json::jrp::JREQCB request,json::jrp::JREQERRCB requesterr,json::jrp::JREQCB kill,json::jrp::JREQDATACB push,json::jrp::JREQDATACB response,json::jrp::JREQCB close);
        
        //! Create a client connection(monitor mode).Timeout is managed within the event
        /*!
            @param peer peer address
            @param proxy proxy information,if use proxy
            @return event cell or except in case the failure
        */
        event::Cell *new_client(net::Address *peer,net::ProxyInfo *proxy=0);
        
        //! Listen server jrp(monitor manager).Timeout is managed within the event
        /*!
            @param local listen address
            @return event cell or except in case the failure
        */
        event::Cell *new_listen(net::Address *local);
        
        //! Renegotiate the session key with the peer
	/*!
	    @param cell event cell
	    @return void or except in case of failure
	*/
	void key_update(event::Cell *cell);
	
	//!Gets socket address local
        /*!
            @param cell cell associate
            @return local addres
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return peer address 
        */
        net::Address *peeraddr(event::Cell *cell);

        //! Gets peer proxy info(for client connection)
        /*!
            @param cell cell associated
            @return proxy info or except in case of failure
        */
        net::ProxyInfo *proxyinfo(event::Cell *cell);
        
        //! Return the jrp session values that local peer sends to the remote
        /*!
            @return jrp session object
        */
        json::jrp::JrpReqSession *session_localpeer();
        
        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventCoonection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
       
        //! Returns associated context
        /*!
            @sa ctx_
            @return context associated
        */
        openssl::pkg::Ctx_Peer *ctx();
        
        //! Returns associated logger
        /*!
            @sa logger_
            @return associated logger or except in case no logger is associated
        */
        vmp::utils::Logger *logger();

        //! Permission assigned by remote peer to local peer(monitor mode)
        /*!
            @param cell cell associated
            @return permits assigned
        */
        vmp_uint permits(event::Cell *cell);
        
        //!Gets the peer's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_peer_x509(event::Cell *cell,openssl::pkg::X509_Wrap *cout);
        
        //! Returns the fingerprint of the peer associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return fingerprint peer
        */
        vmp::str peer_fingerprint(event::Cell *cell);

        //! Returns the subject of the peer associated with the input cell(monitor mode)
        /*!
            @param cell event cell
            @return subject peer
        */
        vmp::str peer_subject(event::Cell *cell);

        //! Returns the permits of the peer associated with the input cell(monitor mode)
        /*!
            @param cell input cell
            @return permits peer
        */
        vmp_uint peer_permits(event::Cell *cell);
        
        //! Returns the subject of the peer associated with the input fingerprint(monitor mode) 
        /*!
            @param fingerprint input fingerprint
            @return subject peer or except in case the failure
        */
        vmp::str peer_subject_f(vmp::str fingerprint);
        
        //! Returns the permits of the peer associated with the input fingerprint(monitor manager)
        /*!
            @param fingerprint input fingerprint
            @return permits peer or except in case the failure
        */
        vmp_uint peer_permits_f(vmp::str fingerprint);

        //! Adds a request data
        /*!
            @param input input request data
            @param push data push associated with the input
            @param response data response associated with the input
            @param permits permits string(see json::jrp::JrpCommon_I)
            @return void or except in case of failure
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits="*");

        //! Add default status value
        /*!
            @param permits permits string(see json::jrp::JrpCommon_I)
            @return void or except in case of failure
        */
        void add_reqstatus(vmp::str permits="*");
        
        //! Returns the fingerprints of the peer associated with the input fingerprint(monitor mode)       
        /*!
            @param input input request data required(input ="" return all peer fingerprint)
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer(vmp::str input="");
        
        //! Returns the fingerprints of the peer which have at least one push data of the assigned type (monitor mode)       
        /*!
            @param push push data required
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer_p(vmp::str push);
        
        //! Returns the fingerprints of the peer which have at least one response of the assigned type (monitor mode)       
        /*!
            @param response push data required
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer_o(vmp::str response);
        
        //! Return all peer inputs
        /*!
            @param cell event cell associated with the peer
            @return input list
        */
        vmp::vector<vmp::str> all_input(event::Cell *cell);
        
        //! Return all peer inputs
        /*!
            @param fingerprint fingerprint event associated with the peer
            @return input list
        */
        vmp::vector<vmp::str> all_input_f(vmp::str fingerprint);
        
        //!Return all peer inputs that have a given push
        /*!
            @param cell event cell associated with the peer
            @param push push data
            @return input list
        */
        vmp::vector<vmp::str> search_input_push(event::Cell *cell,vmp::str push);
        
        //!Return all peer inputs that have a given push
        /*!
            @param fingerprint fingerprint event associated with the peer
            @param push push data
            @return input list
        */
        vmp::vector<vmp::str> search_input_push_f(vmp::str fingerprint,vmp::str push);
        
        //!Return all peer inputs that have a given response
        /*!
            @param cell event cell associated with the peer
            @param response response data
            @return input list
        */
        vmp::vector<vmp::str> search_input_response(event::Cell *cell,vmp::str response);
        
        //!Return all peer inputs that have a given response
        /*!
            @param fingerprint fingerprint event associated with the peer
            @param response response data
            @return input list
        */
        vmp::vector<vmp::str> search_input_response_f(vmp::str fingerprint,vmp::str response);
        
        //! Returns the list of push data that have an input of the topic(monitor mode)
        /*!
            @param cell event cell associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_push(event::Cell *cell,vmp::str input);
        
        //! Returns the list of outputs that have an input of the topic(monitor manager)
        /*!
            @param fingerprint fingerprint associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_push_f(vmp::str fingerprint,vmp::str input);
        
        //! Returns the list of response data that have an input of the topic(monitor mode)
        /*!
            @param cell event cell associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_response(event::Cell *cell,vmp::str input);
        
        //! Returns the list of response that have an input of the topic(monitor manager)
        /*!
            @param fingerprint fingerprint associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_response_f(vmp::str fingerprint,vmp::str input);
        
        //!Send a request jrp a peer associated with the cell(monitor manager)
        /*!
            @param cell input cell
            @param jdata jdata input
            @return the structure associated with the request or except in case of failure("Overflow request" if the number of requests exceeds the maximum number)
        */
        json::jrp::JrpReq *request(event::Cell *cell,json::JsonObj *jdata);

        //!Send a request jrp a peer associated with the fingerprint(monitor manager)
        /*!
            @param fingerprint input cell
            @param jdata jdata input
            @return the structure associated with the request or except in case of failure("Overflow request" if the number of requests exceeds the maximum number)
        */
        json::jrp::JrpReq *request_f(vmp::str fingerprint,json::JsonObj *jdata);

};

}}}

#endif



