/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/02/2024
 */
 
#ifndef VAMPIRIA_OPENSSL_PKG_SSL_H

#define VAMPIRIA_OPENSSL_PKG_SSL_H 1

namespace vampiria { namespace openssl { namespace pkg {

//!Class used for the connections ssl
class Ssl
{
    private:
        openssl::pkg::Ctx ctx_;/*!<Connection context*/
        vmp_bool setting_;/*!<Setting connection?*/
        SSL *ssl_;/*!<Ssl handler*/
        
        vmp_int errno_;/*!<error value set by the last operation performed*/
        
        //!Internal usage
        void except_null_pointer(vmp::str func);
    public:
        //! A Constructor
        Ssl();
        
        //!A Destructor
        ~Ssl();

        //! Reset ssl connection
        void reset();
        
        //! Create an ssl structure associated with the context
        /*!
            @param ctx context associated with ssl connection
            @param sock socket associated
            @return void or except in case of failure
        */
        void set(openssl::pkg::Ctx *ctx,net::Socket sock);
        
        //! Returns ssl socket associated
        /*!
            @return the socket linked to ssl,-1 error socket(see this->errno() value for error type or for error string use openssl::pkg::err_string()) or except in case of failure
        */
        net::Socket get_socket();
        
        //! Returns the ssl error associated with the ret value
        /*!
            @sa errno_
            @return errno value
        */
        vmp_int get_errno();
        
        //! Returns the context associated with the connection
        /*!
            @sa ctx_
            @return the context or except in case of failure
            
        */
        openssl::pkg::Ctx *ctx();
        
        //!Gets the local's certificate
        /*!
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_local_x509(openssl::pkg::X509_Wrap *cout);
        
        //!Gets the peer's certificate
        /*!
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_peer_x509(openssl::pkg::X509_Wrap *cout);
        
        //! Send data
        /*!
            @param buf data to send
            @return size packet to send,-1 error socket(see this->errno() value for error type or for error string use openssl::pkg::err_string()) or except in case of failure
        */
        vmp_int send(vmp::Buf *buf);
        
        //! Recv data
        /*!
            @param buf received data
            @return size packet to send,0 socket close,-1 error socket(see this->errno() value for error type or for error string use openssl::pkg::err_string()),
             -2 no data received from non blocking socket or except in case of failure
*/
        vmp_int recv(vmp::Buf *buf);
        
        //!Renegotiate the session key with the peer
        /*!
            @return 0 ok,-1 error update or except in case of failure
        */
        vmp_int key_update();
        
        //! Runs connect ssl handshake
        /*!
            @return 0 connect ok,-1 error socket(see this->errno() value for error type or for error string use openssl::pkg::err_string()),
             -2 no data received from non blocking socket or except in case of failure   
        */
        vmp_int connect();
        
        //! Runs connect ssl handshake
        /*!
            @return 0 accept ok,-1 error socket(see this->errno() value for error type or for error string use openssl::pkg::err_string()),
             -2 no data received from non blocking socket or except in case of failure   
        */
        vmp_int accept();

        //! Close ssl connection
        /*!
            @return 0 ok,-1 error close (see this->errno() value for error type or for error string use openssl::pkg::err_string()) or except in case of failure
        */
        vmp_int close();
};

}}}

#endif

